"""
    Common settings of search verticals
"""

import os.path
import logging
import itertools

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import apihelpers
from sandbox.projects.common import decorators
from sandbox.projects.common import dolbilka
from sandbox.projects.common.nanny.nanny import NannyClient
from sandbox.projects.common.search import cluster_names
from sandbox.projects.images import resource_types as images_resource_types
from sandbox.projects.images.models import resources as images_models_resources
from sandbox.projects.images.bans import resources as images_bans_resources
from sandbox.projects.images.basesearch import resources as images_basesearch_resources
from sandbox.projects.images.daemons import resources as images_daemons_resources
from sandbox.projects.images.rq import resources as images_rq_resources
from sandbox.projects.images.balancer import resources as images_balancer_resources
from sandbox.projects.VideoSearch import video_resource_types as video_resource_types

import sandbox.projects.common.thumbdaemon.utils as thumbdaemon_utils

from sandbox.projects.tank import offline


# Thumbdaemon constants
IMPROXY_PRIEMKA = thumbdaemon_utils.IMPROXY_PRIEMKA

SAS_REGION = 'sas'  # Sasovo DC
MAN_REGION = 'man'  # Finland DC
VLA_REGION = 'vla'  # Vladimir DC
PRIEMKA_REGION = 'priemka'  # Fake Region - acceptance
DEFAULT_REGIONS = (VLA_REGION, SAS_REGION, MAN_REGION)

# Index names
INDEX_MAIN = "main"  # Main index
INDEX_QUICK = "quick"  # Quick index
INDEX_ULTRA = "ultra"  # Ultra quick index
INDEX_GARBAGE = "garbage"  # Garbage index
INDEX_CBIR_MAIN = "cbir"  # Cbir main index
INDEX_CBIR_QUICK = "cbir-quick"  # Cbir quick index
INDEX_CBIR_GARBAGE = "cbir-garbage"  # Cbir garbage index
INDEX_CBIR_DAEMON = "cbir-daemon"  # Fake index for cbirdaemon
INDEX_THUMB_MAIN = "thumb-main"  # Thumb main index
INDEX_THUMB_QUICK = "thumb-quick"  # Thumb quick index
INDEX_RQ = "rq"  # Related queries index
INDEX_ALL = "all"  # Fake index - all possible index types
INDEX_MIDDLE = "mmeta"  # index on middlesearch
INDEX_MIDDLE_RQ = "mmeta-rq"  # middlesearch for related queries
INDEX_INT = "int"  # Fake index - intermediate search
INDEX_CBIR_INT = "cbir-int"  # cbir intermediate search
INDEX_RIM = "rim"  # RIM index
INDEX_RIMPATCH = "rimpatch"  # Fake RIM index - rimpatch
INDEX_INVERTED = "imgsinverted"  # Partitioned panther (P-daemon index)
INDEX_EMBEDDING = "imgsembedding"  # Embedding storage (E-daemon index)

# Types of dynamic models
MODELS_PRODUCTION = "prod"  # Production formulas
MODELS_EXPERIMENTAL = "exp"  # Experimental formulas
MODELS_FINAL = "final"  # Mix of production and experimental formulas for upload to production

SHARD_TOTAL_SIZE_ATTRIBUTE_NAME = 'all_files_size'  # Basesearch shard size (total)
SHARD_MAPPED_SIZE_ATTRIBUTE_NAME = 'memory_mapped_files_size'  # Basesearch shard size (mapped to memory)
SHARD_INSTANCE_ATTRIBUTE_NAME = 'shard_instance'  # Basesearch shard name
RTINDEX_SHARD_INSTANCE_ATTRIBUTE_NAME = 'shard'  # Rtyserver shard attribute name.
RTINDEX_SHARD_TIMESTAMP_ATTRIBUTE_NAME = 'index_timestamp'  # Rtyserver shard timestamp attribute name.
RTINDEX_SHARD_SERVICE_ATTRIBUTE_NAME = 'service'  # Rtyserver shard index attribute name.
META_SHARD_INSTANCE_ATTRIBUTE_NAME = 'meta_shard_instance'  # Middlesearch shard name
SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME = 'snip_shard_instance'  # Snippetizer shard name
INDEX_STATE_ATTRIBUTE_NAME = 'index_state'

_PRE_STABLE_PREFIX = "pre-"  # Prefix for stable branches


_PRISM_REMOVED = (
    'Prism for images basesearch aka staging is deprecated, '
    'services were removed, contact anskor@'
)


def region2ctype(region):
    """
        Returns ctype tag for region
    """
    return "prestable" if region == SAS_REGION else "prod"


class WebSettings:
    NAME = 'web'
    # it does not change
    MIDDLESEARCH_DATA = 'rbtorrent:3b3fe1138095fc0f387155bf2917064ab0d630c1'
    DEFAULT_BASESEARCH_BINARY_RES_TYPE = resource_types.BASESEARCH_EXECUTABLE
    DEFAULT_BASESEARCH_MODELS_RES_TYPE = resource_types.DYNAMIC_MODELS_ARCHIVE_BASE

    @staticmethod
    def basesearch_config_resource():
        return resource_types.SEARCH_CONFIG

    @staticmethod
    def basesearch_database_resource():
        return resource_types.SEARCH_DATABASE

    @classmethod
    def default_basesearch_attrs_1(cls):
        return cls.testenv_base_resources_attr_name("PlatinumTier0")

    @classmethod
    def default_middle_service(cls, hamster=True):
        if hamster:
            return "vla_jupiter_mmeta_hamster_yp"
        return "vla_jupiter_mmeta_yp"

    @classmethod
    def testenv_middle_cfg_attr_name(cls, middle_type, hamster_cfg=True):
        if hamster_cfg and middle_type in {"jupiter_mmeta", "jupiter_int"}:
            if middle_type == "jupiter_int":
                middle_type = "sas_web_int_l1_hamster"
            else:
                middle_type = cls.default_middle_service(
                    hamster=True
                )
            source = 'hamster'
        else:
            source = 'production'
        return "TE_{}_{}_{}_cfg".format(cls.NAME, source, middle_type)

    @classmethod
    def testenv_middle_resources_attr_name(cls, middle_type, res_name):
        return "TE_{}_production_{}_{}".format(cls.NAME, middle_type, res_name)

    @staticmethod
    def testenv_middle_wiz_queries_attr_name():
        return "TE_web_meta_wizarded_queries"

    @classmethod
    def testenv_base_resources_attr_name(cls, tier):
        return "TE_web_base_prod_resources_{}".format(tier)


class QuickSettings:
    NAME = 'quick'

    @classmethod
    def testenv_middle_resources_attr_name(cls, middle_type):
        return "TE_{}_production_{}".format(cls.NAME, middle_type)


class ImagesSettings:
    NAME = 'images'
    INDICES = (INDEX_MAIN, INDEX_QUICK)

    PRIEMKA_SOURCE = (INDEX_MAIN, SAS_REGION)

    COLLECTION_ALL = 'all'
    COLLECTION_MAIN = 'yandsearch'
    COLLECTION_QUICK = 'quick'
    COLLECTION_ULTRA = 'imagesultra'
    COLLECTION_CBIR = 'imgscbir'

    IMPROXY_CONFIG_LOCATIONS = (PRIEMKA_REGION, VLA_REGION, SAS_REGION, MAN_REGION)
    IMPROXY_PRODUCTION_CONFIG_LOCATIONS = (VLA_REGION, SAS_REGION, MAN_REGION)
    IMPROXY_PRODUCTION_CONFIGS = ['improxy-images-{}'.format(x) for x in IMPROXY_PRODUCTION_CONFIG_LOCATIONS]
    IMPROXY_CONFIGS = ['improxy-images-{}'.format(x) for x in IMPROXY_CONFIG_LOCATIONS]

    IMPROXY_CGROUP_LOW_LIMIT = 5368709120
    IMPROXY_CGROUP_LIMIT = 10737418240

    IMPROXY_WEBP_HOST = "im0-tub-ru.yandex.net"

    MEDIA_BDI_CONFIG_LOCATIONS = (SAS_REGION, MAN_REGION, VLA_REGION)

    RESOURCE_PRIEMKA_ATTRIBUTES = ('images_priemka', 'yes')

    QUERIES_ATTR_NAME = "images_queries_type"

    QUERIES_TYPE_QLANG = "QLANG"
    QUERIES_TYPE_CBIR = "CBIR"
    QUERIES_TYPE_CBIR_SIGN = "CBIR_SIGN"
    QUERIES_TYPE_SERP = "SERP"

    # generated queries
    SERP_TYPE_MAIN = "SERP"
    SERP_TYPE_MAIN_XML = "XML_SERP"
    SERP_TYPE_MAIN_QLANG = "QLANG_SERP"
    SERP_TYPE_IMAGEVIEW = "IMAGEVIEW"
    SERP_TYPE_IMAGEDUPS = "IMAGEDUPS"
    SERP_TYPE_IMAGELIKE = "IMAGELIKE"
    SERP_TYPE_IMAGEDUPS_XML = "XML_IMAGEDUPS"
    SERP_TYPE_IMAGELIKE_XML = "XML_IMAGELIKE"
    SERP_TYPE_IMAGESIGN_XML = "XML_IMAGESIGN"
    # filtered queries
    SERP_TYPE_FILTER_ATTRS = "FILTER_ATTRS"  # text queries with attributes
    SERP_TYPE_FILTER_DUPS = "FILTER_DUPS"  # cbir queries for copies
    SERP_TYPE_FILTER_SIMILAR = "FILTER_SIMILAR"  # cbir queries for similar images
    SERP_TYPE_FILTER_WEB_SLOW = "FILTER_WEB_SLOW"  # slow queries from web search

    HAMSTER_URL = "https://hamster.yandex.{}/"
    XML_HAMSTER_URL = "https://hamster.yandex.{}/"

    # Tier0 requires 40Gb
    # Tier1 (garbage) requires 56Gb
    # Use max(40, 56)
    SNIPPETS_RAMDRIVE_SIZE = 56  # GB
    SNIPPETS_RAMDRIVE_FILES = (
        "indexarc",
        "indexdir",
        "indexannfactorsarc",
        "indexannfactorsdir",
        "indexannarc",
        "indexanndir",
        "indexsemdesc",
        "index.extinfoarc.wad",
        "index.wad",
    )

    DEFAULT_BASESEARCH_BINARY_RES_TYPE = images_basesearch_resources.IMGSEARCH_EXECUTABLE
    DEFAULT_BASESEARCH_MODELS_RES_TYPE = images_models_resources.IMAGES_DYNAMIC_MODELS_ARCHIVE
    AUTO_DATABASES = (
        resource_types.IMAGES_SEARCH_DATABASE,
        images_daemons_resources.CBIR_DAEMON2_SETTINGS,
    )

    DATABASE_TIERS = {
        "ImgMmetaTier0": INDEX_MIDDLE,
        "ImgTier0": (INDEX_MAIN, INDEX_CBIR_MAIN),
        "ImgTier1": (INDEX_GARBAGE, INDEX_CBIR_GARBAGE),
        "ImgCBIRTier0": INDEX_CBIR_MAIN,
        "ImgQuickTier0": INDEX_QUICK,  # Obsolete.
        "ImgsSaasQuickTier0": INDEX_QUICK,
        "ImgQuickCBIRTier0": INDEX_CBIR_QUICK,
        "ImgRQ2BaseTier0": INDEX_RQ,
        "ImgRimTier0": INDEX_RIM,
    }

    DATABASE_PREFIXES = (
        (INDEX_GARBAGE, "imgsidxt1"),
        (INDEX_MAIN, "imgsidx"),
        (INDEX_QUICK, "imgfidx"),
        (INDEX_CBIR_MAIN, "imgsidx"),
        (INDEX_CBIR_QUICK, "imgfcbr"),
        (INDEX_CBIR_GARBAGE, "imgsidxt1"),
        (INDEX_MIDDLE, "imgmmeta"),
        (INDEX_THUMB_MAIN, "imgsth_wide"),
        (INDEX_THUMB_QUICK, "imgfth"),
        (INDEX_CBIR_DAEMON, "cbirdaemon"),
        (INDEX_RQ, "imgsrq2baseidx"),
        (INDEX_RIM, "imgsrim3k"),
    )

    FAKE_SHARDMAP_CONFIGURATIONS = (
        (INDEX_MAIN, 432),
        (INDEX_GARBAGE, 432),
        (INDEX_CBIR_MAIN, 432),
        (INDEX_CBIR_GARBAGE, 432),
        (INDEX_MIDDLE, 1),
    )

    MODELS = {
        "l2": {
            MODELS_PRODUCTION: images_models_resources.IMAGES_BASESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE,
            MODELS_EXPERIMENTAL: images_models_resources.IMAGES_BASESEARCH_EXP_DYNAMIC_MODELS_ARCHIVE,
            MODELS_FINAL: DEFAULT_BASESEARCH_MODELS_RES_TYPE
        },
        "l3": {
            MODELS_PRODUCTION: images_models_resources.IMAGES_MIDDLESEARCH_PROD_DYNAMIC_MODELS_ARCHIVE,
            MODELS_EXPERIMENTAL: images_models_resources.IMAGES_MIDDLESEARCH_EXP_DYNAMIC_MODELS_ARCHIVE,
            MODELS_FINAL: images_models_resources.IMAGES_MIDDLESEARCH_DYNAMIC_MODELS_ARCHIVE,
        }
    }

    DEFAULT_REQUIRED_RAM = 30 * 1024
    DEFAULT_REQUIRED_DISK = 30 * 1024

    # Note: Use minimal values from all locations
    CGROUPS = {
        INDEX_QUICK: {
            "memory": {
                "low_limit_in_bytes": "3G",
                "limit_in_bytes": "4G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_CBIR_QUICK: {
            "memory": {
                "low_limit_in_bytes": "3G",
                "limit_in_bytes": "4G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_RQ: {
            "memory": {
                "low_limit_in_bytes": "7045M",  # 6.88G, kernel doesn't accept '.'
                "limit_in_bytes": "8069M",  # 7.88G, kernel doesn't accept '.'
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_CBIR_MAIN: {
            "memory": {
                "low_limit_in_bytes": "44G",
                "limit_in_bytes": "45G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_CBIR_GARBAGE: {
            "memory": {
                "low_limit_in_bytes": "23G",
                "limit_in_bytes": "24G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_MAIN: {
            "memory": {
                # Temporary raise limits
                # Waiting for patch with correct usage of WAD files from RAM disk (tmpfs)
                "low_limit_in_bytes": "75G",
                "limit_in_bytes": "80G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_GARBAGE: {
            "memory": {
                # Temporary raise limits
                # Waiting for patch with correct usage of WAD files from RAM disk (tmpfs)
                "low_limit_in_bytes": "27G",
                "limit_in_bytes": "39G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_THUMB_QUICK: {
            "memory": {
                "low_limit_in_bytes": "195M",  # 0.19G, kernel doesn't accept '.'
                "limit_in_bytes": "1219M",  # 1.19G, kernel doesn't accept '.'
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_CBIR_DAEMON: {
            "memory": {
                "low_limit_in_bytes": "11G",
                "limit_in_bytes": "12G",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_RIM: {
            "memory": {
                "low_limit_in_bytes": "1024M",
                "limit_in_bytes": "1124M",
                "recharge_on_pgfault": "1",
            }
        },
        INDEX_RIMPATCH: {
            "memory": {
                "low_limit_in_bytes": "4096M",
                "limit_in_bytes": "4196M",
                "recharge_on_pgfault": "1",
            }
        },
    }

    DISK_LIMITS = {
        INDEX_QUICK: "10G",
        INDEX_CBIR_QUICK: "10G",
        INDEX_RQ: "6G",
        INDEX_CBIR_MAIN: "120G",
        INDEX_CBIR_GARBAGE: "180G",
        INDEX_MAIN: "120G",
        INDEX_GARBAGE: "180G",
        INDEX_THUMB_QUICK: "24G",
        INDEX_CBIR_DAEMON: "4G",
        INDEX_RIM: "60G",
        INDEX_RIMPATCH: "16G",
    }

    USE_SAAS_QUICK = False

    @classmethod
    def zookeeper_path(cls, index_type, staging=False):
        return "/media-services/images/flags/deployment/{}".format(
            cls.service_id(index_type, staging=staging)
        )

    @classmethod
    def service_id(cls, index_type, meta_type=None, staging=False, is_hamster=False):
        _, region = cls.PRIEMKA_SOURCE
        if index_type == INDEX_MAIN:
            if staging:
                raise SandboxTaskFailureError(_PRISM_REMOVED)
            elif region in ['sas', 'man', 'vla']:
                return '{}-images-base-resources'.format(region)
            else:
                return 'production_imgsbase_{}'.format(region)
        elif index_type == INDEX_QUICK:
            if cls.USE_SAAS_QUICK:
                if region in ['sas', 'man']:
                    return 'prod-imgs-quick-base-{}'.format(region)
                else:
                    return 'saas_refresh_production_images_quick_base_{}'.format(region)
            else:
                # Obsolete quick services (deprecated).
                if staging:
                    raise SandboxTaskFailureError(_PRISM_REMOVED)
                    return 'prism_imgfbase_vla'
                else:
                    return 'production_{}_imgfbase'.format(region)
        elif index_type == INDEX_ULTRA:
            return 'saas_refresh_production_images_base'
        elif index_type == INDEX_GARBAGE:
            if region in ['sas', 'man', 'vla']:
                return '{}-images-base-resources'.format(region)
            else:
                return 'production_imgsbase_tier1_{}'.format(region)
        elif index_type == INDEX_CBIR_MAIN:
            if region in ['sas', 'man', 'vla']:
                return '{}-images-base-resources'.format(region)
            else:
                return 'production_imgsbase_{}'.format(region)
        elif index_type == INDEX_CBIR_QUICK:
            return 'production_{}_imgfcbr'.format(region)
        elif index_type == INDEX_CBIR_GARBAGE:
            if region in ['sas', 'man', 'vla']:
                return '{}-images-base-resources'.format(region)
            else:
                return 'production_imgsbase_tier1_{}'.format(region)
        elif index_type == INDEX_CBIR_DAEMON:
            if staging:
                return 'prism-sas-imgcbrd-yp'
            else:
                return 'production-{}-imgcbrd-yp'.format(region)
        elif index_type == INDEX_THUMB_QUICK:
            if staging:
                return 'prism_sas_imgfth'
            else:
                return 'production_{}_imgfth'.format(region)
        elif index_type == INDEX_RQ:
            return 'imgrq-base-production-{}-yp'.format(region)
        elif index_type == INDEX_MIDDLE:
            if is_hamster:
                return "hamster_imgmmeta_{}".format(region)
            else:
                return 'production_imgmmeta_{}'.format(region)
        elif index_type == INDEX_MIDDLE_RQ:
            return 'imgrq-production-{}-yp'.format(region)
        elif index_type == INDEX_INT:
            return 'production_imgsint_{}'.format(region)
        elif index_type == INDEX_CBIR_INT:
            return 'production_imgscbrint_{}'.format(region)
        elif index_type == INDEX_RIM:
            return '{}-images-rim-resources'.format(region)
        elif index_type == INDEX_RIMPATCH:
            return 'production_rimpatch_{}'.format(region)
        else:
            raise SandboxTaskFailureError("Unsupported index_type {}".format(index_type))

    @staticmethod
    def service_tags(index_type, staging=False):
        if not staging:
            if index_type == INDEX_CBIR_DAEMON:
                return "itype=cbrd;ctype=prestable;geo=sas;prj=imgs-cbir"
            else:
                raise NotImplementedError()

        if index_type == INDEX_MAIN:
            raise SandboxTaskFailureError(_PRISM_REMOVED)
            return "itype=base;ctype=prism;prj=imgs-main"
        elif index_type == INDEX_QUICK:
            raise SandboxTaskFailureError(_PRISM_REMOVED)
            return "itype=base;ctype=prism;prj=imgs-quick"
        elif index_type == INDEX_THUMB_QUICK:
            return "itype=thumb;ctype=prism;prj=imgs-quick"
        elif index_type == INDEX_CBIR_DAEMON:
            return "itype=cbrd;ctype=prism;prj=imgs-cbir"
        else:
            raise SandboxTaskFailureError("Unsupported index_type {}".format(index_type))

    @classmethod
    def index_type_for_tier(cls, tier):
        return cls.DATABASE_TIERS.get(tier)

    @classmethod
    def models_resource(cls, index_type, models_type=MODELS_FINAL):
        if index_type in (INDEX_MAIN, INDEX_QUICK, INDEX_ULTRA, INDEX_GARBAGE):
            return cls.MODELS["l2"].get(models_type)
        elif index_type == INDEX_MIDDLE:
            return cls.MODELS["l3"].get(models_type)
        else:
            raise SandboxTaskFailureError("Unsupported index type: {}".format(index_type))

    @staticmethod
    def models_url(index_type, models_type=MODELS_FINAL):
        url = "arcadia:/robots/branches/images/ranking_models/{}/{}"
        if index_type in (INDEX_MAIN, INDEX_QUICK, INDEX_ULTRA, INDEX_GARBAGE):
            if models_type == MODELS_FINAL:
                return "arcadia:/robots/branches/images/dynamic_ranking_models"
            else:
                return url.format("l2", models_type)
        elif index_type == INDEX_MIDDLE:
            if models_type == MODELS_FINAL:
                return "arcadia:/robots/branches/images/middle_dynamic_ranking_models"
            else:
                return url.format("l3", models_type)
        else:
            raise SandboxTaskFailureError("Unsupported index_type: {}".format(index_type))

    @classmethod
    def basesearch_executable_resource(cls, index_type, staging=False):
        logging.info(
            'Getting basesearch executable resource type for index_type=%s, use_saas_quick=%s',
            index_type, cls.USE_SAAS_QUICK
        )
        if cls.USE_SAAS_QUICK and index_type == INDEX_QUICK:
            logging.info('IMGSEARCH_RTYSERVER_EXECUTABLE as type for baseseaerch retrieved.')
            return resource_types.IMGSEARCH_RTYSERVER_EXECUTABLE
        elif index_type in (
            INDEX_MAIN,
            INDEX_QUICK,
            INDEX_GARBAGE,
            INDEX_CBIR_MAIN,
            INDEX_CBIR_QUICK,
            INDEX_CBIR_GARBAGE,
        ):
            if staging:
                return images_basesearch_resources.IMGSEARCH_EXECUTABLE_STAGING
            else:
                return images_basesearch_resources.IMGSEARCH_EXECUTABLE
        elif index_type == INDEX_CBIR_DAEMON:
            if staging:
                return images_daemons_resources.CBIR_DAEMON2_EXECUTABLE_STAGING
            else:
                return images_daemons_resources.CBIR_DAEMON2_EXECUTABLE
        elif index_type in (INDEX_THUMB_MAIN, INDEX_THUMB_QUICK):
            if staging:
                return images_daemons_resources.NAIL_DAEMON_EXECUTABLE_STAGING
            else:
                return images_daemons_resources.NAIL_DAEMON_EXECUTABLE
        elif index_type == INDEX_RQ:
            return images_rq_resources.IMAGES_RQ_BASESEARCH_EXECUTABLE
        elif index_type == INDEX_RIM:
            return images_daemons_resources.IMAGES_RIM_DAEMON_EXECUTABLE
        elif index_type == INDEX_RIMPATCH:
            return images_daemons_resources.IMAGES_RIMPATCH_DAEMON_EXECUTABLE
        else:
            raise SandboxTaskFailureError("Unsupported index type {}".format(index_type))

    @classmethod
    def _parse_cgroup_size(cls, cgroup_size):
        if cgroup_size.endswith("G"):
            return int(cgroup_size[:-1]) * 1024
        elif cgroup_size.endswith("M"):
            return int(cgroup_size[:-1])
        else:
            return int(cgroup_size)

    @classmethod
    def basesearch_executable_memory(cls, ctx, database_parameter):
        index_type = cls.basesearch_database_index_type(ctx, database_parameter)
        if not index_type:
            return cls.DEFAULT_REQUIRED_RAM

        cgroup_size = cls.CGROUPS[index_type]["memory"]["limit_in_bytes"]
        return cls._parse_cgroup_size(cgroup_size)

    @classmethod
    def basesearch_executable_disk(cls, ctx, database_parameter):
        index_type = cls.basesearch_database_index_type(ctx, database_parameter)
        if not index_type:
            return cls.DEFAULT_REQUIRED_DISK

        cgroup_size = cls.DISK_LIMITS[index_type]
        return cls._parse_cgroup_size(cgroup_size)

    @classmethod
    def basesearch_config_resource(cls, index_type=INDEX_MAIN, staging=False):
        if index_type in (INDEX_MAIN,):
            if staging:
                return images_basesearch_resources.IMAGES_SEARCH_CONFIG_STAGING
            else:
                return images_basesearch_resources.IMAGES_SEARCH_CONFIG
        elif index_type == INDEX_QUICK:
            if cls.USE_SAAS_QUICK:
                return images_basesearch_resources.IMGQUICK_SAAS_RTYSERVER_CONFIGS_BUNDLE
            else:
                if staging:
                    return images_basesearch_resources.IMAGES_QUICK_SEARCH_CONFIG_STAGING
                else:
                    return images_basesearch_resources.IMAGES_QUICK_SEARCH_CONFIG
        elif index_type == INDEX_GARBAGE:
            return images_basesearch_resources.IMAGES_SEARCH_TIER1_CONFIG
        elif index_type in (INDEX_CBIR_MAIN,):
            return images_basesearch_resources.IMAGES_CBIR_SEARCH_CONFIG
        elif index_type in (INDEX_CBIR_GARBAGE,):
            return images_basesearch_resources.IMAGES_CBIR_SEARCH_TIER1_CONFIG
        elif index_type == INDEX_CBIR_QUICK:
            return images_basesearch_resources.IMAGES_CBIR_QUICK_SEARCH_CONFIG
        elif index_type == INDEX_CBIR_DAEMON:
            if staging:
                return images_daemons_resources.CBIR_DAEMON2_CONFIG_STAGING
            else:
                return images_daemons_resources.CBIR_DAEMON2_CONFIG
        elif index_type in (INDEX_THUMB_MAIN, INDEX_THUMB_QUICK):
            if staging:
                return images_daemons_resources.NAIL_DAEMON_CONFIG_STAGING
            else:
                return images_daemons_resources.NAIL_DAEMON_CONFIG
        elif index_type == INDEX_RQ:
            return images_rq_resources.IMAGES_RQ_BASESEARCH_CONFIG
        elif index_type == INDEX_RIM:
            return images_daemons_resources.IMAGES_RIM_DAEMON_CONFIG
        elif index_type == INDEX_RIMPATCH:
            return images_daemons_resources.IMAGES_RIMPATCH_DAEMON_CONFIG
        else:
            raise SandboxTaskFailureError("Unsupported index type {}".format(index_type))

    @staticmethod
    def basesearch_settings_resource(index_type=INDEX_MAIN):
        if index_type == INDEX_CBIR_DAEMON:
            return images_daemons_resources.CBIR_DAEMON2_SETTINGS
        else:
            raise SandboxTaskFailureError("Unsupported index type {}".format(index_type))

    @classmethod
    def basesearch_database_prefix(cls, index_type):
        for shard_index, shard_prefix in cls.DATABASE_PREFIXES:
            if index_type == shard_index:
                return shard_prefix
        else:
            raise SandboxTaskFailureError("Unsupported index_type {}".format(index_type))

    @classmethod
    def basesearch_database_resource(cls, index_type=INDEX_MAIN):
        if cls.USE_SAAS_QUICK and index_type == INDEX_QUICK:
            return resource_types.RTYSERVER_SEARCH_DATABASE
        elif index_type in (
            INDEX_MAIN,
            INDEX_QUICK,
            INDEX_GARBAGE,
            INDEX_CBIR_MAIN,
            INDEX_CBIR_QUICK,
            INDEX_CBIR_GARBAGE,
            INDEX_RQ,
            INDEX_RIM,
        ):
            return resource_types.IMAGES_SEARCH_DATABASE
        elif index_type == INDEX_MIDDLE:
            return resource_types.IMAGES_MIDDLESEARCH_INDEX
        elif index_type in (INDEX_THUMB_MAIN, INDEX_THUMB_QUICK):
            return resource_types.IMAGES_SEARCH_DATABASE
        elif index_type == INDEX_CBIR_DAEMON:
            return images_daemons_resources.CBIR_DAEMON2_SETTINGS
        elif index_type == INDEX_INVERTED:
            return images_resource_types.IMAGES_INVERTED_INDEX_DATABASE
        elif index_type == INDEX_EMBEDDING:
            return images_resource_types.IMAGES_EMBEDDING_DATABASE
        else:
            raise SandboxTaskFailureError("Unsupported index type {}".format(index_type))

    @classmethod
    def basesearch_database_index_type(cls, ctx, database_parameter):
        database_resource = channel.sandbox.get_resource(ctx[database_parameter.name])
        if not database_resource:
            raise SandboxTaskFailureError('Failed to find database resource')
        if SHARD_INSTANCE_ATTRIBUTE_NAME not in database_resource.attributes:
            return None

        database_prefix = database_resource.attributes[SHARD_INSTANCE_ATTRIBUTE_NAME].split("-")[0]
        for shard_index, shard_prefix in cls.DATABASE_PREFIXES:
            if database_prefix.startswith(shard_prefix):
                return shard_index
        return None

    @staticmethod
    def basesearch_shooting_parameters(index_type):

        simultaneous_requests = 20
        shooting_time = 600
        if index_type == INDEX_CBIR_DAEMON:
            simultaneous_requests = 10
        elif index_type in (
            INDEX_MIDDLE,
            INDEX_MIDDLE_RQ,
            INDEX_INT
        ):
            simultaneous_requests = 10
            shooting_time = 1200

        return {
            dolbilka.DolbilkaExecutorMode.name: 'finger',
            dolbilka.DolbilkaMaximumSimultaneousRequests.name: simultaneous_requests,
            dolbilka.DolbilkaExecutorRequestsLimit.name: 0,
            dolbilka.DolbilkaExecutorTimeLimit.name: shooting_time,
            offline.WARMUP_TIME_KEY: 120,
            offline.SHUTDOWN_TIME_KEY: 2,
        }

    @staticmethod
    def middlesearch_data_resources(index_type):
        if index_type == INDEX_MIDDLE:
            return (
                images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPIRATE_URLS_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_PORNO_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_GRUESOME_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ANTIPORNO_PERVERSION_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ANTISPAM_OWNERS_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_CHILDPORNO_HOSTS_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_CV_BAN_FOLDER,
                images_bans_resources.IMAGES_MIDDLESEARCH_MALWARE_MASKS_TRIE,
                images_bans_resources.IMAGES_MIDDLESEARCH_QUERY_URL_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ROTTEN_HOSTS_BAN_FILE,
                images_bans_resources.IMAGES_MIDDLESEARCH_ROTTEN_URLS_BLOOM_FILTER,
            )
        elif index_type == INDEX_MIDDLE_RQ:
            return (
                images_rq_resources.IMAGES_RQ_MIDDLESEARCH_QUERY_BAN_FILE,
                images_rq_resources.IMAGES_RQ_MIDDLESEARCH_QUERY_PAIR_BAN_FILE,
            )
        else:
            raise SandboxTaskFailureError("Unsupported index type {}".format(index_type))

    @staticmethod
    def snippetizer_index_type(index_type):
        if index_type == INDEX_CBIR_MAIN:
            return INDEX_MAIN
        elif index_type == INDEX_CBIR_GARBAGE:
            return INDEX_GARBAGE
        else:
            return index_type

    @staticmethod
    def pumpkin_resource_attributes(domain):
        key = 'images_pumpkin_{domain}'.format(domain=domain)
        return (key, 'yes')

    @classmethod
    def testenv_resource_attributes(cls, search_type, index_type=INDEX_MAIN, is_hamster=False):
        key = 'images_testenv_{}_{}'.format(search_type, SAS_REGION)
        if search_type == INDEX_INT and index_type in (INDEX_CBIR_MAIN, INDEX_CBIR_QUICK):
            key += '_{}'.format(index_type)

        if is_hamster and search_type in (INDEX_MIDDLE) and index_type in (INDEX_MAIN):
            key += '_hamster'
        return key, 'yes'

    @classmethod
    def testenv_basesearch_queries_attributes(cls, index_type, filter_type=None, serp_type=None):
        attr_key = 'images_testenv_base_{}_{}'.format(index_type, SAS_REGION) + '_{}'
        if filter_type is not None:
            attr_key = attr_key.format(filter_type)
        if serp_type is not None:
            attr_key += "_{}".format(serp_type)
        if index_type == INDEX_QUICK and cls.USE_SAAS_QUICK:
            attr_key += "_rtindex"
        return attr_key, 'yes'

    @staticmethod
    def testenv_middlesearch_queries_attributes(search_type, filter_type=None, serp_type=None):
        attr_key = 'images_testenv_{}_{}'.format(search_type, SAS_REGION) + '_{}'
        if filter_type is not None:
            attr_key = attr_key.format(filter_type)
        if serp_type is not None:
            attr_key += "_{}".format(serp_type)
        return attr_key, 'yes'

    @classmethod
    def ensure_search_database(cls, task, queries_parameter, database_parameter):
        resource_ids = _ensure_search_database(
            task.ctx,
            queries_parameter,
            cls.AUTO_DATABASES,
            SHARD_INSTANCE_ATTRIBUTE_NAME,
            database_parameter
        )
        if resource_ids:
            task._register_dep_resource(resource_ids[0])

    @classmethod
    def ensure_snippetizer_database(cls, task, queries_parameter, database_parameter):
        resource_ids = _ensure_search_database(
            task.ctx,
            queries_parameter,
            cls.AUTO_DATABASES,
            SNIP_SHARD_INSTANCE_ATTRIBUTE_NAME,
            database_parameter
        )
        if resource_ids:
            task._register_dep_resource(resource_ids[0])

    @staticmethod
    def ensure_middlesearch_index(task, queries_parameter, index_parameter):
        resource_ids = _ensure_search_database(
            task.ctx,
            queries_parameter,
            (resource_types.IMAGES_MIDDLESEARCH_INDEX,),
            META_SHARD_INSTANCE_ATTRIBUTE_NAME,
            index_parameter
        )
        if resource_ids:
            task._register_dep_resource(resource_ids[0])

    @staticmethod
    def is_main_index(index_type):
        return index_type in (INDEX_CBIR_MAIN, INDEX_MAIN)

    @staticmethod
    @decorators.memoize
    def get_rtyserver_static_data():
        resource = apihelpers.get_last_resource(resource_types.RTYSERVER_MODELS)
        if not resource:
            raise SandboxTaskFailureError("Can't find rtyserver static data.")
        return resource.id

    @staticmethod
    @decorators.memoize
    def get_rtyserver_sw_config():
        resource = apihelpers.get_last_resource(resource_types.IMAGES_SHARDWRITER_CONFIG)
        if not resource:
            raise SandboxTaskFailureError("Can't find rtyserver shardwriter config.")
        return resource.id


class VideoSettings:
    INDICES = [INDEX_MAIN, INDEX_QUICK]
    INDICES_FOR_TESTENV = [INDEX_MAIN]
    NAME = 'video'
    IMPROXY_CONFIG_LOCATIONS = (VLA_REGION, SAS_REGION, MAN_REGION)
    IMPROXY_CONFIGS = ['improxy-video-{}'.format(x) for x in (VLA_REGION, SAS_REGION, MAN_REGION)]
    DYNAMIC_MODELS_SVN_PRODUCTION_TRUNK_URL = "arcadia:/robots/branches/video/dynamic_ranking_models/production"
    DYNAMIC_MODELS_SVN_EXPERIMENT_TRUNK_URL = "arcadia:/robots/branches/video/dynamic_ranking_models/experiment"

    DYNAMIC_MODELS_SVN_TAGS_URL = "arcadia:/robots/tags/video/dynamic_ranking_models/"

    L1_DYNAMIC_MODELS_SVN_PRODUCTION_TRUNK_URL = "arcadia:/robots/branches/video/new_l1"
    L1_DYNAMIC_MODELS_SVN_EXPERIMENT_TRUNK_URL = "arcadia:/robots/branches/video/new_l1_exp"

    MODELS_HUB_MODELS_SVN_PRODUCTION_TRUNK_URL = "arcadia:/robots/branches/video/models_hub"

    MIDDLE_DYNAMIC_MODELS_SVN_PRODUCTION_TRUNK_URL = "arcadia:/robots/branches/video/middlesearch_models/production"
    MIDDLE_DYNAMIC_MODELS_SVN_EXPERIMENT_TRUNK_URL = "arcadia:/robots/branches/video/middlesearch_models/experiment"

    TEST_REGIONS = (VLA_REGION, SAS_REGION)
    HEATUP_REGION = VLA_REGION

    PRIEMKA_SOURCE = (INDEX_MAIN, VLA_REGION)

    MEDIA_BDI_CONFIG_LOCATIONS = (SAS_REGION, MAN_REGION, VLA_REGION)
    MEDIA_BDI_HAMSTER_CONFIG_LOCATIONS = (SAS_REGION, MAN_REGION, VLA_REGION)

    RESOURCE_PRIEMKA_ATTRIBUTES = ('video_priemka', '1')
    RESOURCE_SERP_SEARCH_ATTRIBUTES = ('video_serp_search', '1')
    RESOURCE_SERP_FACTORS_ATTRIBUTES = ('video_serp_factors', '1')
    RESOURCE_RELATED_SEARCH_ATTRIBUTES = ('video_related_search', '1')
    RESOURCE_RELATED_FACTORS_ATTRIBUTES = ('video_related_factors', '1')
    RESOURCE_SNIPPETS_ATTRIBUTES = ('video_snippets', '1')
    RESOURCE_VIDEOTOP_SEARCH_ATTRIBUTES = ('video_vtop_search', '1')
    RESOURCE_VIDEOHUB_SEARCH_ATTRIBUTES = ('video_hub_search', '1')
    RESOURCE_MODELSHUB_FETCH_DOC_ATTRIBUTES = ('video_modelshub_fetch_doc', '1')

    COLLECTION_MAIN = 'yandsearch'
    COLLECTION_QUICK = 'quick'
    COLLECTION_ULTRA = 'ultra'

    INT_ID = "int"  # intermediate metasearch
    MMETA_ID = "mmeta"  # middle metasearch
    MIDDLE_TYPES = (INT_ID, MMETA_ID)  # all metasearch types

    BAN_TRIE = ('VIDEO', 'production_vidban')

    TIERS = {
        'VideoTier0': INDEX_MAIN
    }

    MMETA_NANNY_SERVICE_NAME_MAN = 'production_vidmmeta_man'
    MMETA_NANNY_SERVICE_NAME_VLA = 'production_vidmmeta_vla'
    SMIDDLE_NANNY_SERVICE_NAME = MMETA_NANNY_SERVICE_NAME_VLA
    INT_NANNY_SERVICE_NAME = 'vla_video_int'
    SBASE_NANNY_SERVICE_NAME = 'vla-video-base-resources'
    UBASE_NANNY_SERVICE_NAME = 'production_vidubase'

    DEFAULT_BASESEARCH_BINARY_RES_TYPE = resource_types.VIDEOSEARCH_EXECUTABLE
    DEFAULT_BASESEARCH_MODELS_RES_TYPE = resource_types.VIDEO_DYNAMIC_MODELS_ARCHIVE
    MIDDLE_DATA_SUBRESOURCES = (
        resource_types.VIDEO_MIDDLESEARCH_FRESH_QUERYMARKS_TRIE,
        resource_types.VIDEO_MIDDLESEARCH_VIDEOBAN_SOFT404_HASH_VEC,
        resource_types.VIDEO_MIDDLESEARCH_VIDEOBAN_TRIE,
        resource_types.VIDEO_MIDDLESEARCH_VIDEOBAN_VERSION,
        resource_types.VIDEO_MIDDLESEARCH_FASTDELIVERY_TRIE,
        resource_types.VIDEO_MIDDLESEARCH_CONTENT_ATTRS,
        video_resource_types.VIDEO_MIDDLESEARCH_VH_ATTRS_TRIE,
        video_resource_types.VIDEO_MIDDLESEARCH_VH_DUPS_TRIE,
        video_resource_types.VIDEO_MIDDLESEARCH_VIDEOHUB_TRIE,
        video_resource_types.VIDEO_MIDDLESEARCH_VIDEOFASTDELIVERY_TRIE,
    )

    EXECUTION_SPACE = 75 * 1024  # 75 Gb

    DATABASE_PREFIXES = {
        INDEX_MIDDLE: "vidmmeta",
        INDEX_MAIN: "vidsidx"
    }

    @staticmethod
    def get_nanny_token(task):
        return task.get_vault_data('robot-video-acc', 'robot_video_nanny_key')

    @classmethod
    def index_type_for_tier(cls, tier):
        return cls.TIERS.get(tier)

    @staticmethod
    def tags_dir(tags_url, branch):
        """
            Returns full repository path for specified branch
        """

        tags = branch[len(_PRE_STABLE_PREFIX):] if branch.startswith(_PRE_STABLE_PREFIX) else branch
        return os.path.join(tags_url, tags)

    @staticmethod
    def get_nanny(token):
        if not token:
            raise SandboxTaskFailureError('Use nanny service to get cms configuration instead of HEAD')

        nanny_client = NannyClient(
            api_url='https://nanny.yandex-team.ru/',
            oauth_token=token
        )

        return nanny_client

    @staticmethod
    def _get_cms_configuration(service_name, token):
        nanny_client = VideoSettings.get_nanny(token)

        current_state = nanny_client.get_service_current_state(service_name)
        for snapshot in current_state['content']['active_snapshots']:
            state = snapshot['state']
            if state == 'ACTIVE' or state == 'ACTIVATING':
                return snapshot['conf_id']

        raise SandboxTaskFailureError('Service doesn\'t have snapshot with \'ACTIVE\' or \'ACTIVATING\' state.')

    @classmethod
    def middlesearch_tag(cls, search_type, region, token):
        cms_configuration = VideoSettings._get_cms_configuration(
            cls.MMETA_NANNY_SERVICE_NAME_MAN if region == MAN_REGION else cls.MMETA_NANNY_SERVICE_NAME_VLA,
            token
        )
        logging.info('got middlesearch cms configuration: {}'.format(cms_configuration))

        instance_tag_name = [
            'a_metaprj_video',
            'a_itype_{}'.format(search_type),
        ]

        if region == PRIEMKA_REGION:
            instance_tag_name.append([
                'a_ctype_priemka',
                'a_ctype_priemkainproduction',
            ])
        else:
            instance_tag_name.extend([
                'a_geo_{}'.format(region),
                'a_ctype_{}'.format(region2ctype(region)),
            ])

        return cms_configuration, instance_tag_name

    @staticmethod
    def basesearch_config_resource():
        return resource_types.VIDEO_SEARCH_CONFIG

    @staticmethod
    def basesearch_executable_resource():
        return resource_types.VIDEOSEARCH_EXECUTABLE

    @staticmethod
    def basesearch_database_resource(index_type=INDEX_MAIN):
        if index_type in (INDEX_MAIN,):
            return resource_types.VIDEO_SEARCH_DATABASE
        else:
            raise SandboxTaskFailureError("Unsupported video index type {}".format(index_type))

    @classmethod
    def basesearch_tag(cls, index_type, region, token):
        service_name = cls.UBASE_NANNY_SERVICE_NAME if index_type == INDEX_ULTRA else cls.SBASE_NANNY_SERVICE_NAME
        cms_configuration = VideoSettings._get_cms_configuration(service_name, token)

        instance_tag_name = [
            'a_prj_video-{}'.format(index_type),
            'a_itype_base'
        ]

        if region == PRIEMKA_REGION:
            if index_type == INDEX_MAIN:
                instance_tag_name.append([
                    'a_ctype_priemka', 'a_ctype_priemkainproduction'
                ])
            else:
                raise SandboxTaskFailureError('No acceptance for specified index exist')
        else:
            instance_tag_name.extend([
                'a_geo_{}'.format(region),
                'a_ctype_{}'.format(region2ctype(region)),
            ])

        return cms_configuration, instance_tag_name

    @staticmethod
    def pumpkin_resource_attributes(domain):
        key = 'video_pumpkin_{domain}'.format(domain=domain)
        return (key, 'yes')

    @staticmethod
    def thumbdaemon_tag(index_type, region):
        cms_configuration = 'ULTRA' if index_type == INDEX_ULTRA else 'VIDEO'

        instance_tag_name = [
            'a_prj_video-{}'.format(index_type),
            'a_itype_thumb',
            'a_ctype_{}'.format(region2ctype(region)),
            'a_geo_{}'.format(region)
        ]

        return cms_configuration, instance_tag_name

    @staticmethod
    def testenv_resource_attributes(search_type, region):
        key = 'video_testenv_{search_type}_{region}'.format(
            search_type=search_type,
            region=region,
        )
        return key, 'True'

    @staticmethod
    def testenv_mmeta_hamster_cfg_attributes():
        key = 'TE_video_hamster_{}_cfg'.format(
            cluster_names.C.production_hamster_vidmmeta,
        )
        return key, None

    @staticmethod
    def testenv_basesearch_queries_attributes(index_type, region, filter_type):
        key = 'video_testenv_base_{index_type}_{region}_{filter_type}'.format(
            index_type=index_type,
            region=region,
            filter_type=filter_type)
        return key, 'yes'

    @staticmethod
    def testenv_middlesearch_queries_attributes(search_type, region, filter_type=None):
        # Note: `filter_type` is currently unused
        key = 'video_testenv_{search_type}_{region}'.format(
            search_type=search_type,
            region=region
        )
        return key, 'True'

    @staticmethod
    def heatup_basesearch_attributes(index_type):
        # Note: web uses ('Tier_priemka', 1) attributes in heatup script
        return 'video_heatup_base_{index_type}'.format(index_type=index_type), '1'

    @staticmethod
    def ensure_search_database(ctx, queries_parameter, first_database_parameter, *rest_database_parameters):
        _ensure_search_database(
            ctx,
            queries_parameter,
            (
                resource_types.VIDEO_SEARCH_DATABASE,
                resource_types.VIDEO_QUICK_SEARCH_DATABASE,
                resource_types.VIDEO_ULTRA_SEARCH_DATABASE,
            ),
            SHARD_INSTANCE_ATTRIBUTE_NAME,
            first_database_parameter,
            *rest_database_parameters
        )

    @staticmethod
    def ensure_middlesearch_index(task, queries_parameter, index_parameter):
        resource_ids = _ensure_search_database(
            task.ctx,
            queries_parameter,
            (resource_types.VIDEO_MIDDLESEARCH_INDEX,),
            META_SHARD_INSTANCE_ATTRIBUTE_NAME,
            index_parameter
        )
        if resource_ids:
            task._register_dep_resource(resource_ids[0])

    @staticmethod
    def basesearch_shooting_parameters(index_type):

        simultaneous_requests = 20
        shooting_time = 600
        if index_type in (
            INDEX_MIDDLE,
            INDEX_MIDDLE_RQ,
            INDEX_INT,
        ):
            simultaneous_requests = 10
            shooting_time = 1200

        return {
            dolbilka.DolbilkaExecutorMode.name: 'finger',
            dolbilka.DolbilkaMaximumSimultaneousRequests.name: simultaneous_requests,
            dolbilka.DolbilkaExecutorRequestsLimit.name: 0,
            dolbilka.DolbilkaExecutorTimeLimit.name: shooting_time,
            offline.WARMUP_TIME_KEY: 120,
            offline.SHUTDOWN_TIME_KEY: 2,
        }


class CbirDaemonSettings:
    NAME = 'imgcbir'
    IMPROXY_CONFIG_LOCATIONS = (SAS_REGION, MAN_REGION, VLA_REGION)
    IMPROXY_CONFIGS = ['improxy-imgcbir-{}'.format(x) for x in IMPROXY_CONFIG_LOCATIONS]


class CollectionsSettings:
    NAME = 'collections'
    INDICES = (INDEX_MAIN)

    DYNAMIC_MODELS_SVN_PRODUCTION_TRUNK_URL = "arcadia:/robots/branches/collections/ranking_models/prod"
    DYNAMIC_MODELS_SVN_EXPERIMENT_TRUNK_URL = "arcadia:/robots/branches/collections/ranking_models/exp"

    CGROUPS = {
        INDEX_MAIN: {
            "memory": {
                "low_limit_in_bytes": "5G",
                "limit_in_bytes": "10G",
                "recharge_on_pgfault": "1",
            }
        }
    }

    DEFAULT_REQUIRED_DISK = 10 * 1024

    DEFAULT_REQUIRED_MEMORY = 10 * 1024

    DISK_LIMITS = {
        INDEX_MAIN: "10G",
    }

    DATABASE_PREFIXES = (
        (INDEX_MAIN, "pdbbrds"),
    )

    @classmethod
    def basesearch_database_index_type(cls, ctx, database_parameter):
        database_resource = channel.sandbox.get_resource(ctx[database_parameter.name])
        if not database_resource:
            raise SandboxTaskFailureError('Failed to find database resource')
        if SHARD_INSTANCE_ATTRIBUTE_NAME not in database_resource.attributes:
            return None

        database_prefix = database_resource.attributes[SHARD_INSTANCE_ATTRIBUTE_NAME].split("-")[0]
        for shard_index, shard_prefix in cls.DATABASE_PREFIXES:
            if database_prefix.startswith(shard_prefix):
                return shard_index
        return None

    @classmethod
    def ensure_search_database(cls, task, queries_parameter, database_parameter):
        resource_ids = _ensure_search_database(
            task.ctx,
            queries_parameter,
            (resource_types.RTYSERVER_SEARCH_DATABASE),
            SHARD_INSTANCE_ATTRIBUTE_NAME,
            database_parameter
        )
        if resource_ids:
            task._register_dep_resource(resource_ids[0])

    @classmethod
    def basesearch_executable_disk(cls, ctx, database_parameter):
        return cls.DEFAULT_REQUIRED_DISK

    @classmethod
    def basesearch_database_prefix(cls):
        return cls.DATABASE_PREFIXES[0][1]


ALL_SETTINGS_CLASSES = [ImagesSettings, VideoSettings, CbirDaemonSettings]


ALL_IMPROXY_CONFIGS = ImagesSettings.IMPROXY_CONFIGS + VideoSettings.IMPROXY_CONFIGS


def config_name2resource_type(config_name):
    return images_balancer_resources.__dict__["BALANCER_{}_CONFIG_2".format(config_name.replace('-', '_').upper())]


def config_name_list2resource_type_list(config_name_list):
    return map(config_name2resource_type, config_name_list)


ALL_IMPROXY_RESOURCE_TYPES = config_name_list2resource_type_list(ALL_IMPROXY_CONFIGS)


def _find_search_databases(queries_id, database_types, attribute_name):
    queries_resource = channel.sandbox.get_resource(queries_id)
    if not queries_resource:
        raise SandboxTaskFailureError(
            'Failed to find specified resource with id {}'.format(queries_id))
    if attribute_name not in queries_resource.attributes:
        raise SandboxTaskFailureError(
            'Attribute {} is missing in provided resource {}'.format(attribute_name, queries_id))

    shard_instance_list = queries_resource.attributes[attribute_name].split(',')
    database_resource_list = []

    for shard_instance in shard_instance_list:
        logging.info('Searching for database with {}={}'.format(SHARD_INSTANCE_ATTRIBUTE_NAME, shard_instance))
        for database_type in database_types:
            resource = apihelpers.get_last_resource_with_attribute(
                database_type, SHARD_INSTANCE_ATTRIBUTE_NAME, shard_instance)
            if resource:
                break
        else:
            raise SandboxTaskFailureError(
                'Unable to find database resource for database with {}={}'.format(
                    SHARD_INSTANCE_ATTRIBUTE_NAME, shard_instance
                )
            )

        database_resource_list.append(resource)

    return database_resource_list


def _ensure_search_database(
    ctx,
    queries_parameter,
    database_types,
    attribute_name,
    first_database_parameter,
    *rest_database_parameters
):
    database_parameters = (first_database_parameter,) + rest_database_parameters

    if all(ctx.get(parameter.name) for parameter in database_parameters):
        return

    if not queries_parameter or queries_parameter.name not in ctx:
        return

    queries_id = ctx[queries_parameter.name]
    database_ids = [resource.id for resource in _find_search_databases(queries_id, database_types, attribute_name)]

    database_count = len(database_parameters)
    logging.info("databases required={}, databases found={}".format(database_count, database_ids))
    if database_count > len(database_ids):
        raise SandboxTaskFailureError(
            'Attribute {} should contain as minimum {} shards'.format(attribute_name, database_count))

    for parameter, resource_id in itertools.izip(database_parameters, database_ids):
        logging.info('Setting {} to {}'.format(parameter.name, resource_id))
        ctx[parameter.name] = resource_id

    return database_ids
