import ConfigParser
import contextlib
import json
import logging
import os.path
import tarfile
import threading
import urllib2

from sandbox.sandboxsdk import parameters

from sandbox.projects.yasm import resource_types
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
import sandbox.projects.common.search.components.component as components_common


DEFAULT_INSTANCE_TAG = (
    ("ctype", "ctype"),
    ("geo", "geo"),
    ("prj", "prj"),
)


class YasmCoreParameter(parameters.ResourceSelector):
    name = 'yasmcore_resource_id'
    description = 'Yasm core'
    resource_type = resource_types.YASMCORE
    required = True


class YasmAgent(
    components_common.ProcessComponentMixin,
    components_common.WaitUrlComponentMixin,
    components_common.Component
):

    """Wrapper over yasmagent process"""

    def __init__(self, agent_dir, agent_port, log_dir, instance_type, instance_port, instance_tag):
        instance_autotags = " ".join("a_{}_{}".format(key, value) for key, value in instance_tag)

        self.port = agent_port
        instance_line = "localhost:{}@name a_itype_{} {}".format(
            instance_port,
            instance_type,
            instance_autotags
        )

        components_common.ProcessComponentMixin.__init__(
            self,
            args=[
                os.path.join(agent_dir, "yasmagent", "run.py"),
                "--fg",
                "--http-port={}".format(self.port),
                "--config={}".format(os.path.join(agent_dir, "yasmagent", "agent.conf")),
                "--yasmutil-dir={}".format(os.path.join(agent_dir, "yasmagent", "lib")),
                "--type-conf-dir={}".format(os.path.join(agent_dir, "yasmagent", "CONF")),
                "--log-dir={}".format(log_dir),
                "--instance-getter", "echo '{}'".format(instance_line),
            ],
            log_prefix="yasmagent"
        )
        components_common.WaitUrlComponentMixin.__init__(
            self,
            url="http://localhost:{}/version".format(self.port)
        )


class YasmListener(components_common.Component):

    """Thread to collect signals from agent"""

    class _Thread(threading.Thread):

        def __init__(self, aggr_url, instance_type, instance_tag):
            threading.Thread.__init__(self)
            self.__aggr_url = aggr_url
            self.__agent_signals = set()
            self.__event = threading.Event()
            self.__instance_type = instance_type
            self.__instance_tag = instance_tag

        def run(self):
            self.__sleep()
            while not self.__is_stopped():
                try:
                    self.__query()
                except Exception as e:
                    logging.warning("Failed to retrieve data from agent: {}".format(e))
                self.__sleep()

        def terminate(self):
            self.__event.set()

        def signals(self):
            return self.__agent_signals

        def __is_stopped(self):
            return self.__event.is_set()

        def __sleep(self):
            self.__event.wait(3)

        def __query(self):
            logging.info("Querying agent...")
            with contextlib.closing(urllib2.urlopen(self.__aggr_url)) as connection:
                aggr = json.load(connection)["aggr"]
                logging.debug("Aggr:\n%s", json.dumps(aggr, indent=2))
                for instance_type, instance_tag, instance_signals in aggr:
                    instance_tag = instance_tag.split("|", 1)[0]
                    if instance_type == self.__instance_type and instance_tag == self.__instance_tag:
                        for signal_name in instance_signals:
                            if signal_name.startswith("unistat-"):
                                self.__agent_signals.add(signal_name)

    def __init__(self, agent_port, instance_type, instance_tag):
        self.__aggr_url = "http://localhost:{}/json/aggr".format(agent_port)
        self.__thread = self._Thread(self.__aggr_url, instance_type, instance_tag)

    def dump(self, signals_path):
        with open(signals_path, "w") as signals_file:
            found_signals = self.__thread.signals()
            eh.verify(found_signals, "No signals dumped! Check output from url: {}".format(self.__aggr_url))
            for signal in sorted(found_signals):
                signals_file.write(signal + "\n")

    def start(self):
        self.__thread.start()

    def wait(self):
        pass

    def stop(self):
        self.__thread.terminate()


class YasmAgentTask:
    """
        Mixin class for tests with yasmagent
    """
    input_parameters = (YasmCoreParameter,)

    def _yasmagent(self, target, instance_type="base", instance_tag=DEFAULT_INSTANCE_TAG):
        """Returns yasmagent component"""

        yasmcore_tar = self.sync_resource(self.ctx[YasmCoreParameter.name])
        yasmcore_dir = self.abs_path("yasmcore_dir")
        with tarfile.open(yasmcore_tar) as yasmcore_tar_file:
            yasmcore_tar_file.extractall(yasmcore_dir)

        yasmcore_logs = utils.create_misc_resource_and_dir(
            self.ctx,
            "yasmcore_logs_resource_id",
            "{} yasmcore logs".format(self.descr),
            "yasmcore_logs"
        )

        self._init_yasmagent(yasmcore_dir, instance_type)

        return YasmAgent(
            agent_dir=yasmcore_dir,
            agent_port=int(target.port) + 100,  # FIXME: don't convert self.port to string in SearchExecutableComponent
            log_dir=yasmcore_logs,
            instance_port=target.port,
            instance_type=instance_type,
            instance_tag=instance_tag
        )

    def _init_yasmagent(self, yasmcore_dir, instance_type):
        # Note: This is a basesearch specific.
        #       We need to see all available signals for stability of tests

        config_path = os.path.join(yasmcore_dir, "yasmagent", "CONF", "agent.{}.conf".format(instance_type))
        config_object = ConfigParser.ConfigParser()
        config_object.read(config_path)
        logging.debug("Sections={}".format(config_object.sections()))
        self._set_yasm_config_options(config_object)
        with open(config_path, "w") as config_file:
            config_object.write(config_file)

    def _set_yasm_config_options(self, config_object):
        config_object.set("options_unistat", "url", "/tass?level=-1")

    def _yasmlistener(self, agent, instance_type="base", instance_tag=DEFAULT_INSTANCE_TAG):
        """Returns component to read yasmagent output"""
        instance_taglist = ";".join("{}={}".format(key, value) for key, value in instance_tag)
        return YasmListener(agent_port=agent.port, instance_type=instance_type, instance_tag=instance_taglist)
