# -*- coding: utf-8 -*-

from datetime import datetime
import json
import logging
import os.path

from six.moves import range
import requests


SOLOMON_URL = 'http://solomon.yandex.net/push/json'
SOLOMON_URL_V2 = 'http://solomon.yandex.net/api/v2'


def create_sensors(stats):
    sensors = []
    timestamp = int(datetime.now().strftime('%s'))
    for k, v in stats.items():
        sensors.append({
            'labels': {
                'sensor': str(k),
            },
            'ts': timestamp,
            'value': float(v),
        })
    return sensors


def create_sensors_from_file(filename):
    return create_sensors(json.load(open(filename)))


def upload_to_solomon(common_labels, sensors, try_count=3):
    """
        Загружает в solomon список сенсоров через Push API
    """
    logging.info('Trying to upload stats to solomon')
    solomon_json = {
        'commonLabels': common_labels,
        'sensors': sensors,
    }
    headers = {
        'Content-type': 'application/json',
    }
    for i in range(try_count):
        try:
            logging.info('Attempt to push number {}'.format(i + 1))
            response = requests.post(
                SOLOMON_URL,
                data=json.dumps(solomon_json, separators=(',', ':')),
                headers=headers,
            )
            response.raise_for_status()
            break
        except requests.exceptions.HTTPError:
            if i == try_count - 1:
                raise
    logging.info('response: \n{}'.format(response.text))
    logging.info('Done')


def push_to_solomon_v2(token, params, sensors, common_labels=(('host', 'solomon-push'),), try_count=5, log_level=logging.INFO, dump_response_content=False):
    """
        Загружает в solomon список сенсоров через Push API
    """
    logging.log(log_level, 'Trying to upload stats to solomon')
    if not isinstance(common_labels, dict):
        common_labels = dict(common_labels)

    solomon_json = {
        'commonLabels': common_labels,
        'sensors': sensors,
    }
    headers = {
        'Content-type': 'application/json',
        'Authorization': 'OAuth {}'.format(token),
    }

    for i in range(try_count):
        try:
            logging.log(log_level, 'Attempt to push number %s', i + 1)
            url = os.path.join(SOLOMON_URL_V2, 'push')
            response = requests.post(
                url,
                params=params,
                data=json.dumps(solomon_json, separators=(',', ':')),
                headers=headers,
            )
            if dump_response_content:
                logging.log(log_level, 'Solomon response content: %s', response.content)
            response.raise_for_status()
            break
        except requests.exceptions.RequestException as e:
            logging.debug('error response: %s', e.response)
            if e.response:
                logging.debug('%s', e.response.text)
            if i == try_count - 1:
                raise

    logging.log(log_level, 'response: \n%s', response.text)
    logging.log(log_level, 'Done')
