# -*- coding: utf-8 -*-
"""
Here should be only common patterns for staff usage
PEP8 is obligatory here
"""

import logging

from sandbox.projects.common import requests_wrapper
import sandbox.projects.release_machine.core.const as rm_const


class StaffClient(object):
    STAFF_URL = rm_const.Urls.STAFF_API

    def __init__(self, token):
        self._token = token

    def api_call(self, method, suffix, params):
        """
        :param method: method of request post_r/sget_r complete list in requests_wrapper
        :param suffix: staff api ruchkas
        :param params: params
        :return request.response
        """
        headers = {'Authorization': 'OAuth {}'.format(self._token)}
        logging.debug('api call %s%s with method: %s and params: %s', StaffClient.STAFF_URL, suffix, method, params)
        api_method = getattr(requests_wrapper, method)
        r = api_method('{0}{1}'.format(StaffClient.STAFF_URL, suffix), headers=headers, params=params, verify=False)
        return r

    def persons_info(self, params):
        """
        :return: json response for persons api calls
        """
        response = self.api_call(suffix='persons', method='post_r', params=params)
        try:
            return response.json()
        except ValueError as e:
            logging.exception('Cannot get JSON from response', e)
            return None

    def get_head_for_person(self, login):
        """
        :return list of heads for person
        """
        if not login:
            return None
        params = {
            "login": login,
            "_fields": "department_group.department.heads,department_group.ancestors.department.heads",
        }

        response = self.persons_info(params=params)
        if not response:
            logging.error("Empty json response %s, %s not found in Staff", response, login)
            return None
        result = response.get("result", None)
        if not result:
            logging.error("Field result not found in response %s, login %s not found in Staff", response, login)
            return None
        groups = [result[0]["department_group"]] + list(reversed(result[0]["department_group"]["ancestors"]))
        for group in groups:
            all_heads = [(member["person"]["login"], member["role"]) for member in group["department"]["heads"]]
            chiefs = [head[0] for head in all_heads if head[1] == 'chief']
            if all_heads and login not in chiefs:
                heads = [head[0] for head in all_heads if head[0] != login]
                return heads[0]
        return None

    def get_person_is_dismissed(self, login):
        """
        :return False when staff login is_dismissed, otherwise True
        """
        params = {
            "login": login,
            "_fields": "official.is_dismissed",
        }
        response = self.persons_info(params=params)
        if not response:
            logging.error("Empty json response %s, %s not found in Staff", response, login)
            return True
        result = response.get("result", None)
        if not result:
            logging.error("Field result not found in response %s, %s not found in Staff", response, login)
            return True
        return result[0]['official']['is_dismissed']

    def get_person_messenger(self, login, acctype='telegram'):
        """
        :return tg when staff login has tg, otherwise None
        """
        account = None
        params = {
            "login": login,
            "_fields": "accounts",
        }
        response = self.persons_info(params=params)
        if not response:
            logging.error("Empty json response %s for login %s", response, login)
            return None
        accounts = response.get("result", None)
        if not accounts:
            logging.error('Field result not found in response %s for login %s and type %s', response, login, acctype)
            return None
        accounts = accounts[0].get('accounts', [])
        for item in accounts:
            if not item.get('type') == acctype:
                continue
            if not item['private']:
                return item['value']
            account = item['value']
        return account
