# -*- coding: utf-8 -*-
"""
Here should be only common patterns for startrek usage
PEP8 is obligatory here
"""

import datetime
import logging
import requests
import traceback

from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine.core import const as rm_const
from copy import deepcopy


class ST(object):
    ST_URL = "https://st.yandex-team.ru"

    @staticmethod
    def issue_link(key):
        return "{}/{}".format(ST.ST_URL, key)

    @staticmethod
    def html_issue_link(key):
        return '<a href="{}" target="_blank">{}</a>'.format(ST.issue_link(key), key)

    @staticmethod
    def get_summonees(
        issue,
        importance=1,
        get_devops_from=None,
        extra_summonees=[[], [], []],
        token=None,
    ):
        """
        :param get_devops_from: issue to get devops from
        :param importance: a number of comments of robot in the task
        """
        # находим дежурного
        duty = ST.get_issue_devops(issue, token)
        logging.info('Get_duty %s', duty)
        # ищем исполнителя
        assignee = ST.get_assignee(issue, token)
        logging.info('Get_assignee %s', assignee)
        people = []
        if importance == 1:
            if assignee:
                people = assignee
            else:
                if duty:
                    people = duty
                elif extra_summonees:
                    people = deepcopy(extra_summonees[0])
        elif importance == 2:
            if duty:
                people = duty
            elif extra_summonees:
                people = deepcopy(extra_summonees[0])
        elif importance == 3:
            if assignee:
                people = ST.get_head_for_person(assignee[0], token)
            elif extra_summonees:
                people = deepcopy(extra_summonees[0])
        elif importance >= 4:
            # Проверка extra_summonees для предостережения деления на ноль и
            # выхода за границы списка.
            if extra_summonees:
                extra_summonees_index = (importance - 4) % len(extra_summonees)
                people = deepcopy(extra_summonees[extra_summonees_index])
            elif assignee:
                people = ST.get_head_for_person(assignee[0], token)
        logging.info('people %s', people)
        logging.info('importance %s', importance)
        return people

    @staticmethod
    def get_assignee(issue, token):
        # Исполнитель
        logging.info('issue %s', issue)
        try:
            current_assignee = issue.assignee.login
            logging.info('current_assignee = %s', current_assignee)
            logging.info('get_person_is_dismissed %s', ST.get_person_is_dismissed(current_assignee, token))
            if not ST.get_person_is_dismissed(current_assignee, token):
                logging.info('current_assinee_is_not_dismissed = %s', current_assignee)
                return [current_assignee]
            else:
                logging.info('get_head_for_person %s', ST.get_head_for_person(current_assignee, token))
                return ST.get_head_for_person(current_assignee, token)
        except Exception as e:
            logging.error("failed to get issue devops %s", e)
            return []

    @staticmethod
    def get_issue_devops(issue, token):
        # Дежурный
        try:
            logging.debug("Issue duty type: %s", type(issue.duty))
            logging.debug("Issue duty: %s", issue.duty)
            duty = [person.id for person in issue.duty if not ST.get_person_is_dismissed(person.id, token)]
            logging.debug("get_issue_devops returns: %s", duty)
            return duty
        except Exception as e:
            eh.log_exception("failed to get issue devops", e)
            return []

    @staticmethod
    def update_comment(task, issue, last_comment_id, text, summonees):
        logging.debug(
            "{}update comment '{}' with summonees: {} to {}".format(
                "NOT (because dry_run is True) " if task.Parameters.dry_run else "",
                text,
                summonees,
                issue.key,
            )
        )
        if task.Parameters.dry_run:
            return
        issue.comments[last_comment_id].update(
            text=text,
            summonees=summonees,
            params={'notify': False, 'isAddToFollowers': False},
        )

    @staticmethod
    def ping(task, issue, text, summonees):
        logging.debug(
            "{}posting comment '{}' with summonees: {} to {}".format(
                "NOT (because dry_run is True) " if task.Parameters.dry_run else "",
                text,
                summonees,
                issue.key,
            )
        )
        if task.Parameters.dry_run:
            return
        issue.comments.create(text=text, summonees=summonees, params={'notify': False, 'isAddToFollowers': False})

    @staticmethod
    def count_ping_comments_before(issue, robot_login=rm_const.ROBOT_RELEASER_USER_NAME, substr=''):
        logging.debug(
            "Try to get commented users for issue: %s, robot_login is %s, substr is %s", issue, robot_login, substr
        )
        commented_users = [comment.updatedBy.login for comment in issue.comments.get_all() if substr in comment.text]
        logging.debug("Receive list of commented users: %s", str(commented_users))
        pings = len([i for i in commented_users if i == robot_login])
        return pings

    @staticmethod
    def time_since_last_bot_message(issue, robot_login):
        changelog = list(issue.changelog)
        pings_robot = [update for update in reversed(changelog) if update.updatedBy.login == robot_login]
        if pings_robot:
            last_ping_parsed = datetime.datetime.strptime(pings_robot[0].updatedAt[:-9], '%Y-%m-%dT%H:%M:%S')
            return (datetime.datetime.now() - last_ping_parsed).days
        return 0

    @staticmethod
    def time_since_last_not_ping_update(issue, robot_login):
        changelog = list(issue.changelog)
        last_not_ping = changelog[0].updatedAt
        for update in reversed(changelog):
            if update.updatedBy.login != robot_login:
                last_not_ping = update.updatedAt
                break
        last_not_ping_parsed = datetime.datetime.strptime(last_not_ping[:-9], '%Y-%m-%dT%H:%M:%S')
        return (datetime.datetime.now() - last_not_ping_parsed).days

    @staticmethod
    def is_issue_closed(issue, resolutions=None):
        if str(issue.status.key) == "closed":
            return True
        try:
            if hasattr(issue, 'resolution') and issue.resolution:
                if resolutions is not None:
                    if str(issue.resolution.key) in resolutions:
                        return True
                else:
                    return True
        except Exception as e:
            eh.log_exception('Resolution checking failed', e)
        return False

    @staticmethod
    def is_issue_forsaken(issue, silence_time):
        upd_time = datetime.datetime.strptime(issue.updatedAt[:10], "%Y-%m-%d")
        if (datetime.datetime.now() - upd_time).days > silence_time:
            logging.debug("Issue %s is forsaken", issue.key)
            return True
        logging.debug("Issue %s is relevant enough", issue.key)
        return False

    @staticmethod
    def find_tickets_by_query(st_helper, query):
        logging.info("Process query:\n%s", query)
        issues = list(st_helper.st_client.issues.find(query))
        logging.debug("%s tickets found: %s", len(issues), issues)
        return issues

    class Query(object):
        @staticmethod
        def query_url(query):
            return "{}/filters/filter?query={}".format(ST.ST_URL, query)

        def __init__(
            self,
            product=None,
            queue='SPI',
            sort=True,
            created='2018-01-01',
            not_closed=True,
            extra_filter=None,
            updated=None,
            duty=None,
            tags_to_ignore=[],
            sre_times_filter=None,
        ):
            self.product = product
            self.queue = queue
            self.sort = sort
            self.created = created
            self.not_closed = not_closed
            self.extra_filter = extra_filter
            self.updated = updated
            self.duty = duty
            self.tags_to_ignore = tags_to_ignore
            self.sre_times_filter = sre_times_filter

        def build_query(self):
            q = []
            q.append('Queue: {}'.format(self.queue))
            if self.product:
                q.append('AND Tags: "product:{}"'.format(self.product))
            if self.not_closed:
                q.append('AND "Resolution": empty()')
            if self.extra_filter:
                q.append('AND {}'.format(self.extra_filter))
            if self.duty:
                q.append('AND Duty: {}'.format(self.duty))
            if self.tags_to_ignore:
                q.extend(['AND Tags: !"{}"'.format(tag) for tag in self.tags_to_ignore])
            if self.sre_times_filter:
                q.append('AND {}'.format(self.sre_times_filter))
            if self.sort:
                q.append('"Sort By": Key DESC AND')
            if self.created:
                q.append('Created: >= {}'.format(self.created))
            if self.updated:
                q.append('Updated: {}'.format(self.updated))
            return ' '.join(q)

        def build_short_query_url(self):
            return lb.short_url(self.query_url(self.build_query()))

    @staticmethod
    def forsaken_incidents_queries(days_of_silence, st_product, st_queue, tags_to_ignore, extra_filter):
        query = ST.Query(
            product=st_product,
            extra_filter=extra_filter,
            updated='< now() - {}d'.format(days_of_silence) if days_of_silence else None,
            queue=st_queue,
            tags_to_ignore=tags_to_ignore,
        )
        return query.build_query(), query.build_short_query_url()

    @staticmethod
    def unowned_incidents_queries(st_product, st_queue, tags_to_ignore):
        query = ST.Query(
            product=st_product,
            duty='empty()',
            queue=st_queue,
            tags_to_ignore=tags_to_ignore,
        )
        return query.build_query(), query.build_short_query_url()

    TAGS_POSSIBLE_VALUES = {
        'impact': [
            'internal',
            'external',
        ],
        'support_line': [
            'marty',
            'devops',
            'service',
        ],
        'area': [
            'human',
            'operations',
            'release',
            'bug',
            'tools',
            'service',
            'net',
            'hardware',
            'platform',
            'monitoring',
            'mds',
            'apphost',
            'balancer',
            'saas',
            'external',
            'its',
            'config',
        ]
    }

    @staticmethod
    def tags_not_set_queries(st_product, st_queue, tag):
        query = ST.Query(
            product=st_product,
            created="2018-04-05",
            tags_to_ignore=["{}:{}".format(tag, v) for v in ST.TAGS_POSSIBLE_VALUES[tag]],
            queue=st_queue,
            extra_filter='"created": <= now() - 2d',
        )
        return query.build_query(), query.build_short_query_url()

    @staticmethod
    def sre_time_not_set_queries(sre_times_filter, st_product, st_queue):
        query = ST.Query(
            product=st_product,
            created="2018-04-05",
            sre_times_filter=sre_times_filter,
            queue=st_queue,
            extra_filter='"created": <= now() - 2d',
        )
        return query.build_query(), query.build_short_query_url()

    @staticmethod
    def empty_sre_time_incidents_queries(sre_times_filter, st_queue, tags_to_ignore=[]):
        query = ST.Query(
            sre_times_filter=sre_times_filter,
            queue=st_queue,
            tags_to_ignore=tags_to_ignore,
            created='2018-05-01',
            not_closed=False,
            extra_filter='"created": <= now() - 2d',
        )
        return query.build_query(), query.build_short_query_url()

    @staticmethod
    def prod_incidents_query(st_product, st_queue):
        query = ST.Query(
            product=st_product,
            not_closed=False,
            queue=st_queue,
        )
        return query.build_query()

    @staticmethod
    def check_department_for_person(login, department, oauth_token):
        """It is a piece of staff-api, but it is useful here"""
        if not login:
            return False
        session = requests.Session()
        session.headers.update({'Authorization': 'OAuth {}'.format(oauth_token)})
        params = {
            "login": login,
            "_one": 1,
            "_fields": "login,department_group",
        }

        r = session.post(rm_const.Urls.STAFF_API + "persons", params=params, verify=False)
        try:
            r.raise_for_status()
        except Exception:
            logging.error("POST request failed with code %s", r.status_code)
            logging.debug("Response:\n%s", r.text)
            logging.debug("Traceback:\n%s", traceback.format_exc())
            raise
        result = r.json()
        if result["department_group"]["url"] == department:
            return True
        for one_department in result["department_group"]["ancestors"]:
            if one_department["url"] == department:
                return True
        return False

    @staticmethod
    def get_head_for_person(login, oauth_token):
        """It is a piece of staff-api, but it is useful here"""
        if not login:
            return []
        session = requests.Session()
        session.headers.update({'Authorization': 'OAuth {}'.format(oauth_token)})
        params = {
            "login": login,
            "_fields": "department_group.department.heads,department_group.ancestors.department.heads",
        }

        r = session.post(rm_const.Urls.STAFF_API + "persons", params=params, verify=False)
        try:
            r.raise_for_status()
        except Exception:
            logging.error("POST request failed with code %s", r.status_code)
            logging.debug("Response:\n%s", r.text)
            logging.debug("Traceback:\n%s", traceback.format_exc())
            raise
        result = r.json()["result"][0]
        groups = [result["department_group"]] + list(reversed(result["department_group"]["ancestors"]))
        for group in groups:
            all_heads = [(member["person"]["login"], member["role"]) for member in group["department"]["heads"]]
            chiefs = [head[0] for head in all_heads if head[1] == 'chief']
            logging.info("chiefs %s", chiefs)
            if chiefs and login not in chiefs:
                logging.info("login %s chiefs %s", login, chiefs)
                return [chiefs[0]]
        return []

    @staticmethod
    def is_deadline_ok(issue):
        """
        returns True when deadline is set and not expired, otherwise False
        """
        if not issue.deadline:
            return False
        logging.debug("Checking if deadline = %s is expired", issue.deadline)
        return datetime.datetime.strptime(issue.deadline, "%Y-%m-%d").date() >= datetime.datetime.today().date()

    @staticmethod
    def is_sprint_ok(issue):
        """
        returns True when at least one sprint are not expired, otherwise False
        """
        for sprint in issue.sprint:
            logging.debug("Checking if sprint with endDate = %s is expired", sprint.endDate)
            if datetime.datetime.strptime(sprint.endDate, "%Y-%m-%d").date() >= datetime.datetime.today().date():
                return True
        return False

    @staticmethod
    def get_person_is_dismissed(login, oauth_token):
        """
        :param login: assignee login
        :param oauth_token: oauth token
        returns True when staff login is_dismissed, otherwize False
        """
        session = requests.Session()
        session.headers.update({'Authorization': 'OAuth {}'.format(oauth_token)})
        params = {
            "login": login,
            "_fields": "official.is_dismissed",
        }
        logging.info('result is_dismissed %s', params)
        r = session.post(rm_const.Urls.STAFF_API + "persons", params=params, verify=False)
        try:
            r.raise_for_status()
        except Exception:
            logging.error("POST request failed with code %s", r.status_code)
            logging.debug("Response:\n%s", r.text)
            logging.debug("Traceback:\n%s", traceback.format_exc())
            raise
        result = r.json()["result"][0]
        logging.info('result is_dismissed %s', result)
        employment_status = result['official']['is_dismissed']
        return employment_status
