# -*- coding: utf-8 -*-

"""
    String utils
    Maintainer: mvel@

    NOTE: This module MUST NOT contain/import sandbox-specific things
"""
import six
import hashlib


def shift_right(txt, n=4):
    """
        Shift multiline text to the right
    """
    spaces = " " * n
    return spaces + ("\n" + spaces).join(txt.split("\n"))


def shift_list_right(items, n=4):
    """
        Same as shift_right, but for items
    """
    # first, convert items to string if they are not
    items = [str(item) for item in items]
    spaces = " " * n
    return spaces + ("\n" + spaces).join(items)


def all_to_unicode(any_obj):
    return any_obj.decode("utf-8") if isinstance(any_obj, six.binary_type) else six.text_type(any_obj)


def all_to_str(any_obj):
    if six.PY3:
        return str(any_obj)
    return any_obj.encode("utf-8") if isinstance(any_obj, six.text_type) else six.binary_type(any_obj)


def strings_to_utf8(data):
    if isinstance(data, dict):
        return {strings_to_utf8(key): strings_to_utf8(value) for key, value in data.iteritems()}
    elif isinstance(data, list):
        return [strings_to_utf8(element) for element in data]
    elif isinstance(data, tuple):
        return tuple((strings_to_utf8(element) for element in data))
    elif isinstance(data, six.text_type):
        return to_utf8(data)
    else:
        return data


def to_utf8(value, errors="ignore"):
    """
    Convert value to utf-8 encoding.
    :param value: value (string or unicode string)
    :param errors: can be 'strict' (raises UnicodeDecodeError),
    'ignore', 'replace' or 'backslashreplace'.
    See https://pythononline.ru/osnovy/encode-decode
    """
    if not isinstance(value, six.string_types):
        value = six.text_type(value)
    if type(value) == six.text_type:
        value = value.decode("utf-8", errors=errors)
    return value.encode("utf-8", errors=errors)


def md5_hexdigest(value):
    """
    :param value: any string/unicode
    :return hexdigest of the value (e.g. md5)
    """
    return hashlib.md5(to_utf8(value)).hexdigest()


def dedent(text):
    """
    Fine drop-in replacement for textwrap.dedent stupid function that properly de-indents
    blocks with some lines starting from non-whitespace characters.
    :param text: text to trim at left
    :return: de-indented text
    """
    lines = text.split("\n")
    lines = [line.lstrip() for line in lines]
    return "\n".join(lines)


def left_strip(el, prefix):
    """ Strips prefix at the left of el """
    if prefix and el.startswith(prefix):
        return el[len(prefix):]
    return el


def right_strip(el, tail):
    """ Strips prefix at the left of el """
    if tail and el.endswith(tail):
        return el[:-len(tail)]
    return el


def parse_attrs(s):
    """
    Extract "attributes" from string to { attr_name : attr_value} dict
    Input format: "attr1=value1,attr2=value2,..."
    Please note that spaces are IGNORED in both names and attribute' values.
    Also applies `urllib.parse.unquote` for both keys and values.

    Raises exception on invalid input.
    """
    if not s:
        return {}

    attrs = s.replace(' ', '').split(',')

    result = {}

    for ap in attrs:
        # skip empty pairs, for correct parsing of "attr1=value1," expression
        if not ap:
            continue

        attr = ap.split('=')
        if len(attr) != 2:
            raise Exception('Cannot parse attribute: `{}`'.format(ap))
        key, value = attr

        key = six.moves.urllib.parse.unquote(key)
        value = six.moves.urllib.parse.unquote(value)
        result[key] = value

    return result


def parse_list_unique_sorted(value):
    """
    Given a semicolon-separated :param value: return a list of its items stripped, sorted
    and with all duplicates and empty elements removed. Former `parse_list` (see RMDEV-2465)

    >>> parse_list_unique_sorted("A;B;C")
    ['A', 'B', 'C']
    >>> parse_list_unique_sorted("A;;B;C")  # note the double ';;'
    ['A', 'B', 'C']
    >>> parse_list_unique_sorted("A;   B  ; C")  # note the spaces
    ['A', 'B', 'C']
    >>> parse_list_unique_sorted("B;A;C")  # note the non-lexicographical order
    ['A', 'B', 'C']
    >>> parse_list_unique_sorted("a;b;c")  # note the lowercase letters
    ['a', 'b', 'c']
    >>> parse_list_unique_sorted("A;B;A;C")  # note the duplicate 'A'
    ['A', 'B', 'C']

    :param value: original string with semicolon-separated values
    :return: a list of strings
    """
    return sorted(
        set(
            filter(
                None,
                (
                    e.strip() for e in value.split(';')
                )
            )
        )
    )


def parse_list_exact(value):
    """
    Given a semicolon-separated :param value: return a list of stripped items

    RMDEV-2465

    >>> parse_list_exact("A;B;C")
    ['A', 'B', 'C']
    >>> parse_list_exact("A;;B;C")  # note the double ';;'
    ['A', '', 'B', 'C']
    >>> parse_list_exact("A;   B  ; C")  # note the spaces
    ['A', 'B', 'C']
    >>> parse_list_exact("B;A;C")  # note the non-lexicographical order
    ['B', 'A', 'C']
    >>> parse_list_exact("a;b;c")  # note the lowercase letters
    ['a', 'b', 'c']

    As opposed to the `parse_list_unique_sorted` function (former `parse_list`, see RMDEV-2465)
    `parse_list_exact` DOES NOT change the order of the elements.
    Neither does it perform any deduplication or filtering.
    The only transformation applied to the elements is strip (any leading or trailing whitespaces are removed).

    :param value: original string with semicolon-separated values
    :return: a list of strings, :param value: split by semicolon with each item stripped
    """
    return [item.strip() for item in value.split(";")]
