import six
import pytest

from sandbox.projects.common import string


@pytest.mark.parametrize(
    "el, prefix, expected",
    [
        ("prefix:something", "prefix:", "something"),
        ("bad_prefix:something", "prefix:", "bad_prefix:something"),
        ("fix:something", "prefix:", "fix:something"),
        ("random", "prefix", "random"),
        ("str", "", "str"),
        ("aabc", "a", "abc"),
    ]
)
def test_left_strip(el, prefix, expected):
    result = string.left_strip(el, prefix)
    assert isinstance(result, six.string_types)
    assert result == expected


@pytest.mark.parametrize(
    "el, tail, expected",
    [
        ("something@suffix", "@suffix", "something"),
        ("something@suffix_", "@suffix", "something@suffix_"),
        ("something@suffi", "@suffix", "something@suffi"),
        ("random", "qqq", "random"),
        ("str", "", "str"),
        ("abcc", "c", "abc"),
    ]
)
def test_right_strip(el, tail, expected):
    result = string.right_strip(el, tail)
    assert isinstance(result, six.string_types)
    assert result == expected


def test_parse_attrs():
    d = string.parse_attrs('')
    assert len(d) == 0

    d = string.parse_attrs('x=y')
    assert len(d) == 1

    d = string.parse_attrs('x=y,')
    assert len(d) == 1

    d = string.parse_attrs('x=y, ')
    assert len(d) == 1

    d = string.parse_attrs('x=y,aaaa=bb__bb')
    assert d['x'] == 'y'
    assert d['aaaa'] == 'bb__bb'

    d = string.parse_attrs('x=y,aaaa=bb  bb')
    assert d['x'] == 'y'
    assert d['aaaa'] == 'bbbb'

    # TODO: unquote tests


@pytest.mark.parametrize(["input_str", "expected_result"], [
    ("A;B;C", ["A", "B", "C"]),
    ("A;;B;C", ["A", "", "B", "C"]),
    ("A;   B  ; C", ["A", "B", "C"]),
    ("B;A;C", ["B", "A", "C"]),
    ("a;b;c", ["a", "b", "c"]),
])
def test__parse_list_exact(input_str, expected_result):
    assert string.parse_list_exact(input_str) == expected_result


@pytest.mark.parametrize(["input_str", "expected_result"], [
    ("A;B;C", ["A", "B", "C"]),
    ("A;;B;C", ["A", "B", "C"]),
    ("A;   B  ; C", ["A", "B", "C"]),
    ("B;A;C", ["A", "B", "C"]),
    ("a;b;c", ["a", "b", "c"]),
    ("A;B;A;C", ["A", "B", "C"]),
])
def test__parse_list_unique_sorted(input_str, expected_result):
    assert string.parse_list_unique_sorted(input_str) == expected_result
