# -*- coding: utf-8 -*-

import json
import logging
import os

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types import resource as ctr
import sandbox.projects.common.binary_task as binary_task
import sandbox.projects.common.tasklet.resource_formatter as formatter


class TaskletExecutor(binary_task.LastBinaryTaskRelease, sdk2.Task):
    '''
    Generic task to find sandbox resource with tasklet inside and execute it.
    '''
    class Requirements(sdk2.Task.Requirements):
        disk_space = 1024  # Mb
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(binary_task.LastBinaryReleaseParameters):
        with sdk2.parameters.Group('Lifetime parameters') as life_block:
            resource = sdk2.parameters.Resource('Tasklet resource id')
            resource_owner = sdk2.parameters.String('Tasklet resource owner')
            resource_type = sdk2.parameters.String('Tasklet resource type', default='SANDBOX_TASKS_BINARY')
            resource_attrs = sdk2.parameters.Dict('Tasklet resource attributes')
            tasklet_executable_name = sdk2.parameters.String('Tasklet executable name to run (for bundles)')
        tasklet_name = sdk2.parameters.String('Tasklet name to run')

        tasklet_input_as_json = sdk2.parameters.Bool("Tasklet input as JSON object", default=True)

        with tasklet_input_as_json.value[True]:
            tasklet_input = sdk2.parameters.JSON('Tasklet input parameters', required=True)
        with tasklet_input_as_json.value[False]:
            tasklet_input_str = sdk2.parameters.String('Tasklet input parameters (string with dumped json)', required=True)

        resources_info_name = sdk2.parameters.String('Output parameter array name with result resources info')
        ssh_user = sdk2.parameters.String('Ssh user to access ya vault')
        ssh_key_vault = sdk2.parameters.Vault(
            'Sandbox vault item with ssh key to access ya vault ("name" or "owner:name")')
        yav_token_vault = sdk2.parameters.Vault('Sandbox vault item with yav token')

        with sdk2.parameters.Output:
            tasklet_output = sdk2.parameters.JSON('Tasklet output')

    def run_tasklet(self, tasklet_executable_path, tasklet_name, tasklet_env_patch=None):
        with sdk2.helpers.ProcessLog(self, logger='tasklet') as pl:
            task_env = dict(os.environ)
            tasklet_env = task_env
            if tasklet_env_patch is not None:
                tasklet_env.update(tasklet_env_patch)
            tasklet_env['LOGS_DIR'] = str(self.log_path())
            pl.logger.propagate = 1
            tasklet_input = json.dumps(self.Parameters.tasklet_input) if self.Parameters.tasklet_input_as_json else self.Parameters.tasklet_input_str
            tasklet_cmd_list = [
                tasklet_executable_path,
                'run',
                tasklet_name,
                '--input', tasklet_input,
                '--local',
                '-v',
            ]

            tasklet_run = sp.Popen(
                tasklet_cmd_list,
                stdout=sp.PIPE,
                stderr=pl.stderr,
                env=tasklet_env)
            tasklet_run.wait()

            if tasklet_run.returncode != 0:
                raise common.errors.TaskFailure('Tasklet failed.')

            tasklet_out, _ = tasklet_run.communicate()
            pl.logger.info(tasklet_out)

            self.Parameters.tasklet_output = json.loads(tasklet_out)["output"]

            if self.Parameters.resources_info_name and self.Parameters.resources_info_name in self.Parameters.tasklet_output:
                resources = formatter.decode_resources(
                    self.Parameters.tasklet_output[self.Parameters.resources_info_name]
                )
                for resource_meta in resources:
                    resource_class = sdk2.Resource[resource_meta.resource_type]
                    sdk2.ResourceData(resource_class(
                        self, resource_meta.name, resource_meta.path, None, **resource_meta.attributes))
            elif self.Parameters.resources_info_name:
                logging.error(
                    "resources_info_name provided in tasklet input but its not found in tasklet output")

    def on_execute(self):
        if (
                self.Parameters.resource is None and
                not any((self.Parameters.resource_owner,
                         self.Parameters.resource_type, self.Parameters.resource_attrs))
        ):
            raise common.errors.TaskFailure(
                'Resource id or any of resource search parameters should be specified.')

        if self.Parameters.resource is not None:
            resource = sdk2.Resource.find(  # noqa
                id=self.Parameters.resource.id
            ).first()
        else:
            try:
                resource = sdk2.Resource.find(
                    type=self.Parameters.resource_type,
                    attrs=self.Parameters.resource_attrs,
                    owner=self.Parameters.resource_owner,
                    state=ctr.State.READY,
                ).limit(1).first()
            except IndexError:
                raise common.errors.TaskFailure(
                    'Couldn\'t find any resource with specified parameters.')

        logging.info('Found resource with id %s', resource.id)
        tasklet_executable_path = str(sdk2.ResourceData(resource).path)

        if self.Parameters.tasklet_executable_name:
            tasklet_executable_path = os.path.join(tasklet_executable_path,
                                                   str(self.Parameters.tasklet_executable_name))

        tasklet_name = self.Parameters.tasklet_name or resource.default_tasklet_name
        if not tasklet_name:
            raise common.errors.TaskFailure(
                'Tasklet name should be specified either explicitly via task params or ' +
                'via default_tasklet_name resource attribute.')

        tasklet_env = {}
        if self.Parameters.yav_token_vault:
            tasklet_env['GLYCINE_TOKEN'] = self.Parameters.yav_token_vault.data()

        if self.Parameters.ssh_key_vault:
            ssh_key = self.Parameters.ssh_key_vault.data()
            ssh_agent = sdk2.ssh.SshAgent()
            key_pub = ssh_agent.add(ssh_key)
            ssh_agent.print_keys()
            tasklet_env['USER'] = self.Parameters.ssh_user

        self.run_tasklet(tasklet_executable_path, tasklet_name, tasklet_env)

        if self.Parameters.ssh_key_vault:
            ssh_agent.remove(key_pub)
            ssh_agent.print_keys()
