import logging
import math
import os
import re
import shutil
import tempfile
import sys

import pathlib2
import six

from sandbox.projects.common.teamcity import ant
from sandbox.projects.common.teamcity import exit_stack
from sandbox import sdk2
from sandbox.sdk2.helpers import ProcessLog


class TeamcityServiceMessagesLog(sdk2.Resource):
    """
    Task log that should be printed to Teamcity.
    """
    DEFAULT_TTL = 3

    any_arch = True
    calc_md5 = False
    share = False
    ttl = DEFAULT_TTL
    log_name = sdk2.Attributes.String('Log name', required=False, default='')


class TeamcityArtifacts(sdk2.Resource):
    """
    Teamcity artifacts directory.
    """
    DEFAULT_TTL = 1

    any_arch = True
    auto_backup = False
    calc_md5 = False
    executable = False
    releasable = False
    share = False
    ttl = DEFAULT_TTL
    log_name = sdk2.Attributes.String('Log name', required=False, default='')


def regexp(string_re_pattern):
    """
    :type string_re_pattern: str
    """
    bytes_re_pattern = string_re_pattern.encode()
    return re.compile(bytes_re_pattern)


# See also: https://www.jetbrains.com/help/teamcity/service-messages.html#Escaped+Values
SERVICE_MESSAGE_ESCAPE_RULES = {
    "|'": "'",
    "||": "|",
    "|n": "\n",
    "|r": "\r",
    "|]": "]",
    "|[": "["
}


def get_escape_service_message_regexp():
    # Create regular expression parts for each dict key
    escaped_re_parts = [re.escape(escape_rule)
                        for escape_rule in six.iterkeys(SERVICE_MESSAGE_ESCAPE_RULES)]
    # Add OR part (pipe char)
    any_of_following = "|".join(escaped_re_parts)
    # With capture group
    replace_regexp_str = "({})".format(any_of_following)
    return re.compile(replace_regexp_str)


class _TeamcityResourcesManager(object):
    _trm_logger = logging.getLogger('TeamcityResourcesManager')

    PUBLISH_ARTIFACTS_SERVICE_MESSAGE_RE = regexp(
        r"^##teamcity\[ *publishArtifacts +'(?P<path>.*?)(?: => (?P<target>.*))?' *]\s*$")
    IMPORT_DATA_SERVICE_MESSAGE_RE = regexp(
        r"^##teamcity\[ *importData +path='(?P<path>.*?)'.* *]\s*$")

    BUILD_PARAMETER_SERVICE_MESSAGE_START_RE = regexp(r'^##teamcity\[ *setParameter.*]\s*$')
    BUILD_PARAMETER_SERVICE_MESSAGE_NAME_RE = regexp(r" +name='((?:[^'|]|\|.)+?)'")
    BUILD_PARAMETER_SERVICE_MESSAGE_VALUE_RE = regexp(r" +value='((?:[^'|]|\|.)+?)'")

    BUILD_PROBLEM_SERVICE_MESSAGE_START_RE = regexp(r'^##teamcity\[ *buildProblem.*]\s*$')
    BUILD_PROBLEM_SERVICE_MESSAGE_DESCRIPTION_RE = regexp(r" +description='((?:[^'|]|\|.)+?)'")
    BUILD_PROBLEM_SERVICE_MESSAGE_IDENTITY_RE = regexp(r" +identity='((?:[^'|]|\|.)+?)'")

    BUILD_STATISTIC_START_RE = regexp(r'^##teamcity\[ *buildStatisticValue.*]\s*$')
    BUILD_STATISTIC_VALUE_RE = regexp(r" +value='(\d+(\.\d+)*?)'")
    BUILD_STATISTIC_KEY_RE = regexp(r" +key='((?:[^'|]|\|.)+?)'")

    DEFAULT_TC_SERVICE_MESSAGES_DESCRIPTION = 'Teamcity log'
    DEFAULT_TC_ARTIFACTS_DESCRIPTION = 'Teamcity artifacts'

    SERVICE_MESSAGE_ESCAPE_REGEXP = get_escape_service_message_regexp()

    @property
    def _task(self):
        return sdk2.Task.current

    @property
    def _task_kill_timeout_days(self):
        kill_timeout = sdk2.Task.current.Parameters.kill_timeout
        return int(math.ceil(kill_timeout / (24 * 60 * 60)))

    @classmethod
    def _create_resources(cls, path_suffix, resources_dir=None):
        resources_dir = resources_dir or sdk2.Task.current.path('.teamcity')
        resources_dir.mkdir(parents=True, exist_ok=True)

        log_path = resources_dir.joinpath('teamcity{}.log'.format(path_suffix))
        cls._trm_logger.debug('Created log at %s', log_path)

        artifacts_dir = resources_dir.joinpath('artifacts{}'.format(path_suffix))
        artifacts_dir.mkdir()
        cls._trm_logger.debug('Created artifacts dir at %s', artifacts_dir)

        return log_path, artifacts_dir

    def __init__(self, base_dir=None, resources_dir=None, path_suffix=None, log_name=None,
                 tc_service_messages_ttl=TeamcityServiceMessagesLog.DEFAULT_TTL,
                 tc_artifacts_ttl=TeamcityArtifacts.DEFAULT_TTL,
                 tc_service_messages_description=DEFAULT_TC_SERVICE_MESSAGES_DESCRIPTION,
                 tc_artifacts_description=DEFAULT_TC_ARTIFACTS_DESCRIPTION):
        self._base_dir = (base_dir or pathlib2.Path.cwd()).absolute()

        self._fixed_path_suffix = '' if not path_suffix else '-{}'.format(path_suffix)
        self._log_path, self._artifacts_dir = self._create_resources(self._fixed_path_suffix, resources_dir)

        self._log_name = log_name
        self.log_resource = TeamcityServiceMessagesLog(
            self._task, tc_service_messages_description, self._log_path,
            ttl=max(tc_service_messages_ttl, self._task_kill_timeout_days),
            log_name=log_name, sync_upload_to_mds=False)
        # That allows to view resource in browser even if it is NOT_READY.
        self._task.server.resource[self.log_resource.id].source()

        self._tc_artifacts_ttl = max(tc_artifacts_ttl, self._task_kill_timeout_days)
        self._tc_artifacts_description = tc_artifacts_description

        self.fs_encoding = sys.getfilesystemencoding()
        self.service_messages_encoding = 'utf-8'

        self.build_statistics = {}
        self.build_parameters = {}
        self.build_problems = []

    def __enter__(self):
        return self

    @classmethod
    def _resolve_ant_pattern(cls, ant_pattern):
        """
        :param ant_pattern:
        :return: tuple of:
            - directory that includes all files matched by pattern
            - files matched by pattern
        :type ant_pattern: ant.Pattern
        :rtype: (pathlib2.Path, collections.Iterable[pathlib2.Path])
        """
        # If 'foo' is directory, 'foo' pattern does not match any files in 'foo', but Teamcity matches them anyway.
        # So we should treat 'foo' as 'foo/' here.
        if (ant_pattern.is_constant and
            os.path.isdir(str(ant_pattern)) and
            not ant_pattern.has_trailing_sep()):
            ant_pattern = ant.Pattern(str(ant_pattern) + os.sep)

        constant_part = pathlib2.Path(ant_pattern.constant_part())
        if constant_part.is_file():  # `ant_pattern` is (constant) path to existent file.
            base_dir = constant_part.parent
        else:
            base_dir = constant_part

        return base_dir, ant_pattern.glob()

    def _process_artifacts(self, ant_pattern):
        """
        :type ant_pattern: ant.Pattern
        :rtype: ant.Pattern
        """
        if not ant_pattern.is_absolute():
            ant_pattern = ant_pattern.prepend(self._base_dir)

        if not pathlib2.Path(ant_pattern.constant_part()).exists():
            self._trm_logger.warn('Path %s does not exist', ant_pattern)
            return None

        src_base_path, src_paths = self._resolve_ant_pattern(ant_pattern)
        dst_base_path = pathlib2.Path(tempfile.mktemp(dir=str(self._artifacts_dir)))

        for src_path in src_paths:
            dst_path = dst_base_path.joinpath(src_path.relative_to(src_base_path))
            assert src_path.is_file()

            self._trm_logger.debug('Copying %s to %s', src_path, dst_path)
            dst_path.parent.mkdir(parents=True, exist_ok=True)
            shutil.copy(str(src_path), str(dst_path))

        new_ant_pattern = ant_pattern.relative_to(src_base_path).prepend(dst_base_path)
        return new_ant_pattern.relative_to(self._artifacts_dir).prepend(self._artifacts_dir.name)

    def _match_build_parameter(self, line):
        """
        :type line: six.binary_type
        :rtype: NoneType or (str, str)
        """
        match = self.BUILD_PARAMETER_SERVICE_MESSAGE_START_RE.match(line)
        if not match:
            return None

        name_group = self.BUILD_PARAMETER_SERVICE_MESSAGE_NAME_RE.search(line)
        value_group = self.BUILD_PARAMETER_SERVICE_MESSAGE_VALUE_RE.search(line)
        if not all((name_group, value_group)):
            return None

        param_name = name_group.group(1).decode(self.service_messages_encoding)
        param_value = value_group.group(1).decode(self.service_messages_encoding)

        return self._unescape_service_message(param_name), self._unescape_service_message(param_value)

    def _match_build_problem(self, line):
        """
        :type line: six.binary_type
        :rtype: NoneType or (str, str)
        """
        match = self.BUILD_PROBLEM_SERVICE_MESSAGE_START_RE.match(line)
        if not match:
            return None

        description_group = self.BUILD_PROBLEM_SERVICE_MESSAGE_DESCRIPTION_RE.search(line)
        identity_group = self.BUILD_PROBLEM_SERVICE_MESSAGE_IDENTITY_RE.search(line)
        if not description_group:
            return None

        description = description_group.group(1).decode(self.service_messages_encoding)
        identity = ''  # optional parameter
        if identity_group:
            identity = identity_group.group(1).decode(self.service_messages_encoding)

        return self._unescape_service_message(description), self._unescape_service_message(identity)

    def _match_build_statistics(self, line):
        """
        :type line: six.binary_type
        :rtype: NoneType or (str, str)
        """
        match = self.BUILD_STATISTIC_START_RE.match(line)
        if not match:
            return None
        key_group = self.BUILD_STATISTIC_KEY_RE.search(line)
        value_group = self.BUILD_STATISTIC_VALUE_RE.search(line)
        if not key_group or not value_group:
            return None

        stat_name = key_group.group(1).decode(self.service_messages_encoding)
        stat_value = value_group.group(1).decode(self.service_messages_encoding)

        return self._unescape_service_message(stat_name), stat_value

    def _process_log(self, original_log, processed_log):
        """
        Parse Teamcity log, find all files that should be published and tests XMLs that should be reported, copy
        them to `self._artifacts_dir` and change paths in log.

        Also gather all reported build statistics to `self.build_statistics`.
        """
        for line in original_log:
            match = (self.PUBLISH_ARTIFACTS_SERVICE_MESSAGE_RE.match(line) or
                     self.IMPORT_DATA_SERVICE_MESSAGE_RE.match(line))
            if match:
                old_pattern = match.group('path')
                self._trm_logger.debug('Found pattern %s', old_pattern)

                old_pattern_as_string = old_pattern.decode(self.fs_encoding)
                new_pattern = self._process_artifacts(ant.Pattern(old_pattern_as_string))
                if new_pattern is None:
                    continue

                # Use universal separator because we don't know platform on agent
                # that will upload and publish artifacts.
                new_pattern = str(new_pattern).replace('\\', '/').encode(self.fs_encoding)

                self._trm_logger.debug('Replacing %s with %s', old_pattern, new_pattern)
                line = line.replace(old_pattern, new_pattern)

            match = self._match_build_statistics(line)
            if match:
                self.build_statistics[match[0]] = float(match[1])

            match = self._match_build_parameter(line)
            if match:
                self.build_parameters[match[0]] = match[1]

            match = self._match_build_problem(line)
            if match:
                # Do not use match variable here, due self.build_problems is public field
                # So tuple contract defined here
                # This allow us modify internal match logic inside self._match_build_problem
                problem_description, problem_identity = match
                self.build_problems.append((problem_description, problem_identity))

            processed_log.write(line)

    def _unescape_service_message(self, s):
        """
        :type s: str
        :rtype: str

        See also
        https://stackoverflow.com/questions/15175142/how-can-i-do-multiple-substitutions-using-regex
        """

        # For each match, look-up corresponding value in dictionary for replace
        def replace_callback(mo):
            return SERVICE_MESSAGE_ESCAPE_RULES[mo.group()]

        return self.SERVICE_MESSAGE_ESCAPE_REGEXP.sub(replace_callback, s)

    def __exit__(self, exc_type, exc_val, exc_tb):
        original_log = self._task.log_path('teamcity{}.unprocessed.log'.format(self._fixed_path_suffix))
        shutil.move(str(self._log_path), str(original_log))
        with original_log.open('rb') as input_file, self._log_path.open('wb') as output_file:
            self._process_log(input_file, output_file)
        sdk2.ResourceData(self.log_resource).ready()

        if list(self._artifacts_dir.iterdir()):
            artifacts_resource = TeamcityArtifacts(
                self._task, self._tc_artifacts_description, self._artifacts_dir,
                ttl=self._tc_artifacts_ttl, log_name=self._log_name)
            sdk2.ResourceData(artifacts_resource).ready()


class TeamcityArtifactsContext(object):
    """
    Context that opens output stream for log that should be printed to Teamcity. On exit, the log will be published
    as TeamcityServiceMessagesLog resource. All files from publishArtifacts and importData service messages
    will be gathered to separate directory and published as TeamcityArtifacts resource. Both of these resources
    can be processed by Teamcity-Sandbox integration plugin.

    Usage example:

    process_cwd = self.path('some_dir')
    with TeamcityArtifactsContext(script_cwd) as tac:
        subprocess.check_call(['some', 'process'], cwd=str(process_cwd), stdout=tac.output, stderr=subprocess.STDOUT)
    """

    class LogFormatter(object):
        def __init__(self, prepend_time, secret_tokens):
            self._service_messages_formatter = logging.Formatter()
            if prepend_time:
                self._default_formatter = logging.Formatter('[%(asctime)s] %(message)s', '%H:%M:%S')
            else:
                self._default_formatter = self._service_messages_formatter
            self._secret_tokens = secret_tokens

        @staticmethod
        def is_teamcity_service_message(line):
            return line.startswith('##teamcity')

        def format(self, record):
            for token in self._secret_tokens:
                record.msg = record.msg.replace(token, '******')
            if self.is_teamcity_service_message(record.getMessage()):
                return self._service_messages_formatter.format(record)
            else:
                return self._default_formatter.format(record)

    @classmethod
    def _prepare_logger(cls, logger, log_path, prepend_time, secret_tokens):
        logger.propagate = False

        # Use non-inheritable file handler.
        open_flags = os.O_WRONLY | os.O_TRUNC | os.O_CREAT
        if os.name == 'nt':
            open_flags |= os.O_NOINHERIT | os.O_BINARY
        file_handle = os.open(str(log_path), open_flags)

        handler = logging.StreamHandler(os.fdopen(file_handle, 'w'))
        handler.setFormatter(cls.LogFormatter(prepend_time, secret_tokens))
        logger.addHandler(handler)

        return logger

    def _teardown_logger(self):
        for handler in list(self.logger.handlers):
            self.logger.removeHandler(handler)
            handler.close()

    def __init__(self, base_dir=None, resources_dir=None, logger=None, prepend_time=True, secret_tokens=(),
                 path_suffix=None, log_name=None,
                 tc_service_messages_ttl=TeamcityServiceMessagesLog.DEFAULT_TTL,
                 tc_artifacts_ttl=TeamcityArtifacts.DEFAULT_TTL,
                 tc_service_messages_description=_TeamcityResourcesManager.DEFAULT_TC_SERVICE_MESSAGES_DESCRIPTION,
                 tc_artifacts_description=_TeamcityResourcesManager.DEFAULT_TC_ARTIFACTS_DESCRIPTION):
        """
        :param base_dir: base dir for all relative artifact paths
        :param resources_dir: path to base resources dir
        :param logger: logger for ProcessLog
        :param prepend_time: prepend time to log lines (except service messages)
        :param secret_tokens: tokens that will be masked in report with '******'
        :param path_suffix: additional suffix for paths.
        :param log_name: value of log name attribute.
        :param tc_service_messages_ttl: min value of service messages resource ttl attribute.
        :param tc_artifacts_ttl: min value of artifacts resource ttl attribute.
        :param tc_service_messages_description: description of resource containing service messages.
        :param tc_artifacts_description: description of resource containing artifacts.

        :type base_dir: pathlib2.Path or NoneType
        :type resources_dir: pathlib2.Path or NoneType
        :type logger: logging.Logger or NoneType
        :type prepend_time: bool
        :type secret_tokens: collections.Iterable
        :type path_suffix: str or NoneType
        :type log_name: str or NoneType
        :type tc_service_messages_ttl: int
        :type tc_artifacts_ttl: int
        :type tc_service_messages_description: str
        :type tc_artifacts_description: str
        """
        self._exit_stack = exit_stack.ExitStack()

        self._resource_manager = _TeamcityResourcesManager(
            base_dir, resources_dir, path_suffix, log_name,
            tc_service_messages_ttl, tc_artifacts_ttl,
            tc_service_messages_description, tc_artifacts_description)

        if logger is None:
            # Sometimes TeamcityArtifactsContext is used one inside another.
            logger = logging.getLogger('{} [{}]'.format(__name__, self._resource_manager.log_resource.id))
        self.logger = self._prepare_logger(
            logger, self._resource_manager.log_resource.path, prepend_time, secret_tokens)
        self._log = ProcessLog(logger=self.logger)

    @property
    def output(self):
        return self._log.stdout

    @property
    def build_statistics(self):
        return self._resource_manager.build_statistics

    @property
    def build_parameters(self):
        return self._resource_manager.build_parameters

    @property
    def build_problems(self):
        return self._resource_manager.build_problems

    def __enter__(self):
        self._exit_stack.enter_context(self._resource_manager)
        self._exit_stack.callback(self._teardown_logger)
        self._exit_stack.enter_context(self._log)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self._exit_stack.__exit__(exc_type, exc_val, exc_tb)
