import os
import pathlib2

try:
    # python3
    from builtins import str as unicode
except ImportError:
    pass


def _is_constant(glob_pattern):
    return '?' not in glob_pattern and '*' not in glob_pattern


def _split_pattern(glob_pattern):
    """
    :param glob_pattern:
    :type glob_pattern: basestring
    :return: tuple of constant and glob (variable) part of pattern
    :rtype: (basestring, basestring)
    """
    constant_part = glob_pattern
    while not _is_constant(constant_part):
        constant_part = os.path.dirname(constant_part)
    if glob_pattern == constant_part:
        glob_part = ''
    else:
        glob_part = os.path.relpath(glob_pattern, constant_part)
    return constant_part, glob_part


def _glob(glob_pattern):
    """
    Find files by glob pattern.
    Recursive globbing syntax '**' is supported, like in `glob.glob` from Python 3 (but not Python 2).

    :param glob_pattern:
    :type glob_pattern: basestring
    :return: files matched by pattern
    :rtype: collections.Iterable[pathlib2.Path]
    """
    constant_part, glob_part = _split_pattern(glob_pattern)
    constant_path = pathlib2.Path(constant_part)
    if glob_part:
        return constant_path.glob(glob_part)
    else:
        return [constant_path] if constant_path.exists() else []


class Pattern(object):
    """
    Ant path pattern.
    Documentation: http://ant.apache.org/manual/dirtasks.html#patterns

    WARNING: non-ascii paths on Windows are not supported due to restrictions of pathlib2.
    """
    def __init__(self, pattern):
        """
        :type pattern: basestring
        """
        self._pattern = pattern

    def __str__(self):
        return str(self._pattern)

    def __unicode__(self):
        return unicode(self._pattern)

    def has_trailing_sep(self):
        valid_seps = (os.sep,) + ((os.altsep,) if os.altsep else ())
        return self._pattern.endswith(valid_seps)

    def is_absolute(self):
        return os.path.isabs(self._pattern)

    def is_constant(self):
        return _is_constant(self._to_glob_pattern())

    def prepend(self, path):
        """
        :type path: pathlib2.Path | basestring
        :rtype: ant.Pattern
        """
        if isinstance(path, pathlib2.Path):
            path = str(path)
        new_pattern = os.path.join(path, self._pattern)
        return self.__class__(new_pattern)

    def relative_to(self, path):
        """
        :type path: pathlib2.Path | basestring
        :rtype: ant.Pattern
        """
        if isinstance(path, pathlib2.Path):
            path = str(path)
        new_pattern = os.path.relpath(self._pattern, path)
        # `os.path.relpath` does not respect trailing slashes. Add it if necessary.
        if self.has_trailing_sep():
            new_pattern += os.sep
        return self.__class__(new_pattern)

    def _to_glob_pattern(self):
        glob_pattern = self._pattern

        # 'foo/' means 'foo/**' in Ant.
        if self.has_trailing_sep():
            glob_pattern += '**'

        # In standard glob, '**' means zero or more directories.
        # In Ant, '**' means zero or more directories or files.
        if glob_pattern.endswith('**'):
            glob_pattern += os.sep + '*'

        return glob_pattern

    def glob(self):
        """
        :rtype: collections.Iterable[pathlib2.Path]
        """
        return (
            path for path in _glob(self._to_glob_pattern())
            if path.is_file()  # Ant patterns match files only.
        )

    def constant_part(self):
        """
        :rtype: basestring
        """
        return _split_pattern(self._to_glob_pattern())[0]
