import contextlib
import sys

import six


class _BackportedExitStack(object):
    def __init__(self):
        self._exit_stack = []

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, traceback):
        self._close(exc_type, exc_val, traceback)

    def push(self, exit):
        self._exit_stack.append(exit)

    def enter_context(self, cm):
        value = cm.__enter__()
        self.push(cm.__exit__)
        return value

    def callback(self, callback, *args, **kwargs):
        self.push(lambda exc_type, exc_val, traceback: callback(*args, **kwargs))
        return callback

    def pop_all(self):
        new_stack = self.__class__()
        new_stack._exit_stack = self._exit_stack
        self._exit_stack = []
        return new_stack

    def close(self):
        self._close(None, None, None)

    def _close(self, exc_type, exc_val, traceback):
        pending_raise = False
        for exit_function in reversed(self._exit_stack):
            try:
                exit_function(exc_type, exc_val, traceback)
            except:
                exc_type, exc_val, traceback = sys.exc_info()
                pending_raise = True
        self._exit_stack = []
        if pending_raise:
            six.reraise(exc_type, exc_val, traceback)


ExitStack = getattr(contextlib, 'ExitStack', _BackportedExitStack)
