"""
    Common code for convenient TG notifications
    Maintainers: mvel@, glebov-da@, ilyaturuntaev@ (Release Machine team)
"""
import logging
import traceback

from sandbox import sdk2
from sandbox.common import telegram
from sandbox.projects.common import string
from sandbox.projects.common.telegram_api import chats

logger = logging.getLogger(__name__)


def create_bot(task, token_name, token_owner):
    logger.debug("Try to create telegram bot with token_name=%s and token_owner=%s", token_name, token_owner)
    try:
        bot_token = sdk2.Vault.data(token_owner, token_name)
    except Exception:
        task.set_info(
            "Unable to get telegram token: '{token}'. Probably, task owner '{owner}' have no rights.\n"
            "Check access here: https://sandbox.yandex-team.ru/admin/vault "
            "You must have access for '{token}'".format(
                token=token_name, owner=task.owner,
            )
        )
        raise
    return telegram.TelegramBot(bot_token)


def notify_chats(task, bot, chat_ids, message, pin=False, **kwargs):
    if str(task.id) not in message:
        # append sandbox task id to message if it is not mentioned in the message
        # (for debug/incident management purposes)
        message = '{}\nSent from SB:{}'.format(message, task.id)

    logger.debug("Sending telegram message to %s:\n%s", chat_ids, string.shift_right(message))
    for chat_id in chat_ids:
        try:
            answer = bot.send_message(chat_id, message, parse_mode=telegram.ParseMode.HTML, **kwargs)
            logger.debug("Telegram: bot.send_message invoked for chat id %s", chat_id)
            if pin:
                pin_response_json = bot.pin_message(chat_id, answer["result"]["message_id"])
                if not pin_response_json["ok"]:
                    error_text = "Cannot pin message: error code {}, description: {}".format(
                        pin_response_json["error_code"],
                        pin_response_json["description"],
                    )
                    logger.error(error_text)
                    task.set_info(error_text)

        except Exception:
            logger.error("Unable to send telegram message to '%s':\n%s", chat_id, traceback.format_exc())


def notify_people(task, bot, people, message, pin=False, **kwargs):
    chat_ids = [chats.TELEGRAM_CHATS[p] for p in people if p in chats.TELEGRAM_CHATS]
    not_registered = [p for p in people if p not in chats.TELEGRAM_CHATS]
    logger.debug(
        "Unable to send message to {}, because we don't know their telegram ids. "
        "Please add them to 'TELEGRAM_CHATS' list in "
        "sandbox/projects/common/telegram_api/chats.py".format(not_registered)
    )
    notify_chats(task, bot, chat_ids, message, pin, **kwargs)
