# -*- coding: utf-8 -*-

import os
import re
import subprocess

from sandbox import sandboxsdk
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
from sandbox.projects.common import utils


_SHARD_NAME_RE = re.compile("([a-z]+)-(\d+)-(\d+-\d+)")

IMPROXY_PRIEMKA = ('improxy_priemka', '1')


def thdb_path(database_dir):
    """
        <путь к thdb без суффикса>
    """
    if not os.path.isfile(os.path.join(database_dir, 'thdb.info')):
        raise SandboxTaskFailureError('info file not found in database, cannot compute thdb prefix')
    return os.path.join(database_dir, 'thdb')


def get_thumb_ids(database_resource_id):
    path = thdb_path(channel.task.sync_resource(database_resource_id))

    def exec_path(resource_type):
        exec_resource = utils.get_and_check_last_resource_with_attribute(
            resource_type, *IMPROXY_PRIEMKA, arch=sandboxsdk.util.system_info()['arch']
        ).id
        return channel.task.sync_resource(exec_resource)

    thdb_exec_path = exec_path(resource_types.THDB_EXECUTABLE)
    # 0 and 38 are chunks with thumbnail and nd5, only dump the records where both are present
    # md5 only may be present in deleted records, resulting in 500 when asked from thumbdaemon
    # TODO: this should be fixed on index side some time in the future
    dump_process = run_process([thdb_exec_path, 'dump', '-f', '%m', '-l', '0,38', '-s', '-T', path],
                               wait=False, stdout=subprocess.PIPE)
    return (line.strip() for line in dump_process.stdout)


def get_database_version(database_resource_id):
    database_path = channel.task.sync_resource(database_resource_id)
    version_file_path = os.path.join(database_path, 'shard.conf')

    if not os.path.exists(version_file_path):
        raise SandboxTaskFailureError("Version file %s not found" % version_file_path)

    with open(version_file_path, 'r') as version_file:
        for line in version_file:
            if '%shard' in line:
                shard_name = line.split()[1].strip()
                match = _SHARD_NAME_RE.match(shard_name)
                if match:
                    return match.group(1), match.group(2), match.group(3)
    raise SandboxTaskFailureError("Shard version not found in file %s" % version_file_path)
