import datetime as dt
import dateutil.tz as tz
import dateutil.parser as dt_parser


ZERO_TIME_UTC = dt.datetime(1970, 1, 1, tzinfo=tz.tzutc())


def date_ymd(sep="-", formatter=("%Y", "%m", "%d"), delay_days=None):  # type: (str, tuple, int) -> str
    target_time = dt.datetime.now()
    if delay_days is not None:
        target_time = target_time - dt.timedelta(days=delay_days)
    return target_time.strftime(sep.join(formatter))


def date_ymdhm(sep=" "):  # type: (str) -> str
    formatter = sep.join(("%Y-%m-%d", "%H:%M"))
    return dt.datetime.now().strftime(formatter)


def datetime_utc():  # type: () -> dt.datetime
    """Return current datetime in UT0"""
    return dt.datetime.utcnow().replace(tzinfo=tz.tzutc())


def datetime_utc_iso(force_z=False):  # type: (bool) -> str
    """Return current datetime in UT0 in ISO format."""
    return datetime_to_iso(datetime_utc(), force_z=force_z)


def datetime_to_iso(dt_obj, force_z=False):  # type: (dt.datetime, bool) -> str
    """
    Convert `dt.datetime` instance to ISO datetime string. :param force_z: allows to force 'Z' instead of '+00:00'
    for UTC datetime objects.

    :param dt_obj: `dt.datetime` instance
    :param force_z: Always replace '+00:00' with 'Z'
    :return: ISO datetime string (with 'Z' for timezone if :param force_z: flag is set)
    """

    result = dt_obj.isoformat()

    if force_z:
        result = result.replace("+00:00", "Z")

    return result


def datetime_to_timestamp(dt_obj):  # type: (dt.datetime) -> float
    """
    Convert datetime object to timestamp.

    If datetime object does not have timezone we set UTC timezone
    :param dt_obj: `dt.datetime` object
    :return: timestamp (`float`)
    """
    if not dt_obj.tzinfo:
        dt_obj = dt_obj.replace(tzinfo=tz.tzutc())
    return (dt_obj - ZERO_TIME_UTC).total_seconds()


def to_unixtime(datetime_str):  # type: (str) -> int
    """
    Convert time string to int timestamp using provided string template.

    :param datetime_str: String with time in any format
    :return: timestamp
    """
    dt_obj = dt_parser.parse(datetime_str)
    return int(datetime_to_timestamp(dt_obj))
