# -*- coding: utf-8 -*-

import os
import collections

from sandbox import sdk2
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import environments


class CodeInfoSvn(collections.namedtuple("CodeInfo", ["path", "branch", "rev", "url"])):
    __slots__ = ()

    @property
    def version(self):
        return "{}_r{}".format(self.branch, self.rev)


class CodeInfoGit(CodeInfoSvn):
    @property
    def version(self):
        return "{}_g{}".format(self.branch, self.rev)


def checkout(srcurl, srcdir):
    cmd = ["git", "clone", srcurl[0], srcdir]
    if len(srcurl) > 1:
        cmd[2:0] = ["-b", srcurl[1]]

    process.run_process(cmd, log_prefix="ui_build_fetch_sources")
    branch = process.run_process(
        ["git", "rev-parse", "--abbrev-ref",  "HEAD"],
        work_dir=srcdir,
        outs_to_pipe=True
    ).stdout.read().strip()
    rev = process.run_process(
        ["git", "rev-parse", "--short",  "HEAD"],
        work_dir=srcdir,
        outs_to_pipe=True
    ).stdout.read().strip()
    return CodeInfoGit(srcdir, branch, rev, srcurl)


def build(code_info):
    environments.NodeJS("8.12.0-vanilla").prepare()
    process.run_process(["make", "-C", code_info.path], log_prefix="ui_build_make")

    out_folder = ""
    # Sandbox build output lives in 'dist' folder and CI lives in 'build'
    for d in ("dist", "build"):
        path = os.path.join(code_info.path, d)
        out_folder = path if os.path.exists(path) else out_folder
        with open(os.path.join(out_folder, ".revision"), "w") as f:
            f.write(str(code_info.rev))

    return code_info.__class__(out_folder, *code_info[1:])


def build_with_robot_auth(task, code_info):
    with sdk2.ssh.Key(task, "SANDBOX", "sandbox_robot_ssh"):
        os.environ["LOGIN"] = sdk2.Vault.data("SANDBOX", "sandbox_robot_login")
        os.environ["PASSWORD"] = sdk2.Vault.data("SANDBOX", "sandbox_robot_password")
        return build(code_info)
