#!/usr/bin/env python
import cgi
import re


class HtmlFormatter(object):
    def __init__(self, stream):
        self.stack = []
        self.stream = stream
        self.stream.write("<html>")
        self.enter("head")

    def enter(self, name, style=None, attrs={}):
        self.stack.append(name)
        data = "<" + name + " "
        if style:
            attrs["style"] = style
        for key, value in attrs.iteritems():
            data += key + "=\"" + cgi.escape(value, True) + "\" "
        data += ">\n"
        self.stream.write(data)

    def exit(self):
        self.stream.write("</" + self.stack.pop() + ">\n")

    def escape(self, s):
        return cgi.escape(s, True).replace("\n", "<br/>")

    def tag(self, name, style=None, attrs={}, content=""):
        self.enter(name, style, attrs)
        self.stream.write(content + "\n")
        self.exit()

    def finish(self):
        while self.stack:
            self.exit()


def replace_log_links(log_dir, task_dir, content):
    if not log_dir.endswith("/"):
        log_dir += "/"
    r = re.escape(log_dir)
    prev_pos = 0
    res = ""
    for match in re.finditer(r + r"[^ ;]*\.(?:err|out)", content):
        s = match.group(0)
        if prev_pos < match.start():
            res += content[prev_pos:match.start()]
            res += "<a href='{}' target='_blank'>{}</a>".format(
                s[len(log_dir):],
                "STDOUT" if s.endswith(".out") else "STDERR"
            )
        prev_pos = match.end()
    res += content[prev_pos:]
    if task_dir:
        res = res.replace(task_dir, "/")
    return res


def format_rem_tool_packets_status(src, dst, log_dir=None, task_dir=None):
    def print_color(html, color, msg):
        html.tag(
            "div",
            style="color: {}; background-color: {}; font-family: monospace".format(*color.split(",")),
            content=msg
        )
    empty = True

    with open(src) as inp, open(dst, "w") as out:
        html = HtmlFormatter(out)
        buf = []
        last_color = None

        for s in inp:
            s = s.rstrip()
            packet_start = s.startswith("packet")
            if packet_start:
                in_packet = "state=ERROR" in s

            if not in_packet:
                continue

            if s.startswith("[SUBJOB") or packet_start:
                if buf:
                    empty = False
                    buf.append("-" * 80)
                    buf.insert(0, "-" * 80)
                    data = html.escape("\n".join(buf))
                    if log_dir:
                        data = replace_log_links(log_dir, task_dir, data)
                    print_color(html, last_color, data)
                    buf = []
                if packet_start:
                    last_color = "black,white"

            if s.startswith("done"):
                last_color = "black,palegreen"
            elif s.startswith("errored"):
                last_color = "black,lightcoral"
            elif s.startswith("suspended") or s.startswith("pending"):
                last_color = "gray,lightgray"
            buf.append(s)

        if buf:
            empty = False
            buf.append("-" * 80)
            buf.insert(0, "-" * 80)
            data = html.escape("\n".join(buf))
            if log_dir:
                data = replace_log_links(log_dir, task_dir, data)
            print_color(html, last_color, data)
            buf = []
    return not empty


if __name__ == "__main__":
    import sys
    format_rem_tool_packets_status(sys.argv[1], sys.argv[2], sys.argv[3])
