import os
import logging

import sandbox.sandboxsdk.util as sdk_util

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_unique_file_name, make_folder, remove_path

from sandbox.projects.common.ProcessPool import ProcessPool


class PackagesInstaller(object):
    def __init__(self, require_packages=True, debian=True):
        self.require_packages = require_packages
        self.debian = debian

    def install(self, root, task, package_parameters):
        for pclass in package_parameters:
            if not task.ctx.get(pclass.name, None):
                msg = "Package for {} not supplied, will not 'fake debinstall'".format(pclass.name)
                if self.require_packages:
                    raise SandboxTaskFailureError(msg)
                logging.info(msg)
                return False

        debtmp = get_unique_file_name(task.abs_path(''), 'debtmp')
        if self.debian:
            make_folder(debtmp)

        nproc = max(1, sdk_util.system_info()['ncpu'] / 2)
        ppool = ProcessPool(nproc)

        untar = []
        for pclass in package_parameters:
            resource_id = task.ctx[pclass.name]
            resource = channel.sandbox.get_resource(resource_id)
            tgz = task.sync_resource(resource_id)
            task.ctx['_' + resource.type + '_VER'] = self.get_resource_version(resource)
            task.ctx['_' + resource.type + '_SVN_PATH'] = self.get_resource_svn_path(resource)
            task.ctx.setdefault("packages_list", []).append(resource.type)
            untar.append('tar zxf %s -C %s --no-overwrite-dir' % (tgz, debtmp if self.debian else root))

        ppool.map(lambda cmd: run_process(cmd, shell=True, check=True, log_prefix='untar'), untar)

        if self.debian:
            debs = [os.path.join(debtmp, f) for f in os.listdir(debtmp) if f.endswith('.deb') and '-dbg' not in f]
            for debf in debs:
                run_process('dpkg --extract %s %s' % (debf, root), shell=True, check=True, log_prefix='dpkg')
            remove_path(debtmp)

        return True

    def get_resource_svn_path(self, resource):
        return resource.attributes.get("svn_path")

    def get_resource_version(self, resource):
        if "svn_revision" in resource.attributes:
            return resource.attributes["svn_revision"]
        result = -1
        bestkey = None
        for key, value in resource.attributes:
            if key.endswith("_version"):
                if bestkey is None or len(key) < len(bestkey):
                    result = value
                    bestkey = key
        if isinstance(result, str) and result.isdigit():
            result = int(result)
        return result
