import os
import time
import logging

import sandbox.sandboxsdk.util as sdk_util
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.userdata import formatters


class RemClient(object):
    def __init__(self, rem_url, rem_tool):
        self.rem_url = rem_url
        self.rem_tool = rem_tool

    def connector(self):
        import remclient
        return remclient.Connector(self.rem_url, conn_retries=2, verbose=True)

    def set_optimal_working_limit(self, queues=None, max_parallel=None):
        c = self.connector()
        new_limit = max(1, sdk_util.system_info()['ncpu'] / 2)
        if max_parallel:
            new_limit = min(max_parallel, new_limit)
        for q in queues or self.list_queues():
            logging.info("Setting working limit {} for queue '{}'".format(new_limit, q))
            c.Queue(q).ChangeWorkingLimit(new_limit)

    def list_queues(self):
        return dict(self.connector().ListObjects("queues"))

    def wait_all_packets(self, poll_interval=30, max_parallel=None, callback=None):
        conn = self.connector()
        known_queues = set()
        prev_msg = ""
        waited_intervals = 0
        while True:
            sumwait = 0
            msg = ""
            for qn in sorted(self.list_queues().keys()):
                if qn not in known_queues:
                    self.set_optimal_working_limit([qn], max_parallel=max_parallel)
                    known_queues.add(qn)
                q = conn.Queue(qn)
                status = q.Status()
                if not status or status.get('errored', 0) > 0:
                    raise SandboxTaskFailureError("some tasks failed in '" + qn + "': " + repr(status))

                sumwait += sum([status.get(st, 0) for st in ['working', 'waiting', 'suspended', 'pending']])
                msg += "'" + qn + "' -- " + repr(status) + "; "
            if prev_msg != msg:
                logging.info("Still waiting for packets: " + msg)
                prev_msg = msg
            else:
                logging.info("Nothing changed")

            if sumwait == 0:
                return
            time.sleep(poll_interval)
            waited_intervals += 1
            if waited_intervals % 20 == 0 and callback:
                callback()

    def with_rem_tool(self, cmd, *args, **kwargs):
        prefix = self.rem_tool + " -u " + self.rem_url + " "
        kwargs["shell"] = True
        return run_process(prefix + cmd, *args, **kwargs)

    def dump_history(self, log_dir, task_dir):
        result = []
        self.with_rem_tool(
            "--tags list", check=False, wait=True,
            log_prefix="rem.tags"
        )
        for queue in self.list_queues():
            self.with_rem_tool(
                '-q {} status'.format(queue), check=False, wait=True,
                log_prefix="rem.status." + queue
            )
            self.with_rem_tool(
                '-q {} list'.format(queue), check=False, wait=True,
                log_prefix='rem.list.' + queue
            )
            status_name = os.path.join(log_dir, "rem.detailed." + queue + ".out.txt")
            with open(status_name, "w") as out:
                run_process(
                    "{rem_tool} -u {rem_url} -q {queue} list | grep 'packet' | cut -d \\' -f 2 | "
                    "while read i; do {rem_tool} -u {rem_url} -p $i status; done".format(
                        queue=queue,
                        rem_tool=self.rem_tool,
                        rem_url=self.rem_url
                    ),
                    shell=True, check=False, wait=True,
                    stdout=out
                )
            filtered_short_name = "rem.filtered." + queue + ".out.html"

            filtered_name = os.path.join(log_dir, filtered_short_name)
            if formatters.format_rem_tool_packets_status(status_name, filtered_name, log_dir, task_dir):
                result.append(filtered_short_name)
        return result
