import os
import logging
from datetime import datetime, timedelta
import calendar
import collections
import shutil
import time

import sandbox.sandboxsdk.util as sdk_util
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_unique_file_name
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.ProcessPool import ProcessPool


def get_or_create_resource(task, id, **kwargs):
    if id:
        resource = channel.sandbox.get_resource(id)
        if resource:
            assert not resource.is_ready()
            if "resource_path" in kwargs:
                task.change_resource_basename(id, kwargs["resource_path"])
            return resource
    return task.create_resource(**kwargs)


class ConfigPatcher(object):
    def __init__(self, logs_dir):
        self.data = {}
        self.logs_dir = logs_dir

    def add(self, **kwargs):
        self.data.update(**kwargs)

    def patch(self, cfg, patch_dict, custom_line_processor=None):
        d = dict(self.data)
        d.update(patch_dict)
        log_cfg = get_unique_file_name(self.logs_dir, os.path.basename(cfg))
        with open(cfg) as src, open(log_cfg, 'w') as dst:
            for indented_line in src:
                indent_len = len(indented_line) - len(indented_line.lstrip())
                if "=" in indented_line:
                    parts = indented_line.split("=", 1)
                    key = parts[0].strip()
                    value = None
                    set_value = False
                    if key in d:
                        value = repr(d[key])
                        set_value = True
                    if key + "@" in d:
                        value = d[key + "@"]
                        set_value = True
                    if set_value:
                        indented_line = indented_line[:indent_len] + key + " = " + str(value)
                        indented_line += " # were -- " + parts[1].strip()
                if custom_line_processor:
                    new_line = indented_line[:indent_len] + custom_line_processor(indented_line.lstrip())
                    if new_line != indented_line:
                        new_line += " # were -- " + indented_line.lstrip()
                        indented_line = new_line

                dst.write(indented_line.rstrip() + "\n")
        shutil.copyfile(log_cfg, cfg)


class ProcessRunner(object):
    def __init__(self, pool_size=None):
        if pool_size is None:
            pool_size = max(1, sdk_util.system_info()["ncpu"] / 2)
        logging.info("Using {} processes for pool".format(pool_size))
        self.pool = ProcessPool(pool_size)
        self.cmds = []

    def run(self):
        for tag, cmd in self.cmds:
            logging.debug("Going to run {}: {}".format(tag, cmd))

        self.pool.map(
            lambda (tag, cmd): run_process(cmd, shell=True, wait=True, check=True, log_prefix=tag),
            self.cmds
        )

    def add(self, tag, cmd, *args, **kwargs):
        if args or kwargs:
            cmd = cmd.format(*args, **kwargs)
        self.cmds.append([tag, cmd])


class Poller(object):
    def __init__(self):
        self._pollers = {}
        self._intervals = {}
        self._last_run = {}

    def add(self, name, fn, interval=30):
        if fn is not None:
            self._pollers[name] = fn
            self._intervals[name] = interval
            self._last_run[name] = 0

    def run_once(self):
        now = time.time()
        for name in self._pollers:
            if now - self._last_run[name] < self._intervals[name]:
                continue
            self._last_run[name] = now
            if self._pollers[name]():
                logging.info("Poller {} returned True".format(name))
                return True
        return False

    def run_infinite(self):
        while True:
            if self.run_once():
                return True
            time.sleep(1)


def smart_join_params(base_params, *task_params, **kwargs):
    by_group = collections.defaultdict(list)
    for param in task_params:
        by_group[param.group].append(param)
    for param in base_params:
        by_group[param.group].append(param)
    res = []
    if "binaries_provider" in kwargs:
        res = kwargs["binaries_provider"].task_parameters()
    for group in sorted(by_group):
        if group is not None:
            res += by_group[group]
    return list(res + by_group.get(None, []))


def run_shell_process(tag, cmd, *args, **kwargs):
    cmd = cmd.format(*args, **kwargs)
    logging.debug("Going to run {}: {}".format(tag, cmd))
    run_process(cmd, shell=True, wait=True, check=True, log_prefix=tag)


def lazy_property(fn):
    attr_name = '_lazy_' + fn.__name__

    @property
    def _lazyprop(self):
        if not hasattr(self, attr_name):
            setattr(self, attr_name, fn(self))
        return getattr(self, attr_name)
    return _lazyprop


def str2date(s):
    return datetime.strptime(s, "%Y%m%d").date()


def date2timestamp(d):
    return int(calendar.timegm(d.timetuple()))


def date2str(d):
    return d.strftime("%Y%m%d")


def date2str_yt(d):
    return d.strftime("%Y-%m-%d")


def get_first_date(s, delta):
    return date2str(str2date(s) - timedelta(delta - 1))


def get_last_date(s, delta):
    return date2str(str2date(s) + timedelta(delta - 1))
