# coding: utf-8

"""
    Standard Sandbox Python Library.
    Misc code used in various tasks.

    Please obey PEP8 coding standards and Sandbox StyleGuide here
    https://wiki.yandex-team.ru/sandbox/codestyleguide

    New functions without documentation comments are FORBIDDEN.

    SSPL maintainers: mvel@
"""

import datetime
import errno
import getpass
import logging
import os
import re
import socket
import time
import six
from types import GeneratorType

from sandbox import common
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox.common.types.task import Status

from sandbox import sdk2
import sandbox.sdk2.legacy

from sandbox.sandboxsdk import channel
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process

from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.common import constants
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import file_utils as fu
from sandbox.projects.common import utils2
from sandbox.projects.common import templates
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import requests_wrapper
from sandbox.projects.common.sdk_compat import task_helper

NO_DIFF_NAME = "no_diff_detected.html"
RESTART_CHILD_KEY = "subtasks_were_restarted"

if six.PY34:
    import html
    cgi_escape = html.escape
else:
    import cgi
    cgi_escape = cgi.escape


def no_diff_detected():
    return templates.get_html_template(NO_DIFF_NAME)


################################################################################
#                Создание параметров для выбора типа ресурса


def _create_arch():
    class Arch(sandbox.sdk2.legacy.SandboxStringParameter):
        name = 'resource_arch'
        description = 'Uploaded resource arch'
        choices = [(x, x) for x in ctm.OSFamily]
        default_value = 'any'

    return Arch


def _create_res_select_param(resources_list, fields):
    class Resource(sandbox.sdk2.legacy.SandboxStringParameter):
        name = 'resource_type'
        description = 'Resource type'
        choices = resources_list
        sub_fields = fields
        required = True
        suggest = True

    return Resource


def get_resource_type_params():
    """
        Используется в REMOTE_COPY_RESOURCE и в CREATE_TEXT_RESOURCE
        :return: выбор ресурсов из списка с подполями выбора архитектуры
    """
    arch_choice = _create_arch()
    rt_list = list(resource_types.AbstractResource)
    choices = sorted([(rt.name, rt.name) for rt in rt_list])
    sub_fields = {rt.name: [arch_choice.name] for rt in rt_list if not rt.any_arch}
    resource = _create_res_select_param(choices, sub_fields)

    return [resource, arch_choice]


################################################################################
#                            Операции со списками


def count_avg(array, precision=None):
    if array:
        avg_array = float(sum(array)) / len(array)
        return round(avg_array, precision) if precision is not None else avg_array
    else:
        return None


def count_std_dev(array, precision=None):
    if array:
        avg_array = count_avg(array)
        std_dev = (float(sum((avg_array - val) ** 2 for val in array) / len(array))) ** .5
        return round(std_dev, precision) if precision is not None else std_dev
    else:
        return None


def get_or_default(ctx, param_class):
    return ctx.get(param_class.name, param_class.default_value)


def copy_param_values(ctx_from, cls_from, ctx_to, cls_to):
    """
        Для удобства копирования контекста задачи в дочернюю
    """
    for i in six.moves.xrange(len(cls_from.params)):
        ctx_to[cls_to.params[i].name] = get_or_default(ctx_from, cls_from.params[i])


def check_if_tasks_are_ok(ids, ignore_deleted=False, fail_on_fail=True):
    """
        Проверить, завершился ли таск без ошибок. Позволяет игнорировать удаленные таски.

        !!! DEPRECATED !!!
        Use :func:`check_subtasks_fails` instead if possible.
    """
    for task_id in ids:
        task = channel.channel.sandbox.get_task(task_id)

        if ignore_deleted and task.is_deleted():
            continue

        if not task.is_failure() and fail_on_fail:
            if task.new_status not in [
                Status.SUCCESS,
                Status.RELEASING,
                Status.RELEASED,
                Status.STOPPING,
                Status.STOPPED,
            ]:
                channel.channel.task.ctx.setdefault('failed_childs', []).append(task_id)
                eh.fail("child task {} ({}) is not finished correctly".format(task.status, task_id))
        else:
            eh.check_failed("child task {} ({}) is failed".format(task.status, task_id))


def check_all_subtasks_done():
    """
        :return: True, если все дочерние задачи пришли
            в конечное состояние (невозможно перезапустить).
    """
    # Do not count DRAFTs here
    # See https://ml.yandex-team.ru/thread/2370000002596347627/
    return all([
        subtask_info.get("status") in Status.Group.DRAFT or subtask_info.get("status") in Status.Group.FINISH
        for subtask_info in subtasks_id_and_status(channel.channel.task.id)
    ])


def amount_of_working_subtasks():
    """
        :return: Количество выполняющихся задач
    """
    return len(get_working_subtasks())


def get_working_subtasks():
    working_statuses = Status.Group.QUEUE + Status.Group.EXECUTE + Status.Group.WAIT
    working_subtasks = []
    for task in list_subtasks_any_sdk(load=True):
        if isinstance(task, sdk2.Task):
            # Sdk 2 tasks
            status = task.status
        else:
            # Sdk1 tasks
            try:
                status = task.new_status
            except Exception:
                logging.warning("Error occurred, try to get old status:\n %s", eh.shifted_traceback())
                status = task.status
        if status in working_statuses:
            working_subtasks.append(task)
    return working_subtasks


def check_subtasks_fails(
    stop_on_broken_children=False,
    fail_on_failed_children=True,
    custom_subtasks=None,
    fail_on_first_failure=False,
    stop_children_on_first_failure=False,
):
    """
        Меняет статус родительской задачи в зависимости от состояния ее детей.

        @param stop_on_broken_children: Обрабатывает состояние, когда
            в дочерних задачах нет задач в статусе FAILED,
            но есть хотя бы одна в состоянии BREAK
            (EXCEPTION, NO_RES, TIMEOUT, STOPPING, STOPPED).
            Если True - останавливать родительскую задачу (STOPPED)
            Если False - кидать исключение EXCEPTION.
        @param fail_on_failed_children: Падать в FAILURE, если есть хотя бы одна дочерняя задача в FAILURE.
            Иначе завершаться корректно, но с дополнительным полем в контексте "task_has_failed_children" = True
        @param custom_subtasks: Список задач, состояние которых необходимо проверять.
            Если None - проверяются все дочерние задачи.
        @param fail_on_first_failure: падать, не дожидаясь остальных подзадач, если хотя бы одна из подзадач упала
        @param stop_children_on_first_failure: остановить все незавершенные дочерние задачи при падении
            хотя бы одной из них
    """
    if not fail_on_first_failure and amount_of_working_subtasks():
        wait_all_subtasks_stop()

    myself = channel.channel.task
    subtask_list = list_subtasks_any_sdk(load=True)
    if custom_subtasks:
        subtask_list = [task for task in subtask_list if task.id in custom_subtasks]
    logging.info("Check statuses for tasks: %s", subtask_list)
    for task in subtask_list:
        logging.debug("Child: %s, status: %s", task.id, get_new_status(task))

    failed_subtasks = [
        '{}: {}'.format(lb.task_link(task.id), task_helper.get_task_description(task))
        for task in subtask_list if get_new_status(task) == Status.FAILURE
    ]
    broken_subtasks = [
        '{}: {} [{}]'.format(lb.task_link(task.id), task_helper.get_task_description(task), get_new_status(task))
        for task in subtask_list if get_new_status(task) in Status.Group.BREAK
    ]
    set_children_status_info(myself, failed_subtasks, broken_subtasks)
    if failed_subtasks:
        if fail_on_failed_children:
            if stop_children_on_first_failure:
                for subtask in get_working_subtasks():
                    logging.debug("Stop child: %s", subtask.id)
                    channel.channel.sandbox.server.cancel_task(subtask.id)
            eh.check_failed("See failed child task list above")
        else:
            myself.ctx["task_has_failed_children"] = True

    if broken_subtasks:
        if stop_on_broken_children:
            logging.info("Stopping myself: %s", myself.id)
            myself.set_info("Stopping myself because of broken subtasks...")
            channel.channel.sandbox.server.cancel_task(myself.id)
            return
        else:
            if stop_children_on_first_failure:
                for subtask in get_working_subtasks():
                    logging.debug("Stop child: %s", subtask.id)
                    channel.channel.sandbox.server.cancel_task(subtask.id)
            eh.fail("See broken tasks list above")

    if fail_on_first_failure and amount_of_working_subtasks():
        wait_subtasks_stop(wait_all=False)

    logging.info("All subtasks are in correct state")


def get_new_status(task):
    try:
        return task.new_status
    except Exception:
        return task.status


def set_children_status_info(main_task, failed_subtasks, broken_subtasks):
    if failed_subtasks:
        msg = "Failed child tasks:<br />{}<br />".format("<br />".join(failed_subtasks))
        main_task.set_info(msg, do_escape=False)
    if broken_subtasks:
        msg = "Child tasks finished incorrectly:<br />{}".format("<br />".join(broken_subtasks))
        main_task.set_info(msg, do_escape=False)


def is_task_stop_executing(task):
    """
        Returns true if task stop executing

        This is a pair method to wait_all_tasks_stop_executing()
        Method should be moved to task.is_stop_executing()
        when sandboxsdk2 will be released
    """
    return task.new_status in Status.Group.BREAK or task.new_status in Status.Group.FINISH


def wait_subtasks_stop(wait_all=True):
    """
        Ждать завершения выполнения дочерних задач
        :param wait_all ждать завершения всех подзадач
    """
    subtask_ids = get_working_subtasks()
    if subtask_ids:
        channel.channel.task.wait_tasks(
            subtask_ids,
            statuses=tuple(Status.Group.FINISH) + tuple(Status.Group.BREAK),
            wait_all=wait_all,
            state=None,
        )


def wait_all_subtasks_stop():
    """
        Ждать завершения выполнения всех дочерних задач
    """
    wait_subtasks_stop(wait_all=True)


def subtasks_id_and_status(task_id, limit=500):
    """
        Uses REST API.
        :return Info about subtasks: list with {"id": <id>, "status": <status>}
    """
    rest_client = sandbox.common.rest.Client()
    return rest_client.task.read(
        parent=task_id,
        fields="id,status",
        limit=limit,
        hidden=True,
        children=True,
    ).get("items", [])


def list_subtasks_any_sdk(load=False):
    """
        Gets list to subtasks, works for both sdk1 and sdk2

        :param load: if True, returns list of loaded objects, otherwise list of ids
    """
    rest_client = sandbox.common.rest.Client()
    subtasks_ids = [x["id"] for x in rest_client.task[channel.channel.task.id].children[:]["items"]]
    if not load:
        return subtasks_ids
    subtasks = []
    for subtask_id in subtasks_ids:
        try:
            subtasks.append(channel.channel.sandbox.get_task(subtask_id))
        except Exception:
            logging.debug("Failed to get subtask with sdk1, trying sdk2")
            task = sdk2.Task[subtask_id]
            task.reload()
            subtasks.append(task)
    return subtasks


def restart_broken_subtasks(use_rest=True):
    """
        Перезапускает 1 (один!) раз все дочерние задачи в состояниях Status.Group.BREAK
        (проверяем только непосредственных потомков, внуков не трогаем).
        После перезапуска ждёт, пока они выполнятся.
        Если задачи уже перезапускались, проверяет их корректное завершение
        и дожидается выполнения.
        :param use_rest: use REST API instead of SDK (faster), see MIDDLE-173
    """
    myself = channel.channel.task
    if myself.ctx.get(RESTART_CHILD_KEY):
        logging.info("Subtasks have already been restarted")
        check_subtasks_fails(stop_on_broken_children=True)
        if amount_of_working_subtasks():  # дополнительная проверка - чтобы не зацикливаться
            wait_all_subtasks_stop()  # на случай, если child был перезапущен руками и не успел выполниться
        return

    if use_rest:
        subtasks = subtasks_id_and_status(channel.channel.task.id)
        subtask_ids = [task['id'] for task in subtasks]
        logging.info('restart_broken_subtasks[rest]: full subtasks list (%s): %s', len(subtask_ids), subtask_ids)
        broken_subtask_ids = [task['id'] for task in subtasks if task['status'] in Status.Group.BREAK]
    else:
        subtask_list = myself.list_subtasks(load=True)
        subtask_ids = [task.id for task in subtask_list]
        logging.info('restart_broken_subtasks[sdk]: full subtasks list (%s): %s', len(subtask_ids), subtask_ids)
        broken_subtask_ids = [task.id for task in subtask_list if task.status in Status.Group.BREAK]

    if broken_subtask_ids:
        myself.ctx[RESTART_CHILD_KEY] = True

    for task_id in broken_subtask_ids:
        logging.info("Restarting task #{}", task_id)
        channel.channel.sandbox.server.restart_task(task_id)
    else:
        logging.info("No subtasks to restart")
        check_subtasks_fails()

    wait_all_subtasks_stop()


def flatten(iterable):
    """
        Convert nested lists to flat generator
        Ex.: list(flatten([1,2,[3,[4]],8])) -> [1,2,3,4,8]
    """
    for i in iterable:
        if isinstance(i, (list, tuple, set, GeneratorType)):
            for j in flatten(i):
                yield j
        else:
            yield i


def colored_status(status):
    """ Use class statuses from sandbox"""
    return '<span class="status status_{}">{}</span>'.format(status.lower(), status)


def get_svn_info(path):
    """
        Получить информацию о svn репозитории.

        :param path: путь до папки или url
        :return: словарь с информацией о репозитории
    """
    return sdk2.svn.Arcadia.info(path)


def svn_revision(svn_url):
    """
        Возвращает ревизию по заданному svn_url.
        Если получить ревизию не удалось, возвращает None.

        :param svn_url: путь до папки или url
    """
    logging.info('Get revision for %s', svn_url)
    try:
        return sdk2.svn.Arcadia.get_revision(svn_url)
    except (sdk2.svn.SvnError, sdk2.svn.SvnIncorrectUrl):
        pass


def svn_last_change(svn_url):
    """
        Возвращает ревизию последнего изменения по заданному svn_url.
        Если получить ревизию не удалось, возвращает None.

        :param svn_url: путь до папки или url
        :return: номер ревизии или None, если ревизию получить не удалось
    """
    # TODO: use everywhere it is possible
    logging.info('Get last changed revision for %s', svn_url)
    try:
        return sdk2.svn.Arcadia.info(svn_url)['commit_revision']
    except (sdk2.svn.SvnError, sdk2.svn.SvnIncorrectUrl):
        return None


def branch_tag_from_svn_url(svn_url):
    """
        Вырезает из svn-урла имя ветки или тега, если они там присутствуют.

        :Примеры:
            arcadia:/arc/tags/images/middle/stable-37/r3/arcadia ->
                (None, "images/middle/stable-37/r3")
            svn+ssh://arcadia.yandex.ru/arc/tags/images/middle/stable-37/r3/arcadia ->
                (None, "images/middle/stable-37/r3")
            svn+ssh://sepe@arcadia.yandex.ru/arc/branches/images/base/imgsearch-1/arcadia ->
                ("images/base/imgsearch-1", None)

        :return: (<имя ветки>, <имя тега>)
    """
    parsed_url = sdk2.svn.Arcadia.parse_url(svn_url)
    path_parts = parsed_url.path.split('/')
    logging.debug('Parsed url: %s; parts: %s', parsed_url, path_parts)

    tag, branch = None, None
    if len(path_parts) >= 4:
        tb_type = path_parts[1]
        tb_id = '/'.join(path_parts[2:-1])
        if path_parts[-1] != 'arcadia':
            tb_id += '/' + path_parts[-1]
        if tb_type == 'tags':
            tag = tb_id
        elif tb_type == 'branches':
            branch = tb_id

    return branch, tag


def get_short_branch_name(svn_url):
    """
        Получает короткое имя бранча (последний компонент пути перед /arcadia)
        из @svn_url. Не делает различий между бранчами, тегами и trunk.

        :Примеры:
            arcadia:/arc/branches/robot/robot-stable-2014-02-17/arcadia -> robot-stable-2014-02-17
            svn+ssh://arcadia.yandex.ru/arc/branches/robot/robot-stable-2014-02-17/arcadia -> robot-stable-2014-02-17
            svn+ssh://arcadia.yandex.ru/arc/tags/rca/stable-4-1/arcadia -> stable-4-1
            svn+ssh://arcadia-ro.yandex.ru/arc/trunk/arcadia@1336626 -> trunk

        :return: короткое имя бранча, если найдено, иначе - None.
    """
    logging.info('Get short branch name from svn url %s', svn_url)
    svn_url = sdk2.svn.Arcadia.normalize_url(svn_url)
    search_result = re.search(
        r'/((branches|tags)/(.*/)*(?P<branch>.*)|(?P<trunk>trunk))/arcadia',
        svn_url)
    if not search_result:
        logging.warning('Cannot get short branch name from %s.', svn_url)
        return None
    if search_result.group('branch'):
        short_branch_name = search_result.group('branch')
    elif search_result.group('trunk'):
        short_branch_name = search_result.group('trunk')
    else:
        logging.warning('Unexpected search results.')
        return None
    logging.info('Short branch name: %s.', short_branch_name)
    return short_branch_name


def wait_searcher_start(host, port, timeout=240, subproc_list=None, generate_coredump=True):
    from sandbox.sandboxsdk import network

    finish = time.time() + timeout
    if not subproc_list:
        subproc_list = []
    while time.time() < finish:
        if not network.is_port_free(port, host):
            check_processes(subproc_list)
            return

        check_processes(subproc_list)

        time.sleep(1)

    if subproc_list and generate_coredump:
        for p in subproc_list:
            logging.info('killing process %s', p.pid)
            os.kill(p.pid, 6)  # generate coredump
        time.sleep(10)

    eh.check_failed("Did not connect to {}:{} in {} seconds".format(host, port, timeout))


def wait_searcher_stop(host, port, timeout=240, subproc_list=None):
    # FIXME(mvel): rename to wait_service_start (or wait_server_start, etc)
    from sandbox.sandboxsdk import network

    finish = time.time() + timeout
    if not subproc_list:
        subproc_list = []
    while time.time() < finish:
        if network.is_port_free(port, host):
            check_processes(subproc_list)
            return
        check_processes(subproc_list)
        time.sleep(1)
    eh.check_failed("Port {} on host {} is alive at {} seconds".format(port, host, timeout))


def last_resource_with_released_attribute(
    resource_type, release_status=ctt.ReleaseStatus.STABLE, arch=None, task_type=None, order_by='-id',
):
    """
    Reimplementation of sandbox method: get_last_released_resource.
    Get latest resource of given type with attrs {"released": release_status}. By default, release_status == STABLE.
    If task_type specified - check 10 last releases to have specified task type and return last of them.

    :param resource_type: resource type (string or AbstractResource successor)
    :param release_status release status (STABLE by default)
    :param arch: resource arch

    :return: SandboxResource instance or None if nothing was found
    """
    apihelpers.check_res(resource_type)
    arch = arch if arch else common.utils.get_sysparams()['arch']
    params = {
        "resource_type": resource_type,
        "arch": arch,
        "all_attrs": {
            "released": release_status
        },
        "order_by": order_by,
        "limit": 10,
    }
    logging.debug("[utils] Look for release with params: %s", params)
    stable_releases = channel.channel.sandbox.list_resources(**params)
    logging.debug("[utils] Last 10 stable releases: %s", stable_releases)
    if task_type is None:
        return stable_releases[0] if stable_releases else None
    else:
        for res in stable_releases:
            task_id = res.task_id
            if channel.channel.sandbox.get_task(task_id).type == task_type:
                return res


def get_and_check_last_released_resource_id(
    resource_type,
    release_status=ctt.ReleaseStatus.STABLE,
    arch=None,
    order_by=None,
    use_new_version=False,
):
    # FIXME(mvel): copypaste in apihelpers
    """
        Try to get last resource of the given type.
        By default, searches for STABLE releases and current system's arch.
        Raises SandboxTaskUnknownError when resource is not found.
        :@param release_status release status (STABLE, TESTING, etc)
        :@param arch system architecture
        :@return: resource.id
    """
    logging.info("Obtaining last stable resource of type %s", resource_type)
    func = last_resource_with_released_attribute if use_new_version else apihelpers.get_last_released_resource
    res = func(
        resource_type=resource_type,
        release_status=release_status,
        arch=arch,
        order_by=order_by
    )
    eh.verify(
        res is not None,
        "Cannot find the last {} released resource {}".format(release_status, resource_type)
    )
    return res.id


def get_and_check_last_resource_with_attribute(
    resource_type,
    attr_name=None,
    attr_value=None,
    arch=None
):
    """
        Try to get resource of the given type by attribute
        Raises SandboxTaskUnknownError when resource not found.

        :return: resource object
    """
    resource = apihelpers.get_last_resource_with_attribute(
        resource_type=resource_type,
        attribute_name=attr_name,
        attribute_value=attr_value,
        arch=arch
    )
    if resource:
        return resource
    eh.fail("Can't find {} resource with attr: {}={} for arch {}".format(resource_type, attr_name, attr_value, arch))


def set_resource_attributes(resource, attributes):
    """
        Проставить dict атрибутов ресурсу.

        :param resource: ресурс, либо его id
        :param attributes: dict из атрибутов (или любое другое нечто, у которого есть iteritems).
    """
    for name, value in six.iteritems(attributes):
        logging.info("Set attribute %s=%s to resource %s", name, value, resource)
        channel.channel.sandbox.set_resource_attribute(resource, name, value)


def sync_resource(resource_id):
    """
        Synchronizes resource of the given id and returns its local (on client) path.
        When resource is already synced, just returns path.
        Raises SandboxTaskUnknownError when resource has invalid ID (0, None, etc).
    """
    eh.verify(resource_id, "Cannot sync resource: resource id not set.")
    return channel.channel.task.sync_resource(resource_id)


def sync_last_stable_resource(
    resource_type,
    arch=None,
):
    """
        Synchronizes last stable resource of the given type (and current system's arch)
        and returns path to it. Raises SandboxTaskUnknownError when resource not found.
    """
    res_id = get_and_check_last_released_resource_id(resource_type, arch=arch)
    logging.info("Got last stable resource of type %s: %s, syncing...", resource_type, res_id)
    return channel.channel.task.sync_resource(res_id)


def get_bsconfig():
    """
        Получить путь до последнего релизного bsconfig.

        :return: путь до bsconfig, его его релизы есть в системе, None в противном случае
    """
    logging.error("Are you still using bsconfig, please ask mvel@ why you don't need it!")
    # получаем информацию по релизам в Sandbox-е
    resource = apihelpers.get_last_released_resource(
        resource_types.BSCONFIG
    )
    if resource:
        return channel.channel.task.sync_resource(resource)
    else:
        return None


def check_processes(processes):
    for p in processes:
        if p.poll() is not None:
            process.throw_subprocess_error(p)


def splitthousands(n, sep=' '):
    s = str(n)
    if len(s) <= 3:
        return s
    return splitthousands(s[:-3], sep) + sep + s[-3:]


def create_misc_resource_and_dir(
    ctx, ctx_field_name, descr, dir_name,
    resource_type=resource_types.OTHER_RESOURCE,
    task=None
):
    resource_id = ctx.get(ctx_field_name)
    if not resource_id:
        # Temporary fix for sdk2 compatibility
        if task is not None:
            resource = resource_type(task, descr, dir_name)
        else:
            resource = channel.channel.task.create_resource(descr, dir_name, resource_type)
        ctx[ctx_field_name] = resource.id
    else:
        resource = channel.channel.sandbox.get_resource(resource_id)

    paths.make_folder(str(resource.path))
    # This fixes SEARCH-787 and similar problems
    fu.write_file(
        os.path.join(str(resource.path), dir_name + '-dummy.txt'),
        'Dummy file to prevent empty resource creation',
    )

    return str(resource.path)


@decorators.retries(3, 10)
def get_config_by_info_request(host, port, output_path=None, verbose=True, min_size=1000):
    """
        Download config from host:port instance
        :param host: instance host
        :param port: instance port
        :param output_path: path to save config. When None, saves to temporary file
        :param verbose: print config to debug log when set
        :param min_size: minimal config size in bytes
        :return: config as string
    """
    url_getconf = "http://{}:{}/yandsearch?info=getconfig".format(host, port)
    logging.info('Obtaining config by info request via url %s', url_getconf)
    if output_path is None:
        output_path = paths.get_unique_file_name('', 'tmp_conf_path.cfg')

    r = requests_wrapper.get_r(url_getconf)
    with open(output_path, 'w') as f:
        f.write(r.text)
    config = fu.read_file(output_path)
    if verbose:
        logging.debug("Config:\n%s", config)

    if not config or os.path.getsize(output_path) < min_size:
        eh.check_failed("Error: too short config found, less than {} bytes".format(min_size))

    return config


def is_check_config_request(request):
    return 'info=checkconfig' in request


def get_binary_version(binary_resource):
    # used only for task descriptions
    try:
        arc_url = channel.channel.sandbox.get_task(binary_resource.task_id).ctx.get(constants.ARCADIA_URL_KEY, "")
    except Exception:
        arc_url = (
            getattr(sdk2.Task[binary_resource.task_id].Parameters, constants.ARCADIA_URL_KEY, "") or
            getattr(sdk2.Task[binary_resource.task_id].Context, constants.ARCADIA_URL_KEY, "")
        )
    tag = (
        re.search(r'.*/arc/tags/.*-([0-9]+-[0-9]+)/arcadia.*', arc_url) or
        re.search(r'.*/arc/tags/.*(r[0-9]+)/arcadia.*', arc_url)
    )
    return tag.group(1) if tag else "_unknown_"


def get_user_queries(queries_res):
    queries_resource_local_file = sync_resource(queries_res.id)
    return yield_user_queries(queries_resource_local_file)


def yield_user_queries(queries_resource_file):
    with open(queries_resource_file) as f:
        for line in f:
            stripped_line = line.strip(" \n")
            if stripped_line:
                fields = [i for i in stripped_line.split("\t") if i]
                if len(fields) < 2:
                    logging.info(
                        "Wrong user queries format: {}".format(stripped_line) +
                        "(should be user request \\t region \\t PLATFORM \\t TEST-SET)"
                    )
                else:
                    user_query = fields[0]
                    region = int(fields[1])
                    eh.verify(region > 0, "Invalid region")
                    device = None if len(fields) == 2 else fields[2]
                yield six.moves.urllib.parse.quote(user_query), region, None, device


def check_resources_by_md5(resource1, resource2, path_to_diffs):
    if resource1.file_md5 != resource2.file_md5:
        return False
    logging.info('resources have same md5')
    channel.channel.task.ctx['same_md5'] = resource1.file_md5
    no_diff_name = os.path.join(path_to_diffs, NO_DIFF_NAME)
    fu.write_file(no_diff_name, no_diff_detected())
    return True


def check_tasks_to_finish_correctly(task_ids_list):
    """For tasks, that can NOT be children"""
    tasks = {t_id: task_helper.task_status(t_id) for t_id in task_ids_list}
    stop_statuses = tuple(Status.Group.FINISH) + tuple(Status.Group.BREAK)
    task_statuses_info = "\n".join(["{0}: {1}".format(t_id, t_status) for t_id, t_status in six.iteritems(tasks)])

    if any(map(lambda status: status not in stop_statuses, tasks.values())):
        logging.info("Statuses:\n%s", task_statuses_info)
        channel.channel.task.wait_tasks(task_ids_list, statuses=stop_statuses, wait_all=True)

    bad_tasks = [t_id for t_id, t_status in six.iteritems(tasks) if t_status not in Status.Group.SUCCEED]
    if bad_tasks:
        channel.channel.task.set_info("Some tasks are not finished correctly:")
        for t_id in bad_tasks:
            channel.channel.task.set_info("{0}: {1}".format(lb.task_link(t_id), tasks[t_id]), do_escape=False)
        eh.fail("Tasks are not finished correctly")
    logging.info("Tasks finished successfully")


def terminate_process(proc, kill_timeout=60):
    """
        Gracefully terminate running process
    """

    waitfor = time.time() + kill_timeout
    while time.time() < waitfor:
        try:
            proc.terminate()
        except EnvironmentError as e:
            if e.errno != errno.ESRCH:
                logging.warning("Failed to terminate process: {}".format(e))
            return
        time.sleep(0.1)
    try:
        proc.kill()
    except EnvironmentError as e:
        if e.errno != errno.ESRCH:
            logging.warning("Failed to kill process: {}".format(e))
        return

    proc.wait()


def show_process_tail(dead_process, tail_size=15):
    """
        Shows tail of dead/killed/coredumped process stderr
        and link to complete stderr/stdout
    """
    eh.verify(dead_process is not None, "Process is empty. Did you start it?")

    if dead_process.returncode == 0:
        # process looks fine
        return

    task = channel.channel.task
    std_path = getattr(dead_process, "stderr_path") or getattr(dead_process, "stdout_path")
    if not std_path:
        task.set_info(
            "Cannot show process tail: no stderr and stdout paths. "
            "Probably, you use this function with sdk2-like processes."
        )
        return

    # cut output tail
    tail_process = process.run_process(
        ['tail', '-n', str(tail_size), str(std_path)],
        shell=True,
        outputs_to_one_file=True,
        log_prefix='tail',
    )
    try:
        stderr_tail_contents_html = cgi_escape(fu.read_file(tail_process.stdout_path))
        if len(stderr_tail_contents_html) > 100000:
            stderr_tail_contents_html = '[Output is too large, truncated. See entire log for details]'

        task.set_info(
            "Process output tail:<br/>\n<pre>{}</pre>\n{}".format(
                stderr_tail_contents_html,
                utils2.resource_redirect_link(
                    task._log_resource.id,
                    "Complete process output",
                    path=os.path.basename(std_path),
                ),
            ),
            do_escape=False,
        )
    except Exception as e:
        eh.log_exception("Unable to set info about process output", e)


class TimeCounter(object):
    def __init__(self, activity):
        self.activity = activity

    def __enter__(self):
        logging.info("Started %s", self.activity)
        self.start_time = time.time()

    def __exit__(self, type, value, traceback):
        delta = time.time() - self.start_time
        self.time_spent = str(datetime.timedelta(seconds=delta))
        logging.info("Finished %s. Time spent: %s", self.activity, self.time_spent)


def get_tag_from_binary_version(getversion_text):
    tags = re.search(r"URL:[^\n]+arcadia.yandex.ru/arc/tags/(.*)/arcadia", getversion_text)
    if tags:
        tag = tags.group(1)
        logging.info("Tag got: %s", tag)
        return tag


def receive_skynet_key(vault_owner):
    """
    Obtain a skynet key, which is a must for carrying out operations via cqueue.
    If you find your tasks still using cqueue, please reconsider the usage,
    since it's currently deprecated.
    """

    client_settings = common.config.Registry().client
    skynet_key_path = client_settings.skynet_key.path
    if not skynet_key_path:
        logging.warning("No skynet key path specified in config, returning")

    user = client_settings.sandbox_user or getpass.getuser()
    skynet_key_path = client_settings.skynet_key.path
    skynet_key_path = os.path.join(
        os.path.expanduser("~{}".format(user)),
        skynet_key_path
    )
    try:
        skynet_key = sdk2.Vault.data(vault_owner, client_settings.skynet_key.name)
        logging.info("Creating %r", skynet_key_path)
        skynet_key_dir = os.path.dirname(skynet_key_path)
        if not os.path.exists(skynet_key_dir):
            os.makedirs(skynet_key_dir)
        with open(skynet_key_path, "w") as f:
            f.write(skynet_key)
        mode = 0o600
        logging.info("Set permissions as 0%o on %r", mode, skynet_key_path)
        os.chmod(skynet_key_path, mode)
    except common.errors.TaskFailure as exc:
        logging.error("Skynet key not received: %s. Removing %r", exc, skynet_key_path)
        try:
            os.unlink(skynet_key_path)
        except OSError:
            pass


def gimme_port():
    """
    Obtain random free tcp port.
    :return: random free port
    :rtype: int
    """
    from contextlib import closing

    with closing(socket.socket(socket.AF_INET, socket.SOCK_STREAM)) as sock:
        sock.bind(('', 0))
        return sock.getsockname()[1]


def get_task_log(task_id, get_common=False):
    """
        Gets a log for an incomplete task (differs from the base class method)
    """
    return None


def sandbox_resource(resource_type, attribute_name, attribute_value):
    """
        Find and download resource with given type and attribute_name=attribute_value,
        return local path to its data afterwards.
    """

    resource = apihelpers.get_resources_with_attribute(
        resource_type, attribute_name, attribute_value, limit=1, arch=sandbox.sandboxsdk.util.system_info()['arch'])
    if not resource:
        eh.check_failed("Can't get {} with {} = {}".format(resource_type, attribute_name, attribute_value))

    return channel.channel.task.sync_resource(resource[0])
