# coding: utf-8

"""
    Standard Sandbox Python Library for SDK2
    Please DO NOT USE sdk1 legacy code here (sandbox/sandboxsdk, etc)

    Misc code used in various SDK2 tasks.

    Please obey PEP8 coding standards and Sandbox StyleGuide here
    https://wiki.yandex-team.ru/sandbox/codestyleguide

    New functions without documentation comments are FORBIDDEN.

    SSPL maintainers: mvel@
"""

import logging

from sandbox.projects.common import link_builder as lb
from sandbox.common.types.task import Status
from sandbox.common import errors
from sandbox.common import rest
import sandbox.sdk2 as sdk2


def resource_redirect_url(resource_id, path=None):
    """
        Формирует URL, по которому можно скачать ресурс
        или файл в составе ресурса
        :param resource_id: id ресурса
        :param path: путь до файла внутри ресурса (если ресурс представляет собой директорию)
    """
    url = rest.Client().resource[resource_id][:]["http"]["proxy"]
    if path is not None:
        url += "/{0}".format(path)
    return url


def resource_redirect_link(resource_id, title, path=None):
    """
        Формирует HTML-ссылку, по которой можно скачать ресурс
        или файл в составе ресурса
        :param resource_id: id ресурса
        :param title: заголовок ссылки
        :param path: путь до файла внутри ресурса (если ресурс представляет собой директорию)
    """
    return '<a href="{0}" target="_blank">{1}</a>'.format(
        resource_redirect_url(resource_id, path),
        title,
    )


def is_human(user):
    return not user.startswith("robot-")


def check_tasks_to_finish_correctly(task, task_ids_list):
    """For tasks, that can NOT be children"""
    status_key = "status"
    tasks = task.server.task.read(
        id=task_ids_list,
        fields=status_key,
        hidden=True,
        children=True,
        limit=len(task_ids_list)
    )["items"]
    stop_statuses = tuple(Status.Group.FINISH) + tuple(Status.Group.BREAK)
    task_statuses_info = "\n".join(["{0}: {1}".format(t["id"], t["status"]) for t in tasks])
    logging.info("Statuses:\n%s", task_statuses_info)

    if any(t["status"] not in stop_statuses for t in tasks):
        raise sdk2.WaitTask(task_ids_list, statuses=stop_statuses, wait_all=True)

    bad_tasks = [t for t in tasks if t["status"] not in Status.Group.SUCCEED]
    if bad_tasks:
        task.set_info("Some tasks are not finished correctly:")
        for t in bad_tasks:
            task.set_info("{0}: {1}".format(lb.task_link(t["id"]), t["status"]), do_escape=False)
        raise errors.TaskFailure("Tasks are not finished correctly")
    logging.info("Tasks finished successfully")
