from __future__ import absolute_import

import os
import uuid
import logging
import platform

from six.moves.urllib import parse as urlparse

from sandbox.common import fs
from sandbox.common import errors as common_errors
from sandbox.common import patterns
from sandbox.common import itertools as common_itertools
from sandbox.common.vcs import cache as vcs_cache
from sandbox.common.types import task as ctt

from sandbox import sdk2
from sandbox.sdk2.helpers import process
from sandbox.sdk2 import service_resources
from sandbox.sdk2.vcs import svn

from sandbox.sandboxsdk import channel

import sandbox.agentr.types as agentr_types

from sandbox.projects.common.vcs import util

logger = logging.getLogger("vcs.aapi")


class ArcadiaApiCommandFailed(common_errors.SandboxException):
    pass


class ArcadiaApiUnexpectedInput(ArcadiaApiCommandFailed):
    pass


class ArcadiaApiUnexpectedOuput(ArcadiaApiCommandFailed):
    pass


class ArcadiaApiNoSuchRevision(ArcadiaApiCommandFailed):
    pass


class ArcadiaApi(vcs_cache.CacheableArcadiaApi):
    """ Class for working with Arc repository through Arcadia API """

    @classmethod
    def _disc_store(cls):
        path = os.path.abspath(os.path.join(cls.base_cache_dir, "aapi_store"))

        if not os.path.exists(path):
            fs.make_folder(cls.base_cache_dir)

        cls.add_cache_metadata(path, sdk2.Task.current.id)
        return path

    @patterns.singleton_classproperty
    def _client_bin(cls):
        return sdk2.ResourceData(
            sdk2.Resource.find(
                service_resources.ArcadiaApiClient,
                attrs={"released": ctt.ReleaseStatus.STABLE}
            ).first()
        ).path.as_posix()

    @classmethod
    def _ensure_has_svn_revision(cls, revision):
        if not cls.has_svn_revision(revision):
            raise ArcadiaApiNoSuchRevision("No such revision {} in arcadia-api".format(revision))

    @classmethod
    def export_svn_path(cls, svn_path, dest, revision=None):
        if os.path.exists(dest):
            raise ArcadiaApiUnexpectedInput("Export destanation path {} exists".format(dest))

        if revision:
            cls._ensure_has_svn_revision(revision)

        cmd = [cls._client_bin, "svn-export", svn_path, dest, "-s", cls._disc_store()]

        if revision is not None:
            cmd.extend(["-r", str(revision)])

        with process.ProcessLog(task=sdk2.task.Task.current, logger="aapi") as pl:
            process.subprocess.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr).wait()

    @classmethod
    def mount_svn_path(cls, svn_path, revision=None):
        if not util.fuse_available():
            raise ArcadiaApiCommandFailed("Fuse is not available on {} yet".format(platform.platform()))

        if revision:
            cls._ensure_has_svn_revision(revision)

        mount_point = str(sdk2.task.Task.current.path(agentr_types.FUSE_DIRNAME) / ("mount_point_" + str(uuid.uuid4())))
        fs.make_folder(mount_point)
        mount_cmd = [cls._client_bin, "svn-fuse-mount", "-s", cls._disc_store(), svn_path, mount_point]
        if revision:
            mount_cmd += ["-r", str(revision)]

        return util.MountPoint(mount_point, mount_cmd, util.AAPI_SVN, ArcadiaApiCommandFailed)

    @classmethod
    def svn_head(cls):
        try:
            out = process.subprocess.check_output([cls._client_bin, "svn-head"])
        except process.subprocess.CalledProcessError as e:
            raise ArcadiaApiCommandFailed("Can't get arcadia-api svn-head: {}".format(e))

        try:
            svn_head = int(out)
        except (TypeError, ValueError):
            raise ArcadiaApiUnexpectedOuput("Svn head returned unexpected string: {}".format(out))

        return svn_head

    @classmethod
    def has_svn_revision(cls, revision):
        try:
            r = int(revision)
        except (TypeError, ValueError):
            return False

        return r <= cls.svn_head()

    @classmethod
    def wait_svn_revision(cls, revision, timeout=60):
        has_revision, _ = common_itertools.progressive_waiter(0, 10, timeout, lambda: cls.has_svn_revision(revision))
        return has_revision

    @classmethod
    def extract_path_and_revision(cls, url):
        return util.extract_path_details(url)[:2]

    @classmethod
    def _set_hg_tag(cls):
        if hasattr(channel.channel.task, "tags"):
            # SDK1
            task = channel.channel.task
            task.tags = list(set(task.tags + ["AAPI_HG"]))
        else:
            # SDK2
            task = sdk2.task.Task.current
            task.Parameters.tags = list(set(task.Parameters.tags + ["AAPI_HG"]))

    @classmethod
    def mount_hg_path(cls, hg_path, changeset):
        cls._ensure_is_hg_hash(changeset)

        hg_path = hg_path.strip("/")

        if not util.fuse_available():
            raise ArcadiaApiCommandFailed("Fuse is not available on {} yet".format(platform.platform()))

        cls._ensure_has_hg_changeset(changeset)

        mount_point = str(sdk2.task.Task.current.path(agentr_types.FUSE_DIRNAME) / ("mount_point_" + str(uuid.uuid4())))
        fs.make_folder(mount_point)
        mount_cmd = [cls._client_bin, "hg-fuse-mount", "-s", cls._disc_store(), "-c", changeset, hg_path, mount_point]

        return util.MountPoint(mount_point, mount_cmd, util.AAPI_HG, ArcadiaApiCommandFailed)

    @classmethod
    def is_hg_scheme(cls, url):
        normalized_url = svn.Arcadia.normalize_url(url)
        p = urlparse.urlparse(normalized_url)
        return p.scheme == svn.Arcadia.ARCADIA_HG_SCHEME

    @classmethod
    def _is_hg_hash(cls, s):
        if len(s) != 40:
            return False

        try:
            s.decode("hex")
        except TypeError:
            return False

        return True

    @classmethod
    def _ensure_is_hg_hash(cls, s):
        if not cls._is_hg_hash(s):
            raise ArcadiaApiUnexpectedInput("Expected 20-bytes hex string, got: \"{}\"".format(s))

    @classmethod
    def hg_id(cls, name):
        cls._set_hg_tag()

        try:
            out = process.subprocess.check_output([cls._client_bin, "hg-id", name])
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
            raise ArcadiaApiCommandFailed("Hg identify {} failed: {}".format(name, e))

        out = out.strip()

        if not cls._is_hg_hash(out):
            raise ArcadiaApiUnexpectedOuput("Hg identify returned unexpected string: \"{}\"".format(out))

        logger.info("Hg identify %s -> %s", name, out)

        return out

    @classmethod
    def has_hg_changeset(cls, changeset):
        cls._ensure_is_hg_hash(changeset)

        try:
            out = process.subprocess.check_output([cls._client_bin, "hg-changeset-info", "-c", changeset])
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
            return False

        if changeset not in out:
            raise ArcadiaApiUnexpectedOuput("Hg changeset-info returned unexpected string: \"{}\"".format(out))

        return True

    @classmethod
    def _ensure_has_hg_changeset(cls, changeset):
        if not cls.has_hg_changeset(changeset):
            raise ArcadiaApiNoSuchRevision("No such changeset {} in arcadia-api".format(changeset))

    @classmethod
    def wait_hg_changeset(cls, changeset, timeout=60):
        cls._ensure_is_hg_hash(changeset)
        has_changeset, _ = common_itertools.progressive_waiter(0, 10, timeout, lambda: cls.has_hg_changeset(changeset))
        return has_changeset
