from __future__ import absolute_import

import json
import logging
import os
import platform
import re
import six
import sys
import tempfile
import uuid

from sandbox.common import fs
from sandbox.common import enum
from sandbox.common import errors as common_errors
from sandbox.common import itertools as common_itertools
from sandbox.common import patterns
from sandbox.common.vcs import cache as vcs_cache

from sandbox import sdk2
from sandbox.sdk2.helpers import process

import sandbox.agentr.types as agentr_types

from sandbox.projects.common.vcs import util

logger = logging.getLogger("vcs.arc")


class ResetMode(enum.Enum):
    SOFT = "soft"
    HARD = "hard"
    MIXED = "mixed"


class ArcCommandFailed(common_errors.SandboxException):
    pass


class ArcNoSuchRevision(ArcCommandFailed):
    pass


class ArcBareRepo(object):
    def __init__(self, init_cmd, repo_path=None):
        if not repo_path:
            repo_path = os.path.join(tempfile.gettempdir(), 'arc_bare_repo_' + str(uuid.uuid4()))

        self.repo_path = repo_path
        self.init_cmd = init_cmd

    def __enter__(self):
        fs.make_folder(self.repo_path)

        proc = process.subprocess.Popen(
            self.init_cmd,
            cwd=self.repo_path,
            stdout=process.subprocess.PIPE,
            stderr=process.subprocess.PIPE
        )
        out, err = proc.communicate()
        return_code = proc.poll()
        if return_code:
            raise ArcCommandFailed(
                "Failed to initialize bare arc repo `{}`. Return code: {}.\nOUTPUT: {}\nERROR: {}".format(
                    " ".join(self.init_cmd), return_code, out, err
                )
            )

        return self.repo_path

    def __exit__(self, exc_type, exc_val, exc_tb):
        fs.remove_path(self.repo_path)


class Arc(vcs_cache.CacheableArcVcs):
    """
    Class for working with Arcadia repository through Arc Vcs.

    If you want to only checkout Arcadia repository and not preform any Arc
    specific actions, it's recommended to use `mount_arc_path()` present in
    `sandbox/projects/common/arcadia`. That is a better high-level API with
    more options.

    In case you want to control how and where repository is mounted using
    Arc, then you should use this class.

    The repository can be mounted with `trunk` checked out as:

        with Arc.mount_path("", "trunk"):
            pass

    The mount_path() function should be the entry point. It returns a
    sandbox.projects.common.vcs.util.MountPoint() object which can be used as an
    context manager as demonstrated above.

    mount_path() provides way to pass path to mount_point or store.
    All the mounts done by arc shared a common object store on Sandbox which
    cannot be configured.
    """

    def __init__(self, secret_name="ARC_TOKEN", secret_owner=None, arc_oauth_token=None, yav_oauth_token=None):
        super(Arc, self).__init__()
        self.secret_name = secret_name
        self.secret_owner = secret_owner
        self.arc_oauth_token = arc_oauth_token
        self.yav_oauth_token = yav_oauth_token

    @patterns.singleton_property
    def binary_path(self):
        from sandbox.sdk2 import environments
        return environments.ArcEnvironment(self.secret_name, self.secret_owner, self.arc_oauth_token, self.yav_oauth_token).prepare()

    def _disc_store(self):
        """
        Use this as a shared object store for all Arc repositories.
        """
        path = os.path.abspath(os.path.join(self.base_cache_dir, "arc_vcs"))

        if not os.path.exists(path):
            fs.make_folder(self.base_cache_dir)

        return path

    @patterns.singleton_classproperty
    def _client_bin(cls):
        """ returns the path to the arc binary
        The arc resource is present in compressed format on Sandbox. This finds
        the compressed resource, uncompress it and returns the path of
        resulting binary.
        """
        from sandbox.sdk2 import environments
        return environments.ArcEnvironment().prepare()

    def svn_rev_to_arc_hash(self, mount_point, rev):
        """Get arc hash from svn revision."""
        logger.info("Getting arc hash for svn rev %s", rev)
        return self._execute_command(
            mount_point, [self.binary_path, "dump", "svn-rev", str(rev)],
            fail_msg_prefix="Unable to get arc hash for SVN rev {}".format(rev)
        )

    def rev_parse(self, mount_point, ref):
        """
        Resolve hash of provided reference
        :param mount_point: path where arc mounted
        :param ref: reference (branch, revision, etc.)
        :return: str, command output
        """
        logging.info("Start arc rev-parse for ref {}".format(ref))
        try:
            out = self._execute_command(
                mount_point, [self.binary_path, "rev-parse", ref],
                fail_msg_prefix="Unable to resolve '{}'".format(ref)
            )
            return out.splitlines()[-1].decode("utf-8")
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
        except ArcCommandFailed as arc_e:
            logger.error(arc_e)
        return None

    def describe(
        self,
        mount_point,
        commits=None,
        all=False,
        exact_match=False,
        match=None,
        exclude=None,
        always=False,
        dirty=None,
        svn=False
    ):
        """
        Finds the most recent tag that is reachable from a provided commits.
        :param mount_point: path where arc mounted
        :param commits: commits that should be resolved
        :param all: whether to use use any ref
        :param exact_match: only output exact matches
        :param match: only consider tags matching provided pattern
        :param exclude: do not consider tags matching provided pattern
        :param always: show abbreviated commit object as fallback
        :param dirty: append provided mark on dirty working tree
        :param svn: use svn revisions as tags
        :return: list of str, command output
        """
        logging.info("Start arc describe")
        cmd = [self.binary_path, "describe"]
        if not commits is None:
            if isinstance(commits, str):
                commits = [commits]
            cmd.extend(commits)

        if all:
            cmd.extend(["--all"])
        if exact_match:
            cmd.extend(["--exact-match"])
        if match is not None:
            cmd.extend(["--match", match])
        if exclude is not None:
            cmd.extend(["--exclude", exclude])
        if always:
            cmd.extend(["--always"])
        if dirty is not None:
            if isinstance(dirty, bool):
                if dirty:
                    cmd.extend(["--dirty"])
            else:
                cmd.extend(["--dirty", dirty])
        if svn:
            cmd.extend(["--svn"])

        try:
            out = self._execute_command(mount_point, cmd)
            return out.decode("utf-8").splitlines()
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
        except ArcCommandFailed as arc_e:
            logger.error(arc_e)
        return None

    @staticmethod
    def _execute_command(mount_point, command, fail_msg_prefix="Unable to execute arc command"):
        """
        Execute arc command and catch exceptions
        :param mount_point: path where arc mounted
        :param command: list with strings
        :return: str, command_output
        """
        p = process.subprocess.Popen(command, cwd=mount_point, stdout=process.subprocess.PIPE,
                                     stderr=process.subprocess.PIPE)
        out, err = p.communicate()
        return_code = p.poll()
        if return_code:
            raise ArcCommandFailed("{} `{}`. Return code: {}.\nOUTPUT: {}\nERROR: {}".format(
                fail_msg_prefix, " ".join(command), return_code, out, err
            ))
        return out.strip()

    def fetch(self, mount_point, branch=None):
        """
        Fetch branch from remote repository. Returns True if branch exists and False otherwise.
        Does not perform unmount if the rev does not exists
        :param mount_point: path where arc mounted
        :param branch: branch name
        """
        logger.info("Start arc fetch")
        command = [self.binary_path, "fetch"]
        if branch is not None:
            command.append(branch)
        try:
            self._execute_command(mount_point=mount_point, command=command)
            return True
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
        except ArcCommandFailed as arc_e:
            logger.error(arc_e)
        return False

    def pull(self, mount_point, rebase=False):
        """
        Fetch from remote repository and integrate with a local branch
        :param mount_point: path where arc mounted
        :param rebase: rebase if branches diverge
        :return: str, command output
        """
        logger.info("Start arc pull")
        command = [self.binary_path, "pull"]
        if rebase:
            command.append("--rebase=true")
        return self._execute_command(mount_point=mount_point, command=command)

    def export(self, mount_point, rev, repo_path, export_path):
        """
        Export files from repository
        :param mount_point: path where arc mounted
        :param rev: Arcadia revision
        :param repo_path: path in repository to export
        :param export_path: path where files will be exported
        :return: str, command output
        """
        logger.info("Start arc export")
        command = [self.binary_path, "export", rev, repo_path, "--to", export_path]
        return self._execute_command(mount_point=mount_point, command=command)

    def show(self, mount_point, commit=None, path=None, as_dict=False, name_status=False):
        """
        Show file contents, content of path in commit.
        :param mount_point: path where arc mounted
        :param commit: commit hash
        :param path: path to file
        :param bool as_dict: True for parse json result and return dict
        :param bool name_status: True for show only names and status of changed files
        :return: dict if as_dict == True, str else.
        !ATTENTION! Output with as_dict == False is intended for human consumption and could be changed,
        but with as_dict == True result is guaranteed to stay compatible, e.g. no fields will be removed in future.
        """
        logger.info("Start arc show")
        command = [self.binary_path, "show"]
        if path is not None:
            if commit is not None:
                command.append("{}:{}".format(commit, path))
            else:
                command.append(path)
        else:
            if commit is not None:
                command.append(commit)
        if as_dict:
            command.append("--json")
        if name_status:
            command.append("--name-status")
        out = self._execute_command(mount_point=mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def info(self, mount_point):
        """
        Show current repo state
        """
        logger.info("Start arc info")
        command = [self.binary_path, "info", "--json"]
        out = self._execute_command(mount_point=mount_point, command=command)
        return json.loads(out)


    def rebase(self, mount_point, upstream=None, rebase_onto=None):
        """
        Reapply commits from the current branch on top of UPSTREAM
        :param mount_point: path where arc mounted
        :param upstream: remote branch name
        :param rebase_onto: rebase onto given branch instead of upstream
        :return: str, command output
        """
        logger.info("Start arc rebase")
        command = [self.binary_path, "rebase"]
        if upstream is not None:
            command.append(upstream)
        if rebase_onto is not None:
            command.extend(["--onto", rebase_onto])
        return self._execute_command(mount_point=mount_point, command=command)

    def create_zipatch(
        self,
        mount_point,
        arc_hash,
        output_path,
        svn_prefix="trunk/arcadia",
        svn_revision=None,
        no_copy=False,
        with_revprop=None,
    ):
        """
        Create zipatch for arc_hash.
        :param mount_point: path where arc mounted
        :param arc_hash: commit to build zipatch for
        :param svn_prefix: path to svn branch, where zipatch would be merged
        :param svn_revision: svn revision to use as source for copies
        :param output_path: path to store builded zipatch
        :param no_copy: generate zipatch without copy operations
        :param with_revprop: list with revprops for zipatch, ["key1=val1", "key2=val2"]
        :return: str, command output
        """
        logger.info("Start arc zipatch")
        command = [self.binary_path, "zipatch", "build", arc_hash]
        command.extend(["--svn-prefix", svn_prefix])
        command.extend(["--output", output_path])
        if svn_revision is not None:
            command.extend(["--svn-revision", str(svn_revision)])
        if no_copy:
            command.append("--no-copy")
        if with_revprop is not None:
            command.append("--revprop")
            command.extend(with_revprop)
        return self._execute_command(mount_point=mount_point, command=command)

    def cherry_pick(self, mount_point, commit=None, add_commit_name=False, allow_empty=False):
        """
        Cherry pick in arc, add some patches to current branch
        :param mount_point: path where arc mounted
        :param str commit: commit hash, etc. to cherry-pick
        (see https://a.yandex-team.ru/arc/trunk/arcadia/arc/docs/_includes/arc_rev-parse.md)
        :param bool add_commit_name: add to commit message `cherry picked from commit <base_commit_hash>`
        :param allow_empty: allow empty commits
        :return: str, command output
        """
        logger.info("Start arc cherry-pick")
        command = [self.binary_path, "cherry-pick"]
        if commit is not None:
            command.append(commit)
        if add_commit_name:
            command.append("-x")
        if allow_empty:
            command.append("--allow-empty")
        return self._execute_command(mount_point=mount_point, command=command)

    def delete_tag(self, mount_point, tag_name):
        """
        Delete tag
        :param mount_point: path where arc mounted
        :param str tag_name: tag to delete
        :return: str, command output
        """
        logger.info("Delete tag %s", tag_name)
        return self._tag(mount_point=mount_point, delete_tag=tag_name)

    def list_tags(self, mount_point, points_at=None):
        """
        List tags or print only tags of the object
        :param mount_point: path where arc mounted
        :param str points_at: commit hash, etc.
        :return: str, command output
        """
        logger.info("List tags")
        return self._tag(mount_point=mount_point, list=True, points_at_commit=points_at)

    def _tag(self, mount_point, points_at_commit=None, delete_tag=None, list=False):
        """
        Add, delete and show tags
        :param mount_point: path where arc mounted
        :param str points_at_commit: commit hash, print only tags of the object
        :param str delete_tag: tag to delete
        :param bool list: show all tags
        :return: str, command output
        """
        command = [self.binary_path, "tag"]
        if points_at_commit is not None:
            command.extend(["--points-at", points_at_commit])
        if delete_tag is not None:
            command.extend(["--delete", delete_tag])
        if list:
            command.append("--list")
        return self._execute_command(mount_point, command=command)

    def reset(self, mount_point, mode=ResetMode.MIXED, branch=None, path=None, force=False):
        """
        Change commit or stage-index.

        :param mount_point: path where arc mounted
        :param ResetMode mode: mode to use - soft, hard or mixed
        :param str branch: local branch name
        :param str path: path to file
        :param boolean force: force mode
        :return: str, command output
        """
        logger.info("Start arc reset")
        command = [self.binary_path, "reset"]
        command.append("--{}".format(mode))
        if branch is not None:
            command.append(branch)
        if path is not None:
            command.append(path)
        if force:
            command.append("--force")
        return self._execute_command(mount_point, command=command)

    def add(self, mount_point, path=None, all_changes=False):
        """
        Add changes to index.

        :param mount_point: path where arc mounted
        :param path: path to file for adding to index
        :param boolean all_changes: True for commit all changed files from working tree
        :return: str, command output
        """
        logger.info("Start file adding to arc index")
        command = [self.binary_path, "add"]
        if all_changes:
            command.append("--all")
        elif path is not None:
            command.append(path)
        else:
            raise ArcCommandFailed("Either path or all_changes=True must be specified")

        return self._execute_command(mount_point, command=command)

    def commit(self, mount_point, message, all_changes=False):
        """
        Commit all changes from index.

        :param mount_point: path where arc mounted
        :param str message: commit message
        :param boolean all_changes: True for commit all changed files from working tree
        :return: str, command output
        """
        logger.info("Start arc commit")
        command = [self.binary_path, "commit"]
        command.extend(["--message", message])
        if all_changes:
            command.append("--all")
        return self._execute_command(mount_point, command=command)

    def revert(self, mount_point, commit=None, allow_empty=False):
        """
        Revert some patches from current path.

        :param mount_point: path where arc mounted
        :param str commit: commit hash, etc. to revert
        :param allow_empty: allow empty commit
        :return: str, command output
        """
        logger.info("Start arc revert")
        command = [self.binary_path, "revert"]
        if commit is not None:
            command.append(commit)
        if allow_empty:
            command.append("--allow-empty")
        return self._execute_command(mount_point=mount_point, command=command)

    def log(
        self,
        mount_point,
        branch=None,
        path=None,
        start_commit=None,
        end_commit=None,
        max_count=0,
        author="",
        as_dict=False,
        walk=True,
        name_only=False,
    ):
        """
        Return log for current branch
        :param mount_point: path where arc mounted
        :param str branch: log from this branch, not checkouted branch
        :param str path: path to get log from
        :param int max_count: log length
        :param str author: filter commits with this author
        :param str start_commit: commit hash or HEAD, interval start
        :param str end_commit: commit hash or HEAD, interval end. Use only with commit_start
        :param bool as_dict: True for parse json result and return dict
        :param bool walk: show full history
        :param bool name_only: show only names of changed files
        :return: dict if as_dict == True, str else
        """
        logger.info("Get arc log")
        path_parts = []
        if start_commit is not None:
            if end_commit is not None:
                path_parts.append(
                    "{start_commit}..{end_commit}".format(
                        start_commit=start_commit,
                        end_commit=end_commit,
                    )
                )
            else:
                path_parts.append(str(start_commit))
        if branch is not None:
            path_parts.append(branch)
        if path is not None:
            path_parts.append(path)
        command = [self.binary_path, "log"]
        if path_parts:
            command.extend(path_parts)
        if max_count:
            command.extend(["--max-count", str(max_count)])
        if author:
            command.extend(["--author", str(author)])
        if as_dict:
            command.append("--json")
        if not walk:
            command.append("--no-walk")
        if name_only:
            command.append("--name-only")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def diff(self, mount_point, cached=False, diff_subject=None):
        """
        Get diff.

        Arc diff format is unstable. Do NOT rely on it in your code. It is meant to be read by a human, not a robot.
        Note also that there are absolutely no guarantees that the result can be used by any 'diff apply' instruments.

        :param mount_point: arc mount path
        :param cached: calculate diff with index
        :param diff_subject: "[PATH ...]", "COMMIT1 COMMIT2 [PATH ...]", "BRANCH1 BRANCH2 [PATH ...]"
        :return: diff result, str
        """
        logger.info("Get arc diff")

        command = [self.binary_path, "diff"]

        if cached:
            command.append("--cached")

        if diff_subject:
            command.append(str(diff_subject))

        return self._execute_command(mount_point, command=command)

    def branch(self, mount_point, branch=None, remote_branch=None, all=False, upstream=None, as_dict=False):
        """
        Manages branch
        :param mount_point: path where arc mounted
        :param str branch: local branch name
        :param str remote_branch: remote branch name
        :param bool all: True for add flag `--all`
        :param bool as_dict: True for parse json result and return dict
        :return: dict if as_dict == True, str else
        """
        logger.info("Start arc branch command")
        command = [self.binary_path, "branch"]
        if branch is not None:
            command.append(branch)
            if remote_branch is not None:
                command.append(remote_branch)
        if all:
            command.append("--all")
        if upstream:
            command.extend(["-u", upstream])
        if as_dict:
            command.append("--json")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def status(self, mount_point, as_dict=False):
        """
        Shows status
        :param mount_point: path where arc mounted
        :param bool as_dict: True for parse json result and return dict
        :return: dict if as_dict == True, str else
        """
        logger.info("Start arc status command")
        command = [self.binary_path, "status"]
        if as_dict:
            command.append("--json")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def get_merge_base(self, mount_point, base_branch="trunk", branch="HEAD"):
        """
        finds merge base for given branch by using command merge-base
        :param mount_point: path where arc mounted
        :param base_branch: base branch
        :param branch: target branch
        :return: commit
        """
        logger.info("Getting merge-base for %s %s", base_branch, branch)
        try:
            out = process.subprocess.check_output(
                [self.binary_path, "merge-base", base_branch, branch],
                cwd=mount_point,
            )
            return out.strip()
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
            raise ArcCommandFailed("unable to get arc merge-base for {} {}".format(base_branch, branch))

    def checkout(self, mount_point, branch="trunk", create_branch=False, start_point=None, track=False, force=False):
        """
        create (optionally) and checkout given branch
        :param mount_point: path where arc mounted
        :param branch: branch to checkout
        :param create_branch: whether to create branch or not
        :param start_point: point from where to start a new branch
        :param track: allows to keep upstream configuration
        :param force: throw away local modifications
        :return: str, command output
        """
        logger.info("Checkout branch %s", branch)
        command = [self.binary_path, "checkout"]
        if create_branch:
            command.append("-b")
        command.append(branch)
        if create_branch and start_point:
            command.append(start_point)
        if not track:
            command.append("--no-track")
        if force:
            command.append("-f")
        return self._execute_command(mount_point, command=command)

    def push(self, mount_point, upstream=None, force=False, refspecs=None):
        """
        push current branch
        :param mount_point: path where arc mounted
        :param upstream: remote branch name
        :param force: allows to do force push
        :param refspecs: list with tuples (commit, remote-ref) or just strings (commit or local branch/tag).
        Remote-ref should be trunk, branch or tag, more info here
        https://a.yandex-team.ru/arc/trunk/arcadia/arc/docs/_includes/arc_rev-parse.md
        :return: str, command output
        """
        logger.info("Push branch to {}".format(upstream))
        command = [self.binary_path, "push"]
        if upstream is not None:
            command.extend(["--set-upstream", upstream])
        if force:
            command.append("--force")
        if refspecs:
            for refspec in refspecs:
                if isinstance(refspec, tuple):
                    commit, reference = refspec
                    command.append("{commit}:{ref}".format(commit=commit, ref=reference))
                elif isinstance(refspec, str):
                    command.append(refspec)
                else:
                    logger.warning("Got strange element in refspecs, not tuple or string: %s", refspec)
        return self._execute_command(mount_point, command=command)

    def pr_create(
        self, mount_point, to=None, message=None, publish=False, auto=False,
        no_commits=False, no_edit=False, no_push=False, as_dict=False
    ):
        """
        create pr from current branch
        :param mount_point: path where arc mounted
        :param to: to branch
        :param message: pr message
        :param publish: publish pr
        :param auto: merge PR automatically when checks are passed
        if you want to push a branch when creating a PR, use the push method separately, specifying the full upstream
        :return: str, command output if :param as_dict: is False, dict otherwise
        """
        logger.info("Create pr to {}".format(to if to else "trunk"))
        command = [self.binary_path, "pr", "create"]
        if to:
            command.extend(["--to", to])
        if message:
            command.extend(["--message", message])
        if publish:
            command.append("--publish")
        if auto:
            command.append("--auto")
        if no_commits:
            command.append("--no-commits")
        if no_edit:
            command.append("--no-edit")
        if no_push:
            command.append("--no-push")
        if as_dict:
            command.append("--json")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def pr_checkout(self, mount_point, id, iteration=None, published=False):
        """
        Checkouts PR by ID
        :param mount_point: path where arc mounted
        :param id: PR ID
        :param iteration: iteration of the review
        :param published: accept only published iterations
        :return: str, command output
        """
        logger.info("Checkout PR {}".format(id))
        command = [self.binary_path, "pr", "checkout", str(id)]
        if iteration:
            command.extend(["--iteration", iteration])
        if published:
            command.append(["--published"])
        return self._execute_command(mount_point, command=command)

    def pr_status(self, mount_point, pr=None, as_dict=False):
        """
        Requests pr status for current branch or by id
        :param mount_point: path where arc mounted
        :param pr: pr id
        :param as_dict: return parsed result as dict
        :return: command output, str if :param as_dict: is False, dict otherwise
        """
        logger.info("Requesting pr status")
        command = [self.binary_path, "pr", "status"]
        if pr:
            command.append(str(pr))
        if as_dict:
            command.append("--json")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def pr_discard(self, mount_point, pr=None):
        """
        Discards pr for current branch or by id
        :param mount_point: path where arc mounted
        :param pr: pr id
        :return: command output, str
        """
        logger.info("Discarding pr status")
        command = [self.binary_path, "pr", "discard"]
        if pr:
            command.append(str(pr))
        return self._execute_command(mount_point, command=command)

    def user_info(self, mount_point, as_dict=False):
        """
        Show information about current user
        :param mount_point: path where arc mounted
        :param as_dict: return parsed result as dict
        :return: command output, str if :param as_dict: is False, dict otherwise
        """
        logger.info("Start arc user-info command")
        command = [self.binary_path, "user-info"]
        if as_dict:
            command.append("--json")
        out = self._execute_command(mount_point, command=command)
        if not as_dict:
            return out
        return json.loads(out)

    def _is_commit(self, mp, rev):
        """ checks whether the given rev is a commit or not.
        Runs `arc dump object <rev>` and parses the first line to perform the
        check.
        Returns True if it's a commit and False otherwise.
        Does not unmounts the repository.
        """
        assert mp.mounted

        logger.info("Checking whether %s is a commit or not", rev)

        if re.match(r'^r\d+$', rev):
            # TODO ARC-3720
            logger.info('Commits with r-notation (revisions) are always considered valid commits')
            return True

        try:
            out = process.subprocess.check_output(
                [self.binary_path, "dump", "object", rev],
                cwd=mp._mount_point,
                stderr=process.subprocess.STDOUT
            )
            if out.startswith(b"commit"):
                return True
            return False
        except process.subprocess.CalledProcessError as exc:
            logger.error("Command returned non-zero exit code %d, output '%s'", exc.returncode, exc.output)
            raise

    def _try_checkout_rev(self, mp, rev, timeout=600):
        """ Tries to checkout the rev provided by the user after mounting an Arc
        repository.
        If the rev does not exists, unmounts the repository and raise error.
        Otherwise, checkouts and returns.

        mp is an object of util.MountPoint class
        """
        assert mp.mounted

        try:
            process.subprocess.check_call([self.binary_path, "fetch", "--verbose", rev], cwd=mp._mount_point)
        except process.subprocess.CalledProcessError as e:
            logger.debug(e)

        assert self._is_commit(mp, rev), "{} is not a commit".format(rev)

        logger.info("Checking out rev: %s", rev)

        last_error = {"content": None}

        def safe_checkout():
            try:
                process.subprocess.check_call([self.binary_path, "checkout", rev, "--force"], cwd=mp._mount_point)
                return True
            except process.subprocess.CalledProcessError as e:
                last_error["content"] = e
            return False

        is_checkout_done, elapsed_time = common_itertools.progressive_waiter(0, 10, timeout, safe_checkout)

        if not is_checkout_done:
            logger.error(last_error["content"])
            mp.unmount()
            raise ArcNoSuchRevision("Unable to checkout {}. Waited {}s.".format(rev, elapsed_time))

    def _exists_rev(self, mp, rev):
        """ runs `arc show rev` on a mounted arc repository and returns a
        boolean indicating whether the rev exists or not
        Does not perform unmount if the rev does not exists
        """
        assert mp.mounted

        logger.info("Checking for rev: %s", rev)
        try:
            return self._is_commit(mp, rev)
        except process.subprocess.CalledProcessError as e:
            logger.error(e)
            return False

    def _prepare_command(
        self,
        mount_path=None,
        store_path=None,
        allow_root=False,
        frozen=False,
        full=False,
        object_store_path=None,
        minimize_mount_path=False,
        extra_params=[],
    ):
        """
        Mounts an Arcadia repository, uses Arc VCS to work with it as with SVN repo.
        :param mount_path:          Path where repository should be placed
        :param store_path:          Path to arc store
        :param allow_root:          Same as allow_root for 'arc mount'
        :param frozen:              Same as frozen for 'arc mount'
        :param full:                To mount full repository instead of 'arcadia'
        :param object_store_path    Path to shared object store
        :param minimize_mount_path  Mount to temp directory with the shortest mount prefix possible
        :param extra_params         List of custom parameters for mount command
        :return:                    Command line that should be run to mount repository
        """
        if not mount_path:
            if minimize_mount_path:
                mount_path = util.create_minimized_mount_path()
            else:
                mount_path = str(
                    sdk2.task.Task.current.path(agentr_types.FUSE_DIRNAME) / ("mount_path_" + str(uuid.uuid4()))
                )
        fs.make_folder(mount_path)
        if not store_path:
            store_path = str(
                sdk2.task.Task.current.path(agentr_types.FUSE_DIRNAME) / ("store_path_" + str(uuid.uuid4()))
            )

        windows = platform.system() == "Windows"
        mount_cmd = [self.binary_path, "mount", "--mount", mount_path, "--foreground", "--fetch-trees", "0", "--disable-mount-manager", "1", "--override-lazy-checkout", "0", "--no-auto-unmount"]
        if not windows:
            # we use shared object store on unix platforms
            object_store_path = self._disc_store() if (object_store_path is None) else object_store_path
            mount_cmd.extend(["--store", store_path, "--object-store", object_store_path])
            if allow_root:
                mount_cmd.append("--allow-root")
        if full:
            mount_cmd.extend(["--repository", "full"])

        if frozen:
            mount_cmd.extend(["--frozen"])

        if extra_params:
            mount_cmd.extend(extra_params)

        return mount_cmd, mount_path, store_path

    def _ensure_some_token_in_env(self):
        """ checks that ARC_TOKEN environment variable has some value, throws
        ArcCommandFailed if it is not available or empty
        """
        token = os.environ.get("ARC_TOKEN", None)
        if token is None or token == "":
            raise ArcCommandFailed((
                "ARC_TOKEN is not available. It can be provided with either:\n"
                "1. `arc_secret` task input parameter. It is available for common YA_MAKE-family tasks.\n"
                "   If you use some derived task, you should probably add sandbox.projects.common.build.parameters.ArcSecret() to its own parameters.\n"
                "2. Sandbox vault. Add `ARC_TOKEN` key for owner of tasks with value of token.\n"
                "\n"
                "For more info about acquiring ARC_TOKEN and other authorization issues, see https://docs.yandex-team.ru/devtools/intro/auth\n"
            ))

    def init_bare(self, repo_path=None, object_store_path=None, server=None):
        """
        Initializes bare arc repo to work with.
        :param repo_path:         Path where repository should be placed
        :param object_store_path: Path where objects should be stored. Shared storage used by default
        :return:                  ArcBareRepo context manager
        """

        init_cmd = [self.binary_path, "init", "--bare"]
        if platform.system() != "Windows":
            # we use shared object store on unix platforms
            object_store_path = self._disc_store() if (object_store_path is None) else object_store_path
            init_cmd.extend(["--object-store", object_store_path])

        if server:
            init_cmd.extend(["--server", server])

        self._ensure_some_token_in_env()

        return ArcBareRepo(init_cmd, repo_path)

    def mount_svn_path(
        self,
        svn_path,
        mount_path=None,
        store_path=None,
        allow_root=False,
        revision=None,
        checkout_timeout=600,
        minimize_mount_path=False,
        extra_params=[],
    ):
        """
        Mounts an Arcadia repository use Arc VCS to work with it as with SVN repo.
        :param svn_path:            Path in svn repository that should be considered as root
        :param mount_path:          Path where repository should be placed
        :param store_path:          Path to arc store
        :param allow_root:          Same as allow_root for 'arc mount'
        :param revision:            Revision number that should be mounted (if None, then just mount last revision)
        :param minimize_mount_path  Mount to temp directory with the shortest mount prefix possible
        :param extra_params         List of custom parameters for mount command
        :return:                    MountPoint context manager
        """
        sys_info = util.get_system_info()
        if not sys_info["fuse_available"]:
            raise ArcCommandFailed("Fuse is not available on current host")

        mount_cmd, mount_path, store_path = self._prepare_command(
            mount_path,
            store_path,
            allow_root,
            frozen=False,
            full=True,
            minimize_mount_path=minimize_mount_path,
            extra_params=extra_params,
        )

        self._ensure_some_token_in_env()

        prepared_svn_path = svn_path.lstrip("/")

        mp = util.MountPoint(
            mount_path,
            mount_cmd,
            util.ARC_SVN,
            ArcCommandFailed,
            relative_work_path=prepared_svn_path
        )
        mp.mount()

        if revision is not None:
            self._try_checkout_rev(mp, "r{}".format(revision), timeout=checkout_timeout)

        if not os.path.exists(os.path.join(mount_path, prepared_svn_path)):
            raise ArcCommandFailed("there is no path '{}' in revision {}".format(prepared_svn_path, revision))

        # Preparing __SVNVERSION__ file
        try:
            out = process.subprocess.check_output([self.binary_path, "show", "HEAD", "--json"], cwd=mount_path)
            data = json.loads(out)[0]["commits"][0]
            svnversion = {}
            svnversion["author"] = data["author"]
            svnversion["date"] = data["date"]
            svnversion["repository"] = "svn://arcadia.yandex.ru/arc/" + prepared_svn_path
            svnversion["repository_vcs"] = "subversion"

            # Since the repository is mounted by revision or by the current trunk,
            # the parameter 'revision' always exists
            svnversion["revision"] = data["revision"]
            svnversion["last_revision"] = data["revision"]
            with open(os.path.join(mount_path, prepared_svn_path, "__SVNVERSION__"), "w") as svnversion_file:
                json.dump(svnversion, svnversion_file)
        except:
            exc = sys.exc_info()
            msg = "Can't create __SVNVERSION__ file: {}".format(exc[1])
            logger.exception(msg)
            six.reraise(ArcCommandFailed, msg, exc[2])

        return mp

    def mount_path(
        self,
        path,
        changeset,
        mount_point=None,
        store_path=None,
        allow_root=False,
        object_store_path=None,
        fetch_all=False,
        minimize_mount_path=False,
        extra_params=[],
    ):
        """ Mounts an Arcadia repository use Arc VCS.

        First mounts the repository, then try to checkout the changeset
        provided. If the changeset does not exists, unmounts the repository and
        raise error.
        If the changeset exists, returns a object of util.MountPoint class which
        can be used as a Context Manager. Since the mount was already performed,
        that object will have `mounted = true`.

        changeset can be passed as None to only mount a repo and not checkout
        any changeset

        mount_point and store_path are optional parameters which can be used to
        control where the repo is mounted and store is stored.

        If object_store_path is omitted, shared object store will be used. Ingored on windows platforms.

        extra_params can be used to add any custom parameters to mount command

        minimize_mount_paths allows to to minimize mount point prefix length

        Note: path is unused because Arc does not provide functionality to mount
        a specific path yet
        """
        sys_info = util.get_system_info()
        if not sys_info["fuse_available"]:
            raise ArcCommandFailed("Fuse is not available on current host")

        if path:
            raise ArcCommandFailed("{} provided as path however Arc does support partial checkout yet".format(path))

        mount_cmd, mount_path, store_path = self._prepare_command(
            mount_point,
            store_path,
            allow_root,
            object_store_path=object_store_path,
            minimize_mount_path=minimize_mount_path,
            extra_params=extra_params,
        )
        self._ensure_some_token_in_env()
        mp = util.MountPoint(mount_path, mount_cmd, util.ARC_VCS, ArcCommandFailed)
        if platform.system() != "Windows":
            mp.set_store_path(store_path)

        mp.mount()

        if fetch_all:
            logger.warning("fetch_all=True is no longer supported and should not be needed")
        if changeset:
            self._try_checkout_rev(mp, changeset)

        return mp
