""" Utility functions for vcs related handling in Sandbox """

from __future__ import absolute_import

import itertools
import json
import os
import time
import psutil
import logging
import platform
import subprocess
import tempfile

from six.moves.urllib import parse as urlparse

import sandbox.common.threading
from sandbox.common import fs
from sandbox.common import itertools as common_itertools

from sandbox.sdk2 import paths
from sandbox.sdk2.helpers import process
from sandbox.sdk2.vcs import svn

logger = logging.getLogger("vcs")

AAPI_HG = "AAPI_HG"
ARC_VCS = "ARC_VCS"

SVN = "SVN"
AAPI_SVN = "AAPI_SVN"
ARC_SVN = "ARC_SVN"

AAPI = (AAPI_HG, AAPI_SVN)
ARC = (ARC_VCS, ARC_SVN)


def get_system_info():
    def check_linux_fuse():
        try:
            rc = process.subprocess.call(["fusermount", "--version"])
        except OSError as e:
            logger.error("Run fusermount failed: %s", e)
            return False

        if rc != 0:
            logger.error("Run fusermount failed: rc=%d", rc)
            return False

        if not os.path.exists("/dev/fuse"):
            logger.error("/dev/fuse does not exist")
            return False

        return True

    res = {
        "os": platform.system(),
        "fuse_available": False,
    }

    if res["os"] == "Darwin":
        # To use fuse on Mac, osxfuse/macfuse is required
        # this checks whether it's installed or not
        res["osxfuse"] = os.path.exists("/Library/Filesystems/osxfuse.fs/Contents/Resources/mount_osxfuse")
        res["macfuse"] = os.path.exists("/Library/Filesystems/macfuse.fs/Contents/Resources/mount_macfuse")
        logging.debug("Trying to find FUSE darwin kernel extension: osxfuse=%s macfuse=%s",
                      res["osxfuse"], res["macfuse"])
        res['fuse_available'] = res["osxfuse"] or res["macfuse"]
    elif res['os'] == "Windows":
        winsxs_content = os.listdir(os.path.join(os.environ["WINDIR"], "WinSxS"))
        logging.debug("Trying to find ProjFS in %s", os.path.join(os.environ["WINDIR"], "WinSxS"))
        res["projfs"] = False
        for entry in winsxs_content:
            if "projfs" in entry:
                logging.debug("ProjFS found in %s", os.path.join(os.environ["WINDIR"], "WinSxS"))
                res["projfs"] = True
                res["fuse_available"] = True
                break
    else:
        if check_linux_fuse():
            res["fuse_available"] = True

    logger.debug("FUSE-related system information: %s", json.dumps(res, indent=2))

    return res


def fuse_available():
    return get_system_info().get("fuse_available")


class MountPoint(object):
    """ Context manager to mount Arcadia repository """

    def __init__(self, mount_point, mount_cmd, mount_type, error_on_fail, relative_work_path=""):
        """
        @param mount_point         path where Arcadia should be mounted
        @param mount_cmd           command to be used to mount
        @param mount_type          the kind of mount we are performing, AAPI_{HG|SVN} or ARC
        @param error_on_fail       error to raise when the mount fails
        @param relative_work_path  path in repository that will be considered the root path
        """
        self._mount_point = mount_point
        self._work_path = os.path.join(mount_point, relative_work_path)
        self._mount_cmd = mount_cmd
        self._mount_process_pid = None
        assert mount_type in (AAPI_HG, AAPI_SVN, ARC_VCS, ARC_SVN)
        self.mount_type = mount_type
        self.error = error_on_fail
        # whether mount is already done
        self.mounted = False
        # path to store of mount path, in case we are using a different store per mount
        self.store_path = None
        self.log_file_name = None
        self.log_file = None

    def set_store_path(self, path):
        self.store_path = path

    def _check_mounted(self):
        if self.mount_type in AAPI:
            # fuse client now sets initial root's mtime to zero
            return os.stat(self._mount_point).st_mtime == 0
        elif self.mount_type in ARC:
            system = platform.system()
            if system == "Linux" or system == "Darwin":
                return os.path.isdir(os.path.join(self._mount_point, ".arc"))
            elif system == "Windows":
                time.sleep(1)  # ProjFS is not so fast
                check_cmd = [self._mount_cmd[0], "info", "--json"]
                try:
                    process.subprocess.check_call(check_cmd, cwd=self._mount_point)
                    return True
                except process.subprocess.CalledProcessError:
                    return False
        else:
            raise self.error("Unknown mount type: {}".format(self.mount_type))

    def _close_log(self):
        if self.log_file is not None:
            self.log_file.close()
            self.log_file = None

    def _popen_with_log(self, cmd):
        if self.log_file is not None:
            return process.subprocess.Popen(
                cmd,
                stderr=process.subprocess.STDOUT,
                stdout=self.log_file
            )
        else:
            return process.subprocess.Popen(cmd)

    def mount(self):
        fs.make_folder(self._mount_point)

        logs_folder = paths.get_logs_folder()
        self.log_file_name = fs.get_unique_file_name(logs_folder, "arc-daemon.log")
        self.log_file = open(self.log_file_name, 'w')

        mount_process = self._popen_with_log(self._mount_cmd)
        self._mount_process_pid = mount_process.pid

        while True:
            if mount_process.poll() is None:
                if self._check_mounted():
                    self.mounted = True
                    return
                else:
                    time.sleep(0.5)
                    continue
            else:
                self.unmount()
                self._close_log()
                with open(self.log_file_name) as fh:
                    last_errors = fs.tail(fh, 10)
                    raise self.error("Mount failed:\n{}".format("\n".join(last_errors)))

    def _unmount_arc(self, timeout):
        if self.mount_type in ARC:
            logger.debug("Unmounting with arc unmount")
            cmd = [self._mount_cmd[0], "unmount", self._mount_point]
            try:
                returncode = self._popen_with_log(cmd).wait(timeout=timeout)
                logger.debug("Command '%s' exited with code: %d", " ".join(cmd), returncode)
                return returncode
            except subprocess.TimeoutExpired as e:
                logger.error("arc unmount has not finished in %d seconds: %s", timeout, e)
                return -1
        else:
            return -1

    def _unmount_system(self, timeout):
        _platform = platform.system()
        if _platform == "Windows":
            return -1
        elif _platform == "Darwin":
            cmd = ["umount", "-f"]
        else:
            cmd = ["fusermount", "-uz"]
        cmd.append(self._mount_point)

        logger.debug("Unmounting with system command '%s'", " ".join(cmd))
        try:
            returncode = self._popen_with_log(cmd).wait(timeout=timeout)
            logger.debug("Command '%s' exited with code: %d", " ".join(cmd), returncode)
            return returncode
        except subprocess.TimeoutExpired as e:
            logger.error("system unmount has not finished in %d seconds: %s", timeout, e)
            return -1


    def unmount(self):
        try:
            if self._mount_process_pid is None:
                logger.debug("Mount process is not started")
                return

            p = psutil.Process(self._mount_process_pid)
            p.terminate()
            p.wait(timeout=30)

        except psutil.TimeoutExpired as e:
            logger.error(
                "Mount process %s didn't terminate in 30 seconds after SIGTERM: %s. Sending SIGKILL",
                self._mount_process_pid,
                e
            )

            try:
                p = psutil.Process(self._mount_process_pid)
                p.kill()
            except psutil.Error as e:
                logger.error("Mount process %s send SIGKILL failed: %s", self._mount_process_pid, e)

        except psutil.Error as e:
            logger.error("Mount process %s send SIGTERM failed: %s", self._mount_process_pid, e)

        finally:
            # killed process might have left dead mountpoint, unmount it with system command
            ret = self._unmount_system(timeout=None)
            if ret != 0:
                ret = self._unmount_arc(timeout=30)

            self._close_log()

            def retry_remove(path):
                last_error = {"content": None}
                path = os.path.abspath(str(path))

                def safe_remove_path():
                    try:
                        _platform = platform.system()
                        if _platform == "Linux":
                            logger.info("Remove directory %s", path)
                            if os.path.exists(path):
                                rm_process = process.subprocess.Popen(["rm", "-rf", path])
                                try:
                                    if rm_process.wait(timeout=300) == 0:
                                        return True
                                except process.subprocess.TimeoutExpired:
                                    rm_process.kill()
                                    rm_process.wait()
                                    with process.ProcessLog(logger="vcs_cleanup_fs_dump") as log:
                                        maxdepth = 7
                                        logger.debug(
                                            "Dump files tree (maxdepth=%d) for '%s' before cleanup to '%s'",
                                            maxdepth, path, log.stdout.path.name
                                        )
                                        process.subprocess.call(
                                            ["find", path, "-maxdepth", str(maxdepth), "-printf", "%M %p %s\\n"],
                                            stdout=log.stdout, stderr=log.stderr
                                        )
                                    raise OSError("Failed to execute 'rm -rf' on {}: timed out".format(path))
                                raise OSError(
                                    "Failed to execute 'rm -rf' on {}, exited with code: {}".format(
                                        path, rm_process.returncode
                                    )
                                )
                            else:
                                logger.info("Path does not exist %s, cannot remove it", path)
                        else:
                            fs.remove_path(path)
                        return True
                    except (OSError, subprocess.CalledProcessError) as e:
                        last_error["content"] = e
                    return False
                is_remove_done, _ = common_itertools.progressive_waiter(0, 1, 10, safe_remove_path)
                if is_remove_done:
                    return
                logger.error("Can not remove path: %s error: %s", path, str(last_error["content"]))

            retry_remove(self._mount_point)
            self.mounted = False
            if self.store_path:
                retry_remove(self.store_path)

    def __enter__(self):
        if not self.mounted:
            # Arc must have already mounted the repository to check whether the
            # revision provided by user exists or not. So, check for that before
            # mounting when the context manager is entered
            self.mount()
        return self._work_path

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.unmount()

    def __del__(self):
        if self.mounted:
            # In case of Arc, we perform the mount outside of the context
            # manager, so if the context manager is not used because of the
            # task being stopped in between or some other reasons, unmount does
            # not happen. Let's make sure when the object is destructed, we
            # don't leave the repo mounted
            self.unmount()


def extract_path_details(url):
    """
    Extract details from the url which is provided by user as argument to mount_arc_path().

    Returns (path, commit_id, type) where:
        path: the path we need to mount
        commit_id: id of commit which we should be mounted
        type: a constant describing whether it's a url of ARC_{VCS|SVN} or AAPI_{HG|SVN} type

    Examples:
    """
    normalized_url = svn.Arcadia.normalize_url(url)
    logger.debug("Normalized url %s into %s", url, normalized_url)

    p = urlparse.urlparse(normalized_url)

    if p.scheme == svn.Arcadia.ARCADIA_HG_SCHEME:
        path = p.path.strip("/")

        if "@" in path or not p.fragment:
            # TODO: this was ArcadiaApiUnexpectedInput exception, make it that
            # or a similar error
            raise ValueError("Unexpected arcadia-hg url: {}".format(url))

        return path, p.fragment, AAPI_HG

    elif p.scheme == svn.Arcadia.ARCADIA_SCHEME:
        path = p.path.lstrip("/")

        if path.startswith("arc"):
            path = path[3:]

        path = "/" + path.lstrip("/")

        if "@" in path:
            path, revision = path.rsplit("@", 1)

            try:
                if revision.upper() == "HEAD":
                    return path, None, SVN

                if revision[0] == "r":
                    revision = revision[1:]

                revision = int(revision)
            except (TypeError, ValueError):
                logger.error("Can't parse url %s", url)
                return None, None, None

            return path, revision, SVN

        return path, None, SVN

    elif p.scheme == svn.Arcadia.ARCADIA_ARC_SCHEME:
        path = p.path.strip("/")

        if "@" in path or not p.fragment:
            # TODO: this was ArcadiaApiUnexpectedInput exception, make it that
            # or a similar error
            raise ValueError("Unexpected arcadia-arc url: {}".format(url))

        return path, p.fragment, ARC_VCS

    else:
        logger.error("Unknown url format %s", url)
        return None, None, None


def create_minimized_mount_path():
    # Sandbox overrides tempdir for each task while keeping it short enough
    prefix = tempfile.gettempdir()
    assert prefix is not None, "Cannot acquire temp directory"
    # Flock provides some safety against weird mount point generation usage
    with sandbox.common.threading.FLock(os.path.join(prefix, 'm.lock')):
        for n in itertools.count():
            mount_path = os.path.join(prefix, 'm' + str(n))
            if not os.path.exists(mount_path):
                fs.make_folder(mount_path)
                logger.info('Created minimized mount path: %s', mount_path)
                return mount_path
