# coding: utf-8
import logging

from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.channel import channel


def get_current_production_task_id(service, resource_type, nanny_oauth_token):
    """
        Получить для данного типа сэндбокс-ресурса таск,
        который его создал и который сейчас живёт в продакшене

        Note:
            Если в качестве resource_type будет передан список ресурсов,
            то будет возвращен task_id для первого найденного ресурса из списка

        :param service: название сервиса в няне
        :param resource_type: тип или список типов ресурсов сэндбокса
        :return: id таска, ресурс которого сейчас используется в продакшене
    """
    resource_types = resource_type if isinstance(resource_type, (list, tuple)) else [resource_type]

    nanny_client = nanny.NannyClient(
        api_url='http://nanny.yandex-team.ru/',
        oauth_token=nanny_oauth_token,
    )
    json = nanny_client.get_service_resources(service)
    try:
        for files in json['content']['sandbox_files']:
            if files['resource_type'] in resource_types:
                return files['task_id']
    except KeyError as err:
        logging.error('Error loading current production task for resource {} and service {}. Error: {}'.format(resource_type, service, str(err)))
        raise


def get_resource_id_from_task(task_id, resource_type):
    """
        Получить по данному сэндбокс-таску ресурс соответствующего типа

        :param task_id: id таска
        :param resource_type: тип ресурса сэндбокс
        :return: id ресурса
    """
    resources = channel.sandbox.list_resources(resource_type=resource_type, omit_failed=True, task_id=task_id)
    if resources:
        return resources[0].id


# TODO: change for better way finding production resources
def get_current_production_resource_id(service, resource_type, nanny_oauth_token):
    """
        Получить для данного типа сэндбокс-ресурса номер ресурса,
        который сейчас живёт в продакшене

        :param service: название сервиса в няне
        :param resource_type: тип ресурса сэндбокс
        :return: id ресурса, который сейчас используется в продакшене
    """

    task_id = get_current_production_task_id(service, resource_type, nanny_oauth_token)
    return get_resource_id_from_task(task_id, resource_type)
