#! /usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import print_function

import sys


class CgiParamLogSplitter:
    """
        Класс для записи логов запросов в файлы.
        Умеет разделять запросы по наборам cgi-параметров
        и записывать их в разные файлы.
    """

    # (split params, real arguments) or None -> SplittedFile
    splitted_files = {}
    base_name = ""
    max_requests = None
    split_params = [None]  # Params to split log by. There might be one param or list of them

    class SplittedFile:
        def __init__(self, base_name, param_and_arg, max_requests=None):
            self.name = self.calc_name(base_name, param_and_arg)
            self.output_file = open(self.name, 'w')
            self.max_requests = max_requests
            self.requests_written = 0

        def write(self, request):
            if not self.output_file:
                raise Exception('File is closed')
            if self.max_requests and self.requests_written >= self.max_requests:
                return False
            self.output_file.write(request + '\n')
            self.requests_written += 1
            return True

        def calc_name(self, base_name, param_and_arg):
            if not param_and_arg:
                return base_name + '.txt'

            param, arg = param_and_arg
            if isinstance(param, list):
                if not isinstance(arg, list):
                    raise Exception('param_and_arg var type mismatching: ' + str(param_and_arg))
                if len(param) > len(arg):
                    raise Exception('len of param_and_arg var is wrong: ' + str(param_and_arg))
            else:
                param = [param]
                arg = [arg]

            # name
            name = base_name
            for i in range(len(param)):
                name += '.{}={}'.format(self.normalize_name(param[i]), self.normalize_name(arg[i]))
            name += '.txt'
            return name

        def normalize_name(self, name):
            if name:
                return name.replace('/', '_').replace(':', '_').replace('\n', '').replace('\t', ' ')

        def close(self):
            if self.output_file:
                self.output_file.close()
                self.output_file = None

    def __init__(self, base_name, split_params=[None], max_requests=None):
        self.base_name = base_name
        self.max_requests = max_requests
        self.split_params = split_params

    def process_request(self, request):
        written_some = False
        for split_param in self.split_params:
            written = self.splitted_file(split_param, request).write(request)
            if written:
                written_some = True

        return written_some

    def get_file_names(self):
        ret = []
        for _, f in self.splitted_files.items():
            ret.append(f.name)
        return ret

    def get_file_names_with_lines_count(self):
        ret = []
        for _, f in self.splitted_files.items():
            ret.append((f.name, f.requests_written))
        return ret

    def close(self):
        for _, f in self.splitted_files.items():
            f.close()

    def get_param_value(self, param, request):
        beg = request.find('&{}='.format(param))
        if beg == -1:
            beg = request.find('?{}='.format(param))
        if beg == -1:
            return
        beg = beg + len(param) + 2
        end1 = request.find('&', beg)
        end2 = request.find('#', beg)
        if end1 == -1:
            end = end2
        elif end2 == -1:
            end = end1
        else:
            end = min(end1, end2)
        ret = request[beg:] if end == -1 else request[beg:end]
        if ret:
            return ret

    def splitted_file(self, split_param, request):
        if not split_param:
            param_and_arg = None
        elif isinstance(split_param, list):
            param_and_arg = (split_param, [self.get_param_value(p) for p in split_param])
        else:
            param_and_arg = (split_param, self.get_param_value(split_param, request))
        f = self.splitted_files.get(param_and_arg)
        if not f:
            self.splitted_files[param_and_arg] = self.SplittedFile(self.base_name, param_and_arg, self.max_requests)
            f = self.splitted_files.get(param_and_arg)
        return f


if __name__ == '__main__':
    params = [None if p == 'None' else p for p in sys.argv[1:]]
    splitter = CgiParamLogSplitter('log', params)
    for line in sys.stdin:
        if line[-1] == '\n':
            line = line[:-1]
        splitter.process_request(line)

    print('File names are:')
    for name in splitter.get_file_names():
        print(name)

    splitter.close()
