# coding: utf-8

import logging
import os
import six
import time
import urllib2
import urlparse
import psutil
import threading
import ast

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder, add_write_permissions_for_path
from sandbox.sandboxsdk import process

from sandbox.projects.common.wizard.wizard_builder import WizardBuilder
from sandbox.projects.common.wizard.utils import check_wizard_build
from sandbox.projects.common.wizard.exception_manager import ExceptionManager

from sandbox.projects.common.utils import wait_searcher_start, wait_searcher_stop


class WizardConfigType(object):
    def __init__(self, conf_type):
        self.type = conf_type

    @staticmethod
    def classic():
        return WizardConfigType('classic')

    @staticmethod
    def geo():
        return WizardConfigType('geo')


def construct_wizard_provider(wizard_build, wizard_runtime):
    if not (isinstance(wizard_build, six.string_types) or isinstance(wizard_build, int)):
        raise SandboxTaskFailureError('Incorrect argument type: {}'.format(wizard_build))
    try:
        int(wizard_build)
        return WizardProvider(wizard_build, wizard_runtime)
    except ValueError:
        return RemoteWizardProvider(wizard_build)


class FakeWizardProvider(object):
    def __init__(self, *fake_params):
        self.port = 0
        logging.info('FakeWizardProvider created with params: {}'.format(fake_params))

    def __enter__(self):
        pass

    def __exit__(self, type, value, traceback):
        pass


class RemoteWizardProvider(object):
    _default_port = 8891
    _error_tpl = 'Incorrect wizard url: {}; Example: "host:port (8891 default)"'

    def __init__(self, *params):
        self._check_params(params)
        self.host, self.port = self._parse_url(params[0])
        logging.info('RemoteWizardProvider address: {wizard.host}:{wizard.port}'.format(wizard=self))

    def alive(self):
        return True

    def _check_params(self, params):
        if len(params) != 1:
            raise SandboxTaskFailureError('Remote wizard takes only one argument - it\'s address (host:port)')

    def _parse_url(self, url):
        address = urlparse.urlsplit(url).path
        if not address:
            raise SandboxTaskFailureError(self._error_tpl.format(url))

        splitted_address = address.split(':')
        if len(splitted_address) == 1:
            host = address
            port = self._default_port
        elif len(splitted_address) == 2:
            host, port = splitted_address
        else:
            raise SandboxTaskFailureError(self._error_tpl.format(url))

        return host, port

    def __getattribute__(self, attr):
        try:
            return object.__getattribute__(self, attr)
        except AttributeError:
            return self

    def __call__(self, *args, **kwargs):
        pass

    def __enter__(self):
        return self

    def __exit__(self, type, value, traceback):
        return self


class WizardProvider(object):
    port = 8891
    host = 'localhost'

    def __init__(
        self, build_wizard_id, wizard_runtime_id=None, config_type=WizardConfigType.classic(), no_cache=True,
        stderr=None, work_with_runtime=True, read_only_resources=False
    ):
        check_wizard_build(build_wizard_id)
        self.work_with_runtime = work_with_runtime
        self.wizard = build_wizard_id
        self.wizard_runtime_id = wizard_runtime_id

        self.timestamp = time.time()

        self.data_path = make_folder('wizard_data_{}'.format(self.timestamp))
        self.binary_path = None
        self.shard_path = None
        self.runtime_path = None
        self.config_path = None

        self.config_type = config_type.type
        self.no_cache = no_cache
        self.eventlog_name = 'wizard_eventlog_{}'.format(self.timestamp)
        self.process = None
        self.stderr = stderr
        self.read_only_resources = read_only_resources

        self.sync_resources()
        self.cmd = self.get_run_cmd()

    def get_stat(self):
        stat_url = 'http://{wizard.host}:{wizard.port}/wizard?action=rulestat'.format(wizard=self)
        return urllib2.urlopen(stat_url).read()

    def get_version(self):
        version_url = 'http://{wizard.host}:{wizard.port}/wizard?action=version'.format(wizard=self)
        return urllib2.urlopen(version_url).read()

    def get_exceptions(self):
        exceptions_url = 'http://{wizard.host}:{wizard.port}/wizard?action=exceptions'.format(wizard=self)
        exceptions_data = urllib2.urlopen(exceptions_url).read()
        if not exceptions_data or exceptions_data == 'null\n':
            return []
        logging.info('parsing exceptions data: {}'.format(exceptions_data))
        exceptions = ast.literal_eval(exceptions_data)
        exceptions_500 = [e for e in exceptions if e['code'] == '500']
        return exceptions_500

    def get_memory(self):
        return psutil.Process(self.process.pid).get_memory_info().rss

    def get_wizard_binary(self):
        wizard_bin_id = WizardBuilder.wizard_from_task(self.wizard)
        logging.info('syncing wizard binary resource #{}'.format(wizard_bin_id))
        self.binary_path = channel.task.sync_resource(wizard_bin_id)
        logging.info('synced wizard binary {}'.format(self.binary_path))

    def get_wizard_shard(self):
        shard_id = WizardBuilder.wizard_shard_from_task(self.wizard)
        logging.info('syncing wizard shard resource #{}'.format(shard_id))
        self.shard_path = channel.task.sync_resource(shard_id)
        logging.info('synced wizard shard {}'.format(self.shard_path))

    def get_wizard_runtime(self):
        runtime_task = self.wizard_runtime_id if self.wizard_runtime_id else self.wizard
        runtime_resource = WizardBuilder.runtime_package_from_task(runtime_task)
        logging.info('syncing wizard runtime resource #{}'.format(runtime_resource))
        self.runtime_path = channel.task.sync_resource(runtime_resource)
        logging.info('synced wizard runtime {}'.format(self.runtime_path))

    def get_wizard_resources(self, use_threads=True):
        actions = [
            self.__class__.get_wizard_shard,
            self.__class__.get_wizard_binary,
        ]
        if self.work_with_runtime:
            actions.append(self.__class__.get_wizard_runtime)
        if use_threads:
            emanager = ExceptionManager()
            threads = [threading.Thread(target=ExceptionManager.run_target, args=(emanager, action, self))
                       for action in actions]
            for thread in threads:
                thread.start()
            for thread in threads:
                thread.join()
            emanager.check_exceptions()
        else:
            for action in actions:
                action(self)

    def sync_resources(self):
        logging.info('syncing resources ...')
        self.get_wizard_resources()
        logging.info('done syncing resources ...')

        def move_untar(src, dst, copy=False):
            logging.info('moving {} to {}'.format(src, dst))
            if src.endswith('.tar'):
                command = 'tar -C {} -xf {}'.format(self.data_path, self.runtime_path)
                process.run_process(command.split(), log_prefix="extract_files")
            else:
                if copy:
                    command = 'cp -rf {} {}'.format(src, dst)
                    process.run_process(command.split(), log_prefix="copy_files")
                else:
                    os.symlink(src, dst)

        move_untar(
            src=self.shard_path,
            dst=os.path.join(self.data_path, 'wizard'),
            copy=self.read_only_resources
        )
        if self.work_with_runtime:
            move_untar(
                src=self.runtime_path,
                dst=os.path.join(self.data_path, 'wizard.runtime'),
                copy=self.read_only_resources
            )
        self.config_path = self.generate_config()

    def get_run_cmd(self):
        cmd = '%s --stat -a %s -s %s -e %s' % (self.binary_path, self.data_path, self.config_path, self.eventlog_name)
        return cmd

    def start(self):
        logging.info('starting wizard')
        if not self.alive():
            self.process = process.run_process(
                self.cmd.split(), check=True, outputs_to_one_file=False, wait=False,
                log_prefix='wizard_%s' % self.timestamp, stderr=self.stderr)
            logging.info('===waiting for wizard to start {wizard.host}:{wizard.port}'.format(wizard=self))
            wait_searcher_start(self.host, self.port, subproc_list=[self.process], timeout=5 * 60)
        else:
            logging.info('still alive')

    def stop(self):
        logging.info('stopping wizard')
        if not self.alive():
            raise SandboxTaskFailureError('wizard is dead')
        else:
            url = 'http://{wizard.host}:{wizard.port}/wizard?action=shutdown'.format(wizard=self)
            urllib2.urlopen(url, timeout=5).read()
            wait_searcher_stop(self.host, self.port, timeout=30 * 60)
            self.process = None

    def generate_config(self):
        logging.info("Generating config")
        generated_config = 'wizard_%s.cfg' % self.timestamp
        config_name = {'classic': 'wizard', 'geo': 'geo'}.get(self.config_type)
        wizard_cfg_script = os.path.join(self.data_path, 'wizard/conf/config.py')
        no_cache_param = '--no-cache' if self.no_cache else ''
        process.run_process(
            "%s %s --testing %s > '%s'" % (wizard_cfg_script, config_name, no_cache_param, generated_config),
            shell=True,
            log_prefix='gen_cfg_%s' % self.timestamp
        )
        if config_name == 'geo':
            self.generate_geo_test_configs()
        return generated_config

    def generate_geo_test_configs(self):
        wizard_data_path = os.path.join(self.data_path, 'wizard')
        configs = (('thesaurus', 'generate_geosearch_config.py', 'thesaurus_geo.cfg'))

        for cfg_name, script, test_cfg_name in configs:
            # thesaurus_geo.cfg exists in newer branches, do not overwrite it
            if cfg_name == "thesaurus" and os.path.isfile(test_cfg_name):
                continue

            cfg_dir_path = os.path.join(wizard_data_path, cfg_name)
            script_path = os.path.join(cfg_dir_path, script)

            if not (os.path.isfile(script_path) and os.access(script_path, os.X_OK)):
                raise SandboxTaskFailureError('No such file or it is not executable: %s' % script)

            command = 'cat %s.cfg | ./%s' % (cfg_name, script)
            new_cfg = process.run_process(
                command,
                work_dir=cfg_dir_path,
                shell=True,
                outs_to_pipe=True).stdout.read()

            add_write_permissions_for_path(cfg_dir_path)
            new_cfg_path = os.path.join(cfg_dir_path, test_cfg_name)
            with open(new_cfg_path, 'w') as new_cfg_file:
                new_cfg_file.write(new_cfg)

    def alive(self):
        return self.process and self.process.poll() is None

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, type, value, traceback):
        self.stop()
