import os
import logging
import traceback

from collections import defaultdict
from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types

from sandbox.projects.common import constants
from sandbox.projects.common.utils import get_bsconfig
from sandbox.projects.websearch.begemot.common import Begemots


def check_wizard_build(build_wizard_id):
    try:
        parameters.SandboxIntegerParameter.cast(build_wizard_id)
    except (ValueError, TypeError) as ex:
        raise SandboxTaskFailureError('%s is not a valid task id: %s' % (build_wizard_id, ex))
    build_wizard_id = int(build_wizard_id)
    build_wizard_task = channel.sandbox.get_task(build_wizard_id)
    if not build_wizard_task:
        raise SandboxTaskFailureError('task with id %s not found' % build_wizard_id)
    if build_wizard_task.type not in ['BUILD_WIZARD', 'BUILD_WIZARD_2']:
        raise SandboxTaskFailureError('invalid task type: %s. Expected: BUILD_WIZARD or BUILD_WIZARD_2' % build_wizard_task.type)
    if build_wizard_task.is_done() and not build_wizard_task.is_finished():
        raise SandboxTaskFailureError('wizard build #%s is not finished' % build_wizard_id)
    return build_wizard_id


def tar(src):
    dst = os.path.basename(src) + '.tar'
    cmd = 'tar -cf {} {}'.format(dst, src)
    run_process(cmd.split())
    return dst


def untar(src, dst=None):
    if not src.endswith('.tar'):
        raise Exception('Can untar only tar')
    if not dst:
        dst = '.'
    cmd = 'tar -C {} -xf {}'.format(dst, src)
    run_process(cmd.split())
    result_path = os.path.join(dst, os.path.basename(src)[:-4])
    return result_path


def init_shard(task, shardname):
    max_attempts = 5
    done = False
    for attempt in range(max_attempts):
        try:
            task._subprocess(
                "%s shard_init --torrent %s" % (get_bsconfig(), shardname),
                check=True,
                wait=True,
                log_prefix="bsconfig"
            )
            done = True
            break
        except Exception as e:
            logging.info('%s' % e)
        if not done:
            raise SandboxTaskFailureError('Cant init shard')


_ALL_DEDICATED_SANDBOX_TAGS = [
    Tag.CUSTOM_WIZARD,
    Tag.CUSTOM_WIZARD_BEGEMOT,
    Tag.CUSTOM_BEGEMOT,
    Tag.CUSTOM_BEGEMOT_RELEASES,
    Tag.CUSTOM_BEGEMOT_SPELLCHECKER,
    Tag.CUSTOM_BEGEMOT_RELEASE_SPELLCHECKER,
]


class _CustomSandboxHosts(object):
    def __init__(self, trunk=None, release=None, precommit=None, other=None):
        self.trunk = trunk or (Tag.CUSTOM_WIZARD | Tag.CUSTOM_BEGEMOT | Tag.CUSTOM_WIZARD_BEGEMOT)
        self.release = release or Tag.CUSTOM_BEGEMOT_RELEASES
        self.precommit = precommit or (Tag.CUSTOM_BEGEMOT | Tag.CUSTOM_WIZARD_BEGEMOT)
        self.other = other or Tag.GENERIC


_SHARD_HOST_TAGS = defaultdict(_CustomSandboxHosts)
_SHARD_HOST_TAGS.update({
    'Spellchecker': _CustomSandboxHosts(
        trunk=Tag.CUSTOM_BEGEMOT_SPELLCHECKER,
        release=Tag.CUSTOM_BEGEMOT_RELEASE_SPELLCHECKER,
        precommit=Tag.CUSTOM_BEGEMOT_SPELLCHECKER,
    )
})

# All the tags where wizard and begemot tasks may execute.
# You also need to call 'setup_hosts_sdk1' in 'on_enqueue' for build tasks to precisely specify tags
ALL_SANDBOX_HOSTS_TAGS = Tag.GENERIC
for t in _ALL_DEDICATED_SANDBOX_TAGS:
    ALL_SANDBOX_HOSTS_TAGS |= t
WIZARD_SANDBOX_HOSTS_TAGS = Tag.GENERIC | Tag.CUSTOM_WIZARD | Tag.CUSTOM_WIZARD_BEGEMOT | Tag.CUSTOM_BEGEMOT_RELEASES
BEGEMOT_INVALID_HARDWARE = Tag.INTEL_E5645 # causes SIGILL for begemot binary, has no AVX support

# All performance measurements should be done on similar hardware. Most of the hosts have these parameters:
# LINUX_XENIAL is not really hardware, but affects performance as well
BEGEMOT_PERF_HARDWARE_TAGS = Tag.HDD & Tag.INTEL_E5_2660V1 & Tag.LINUX_XENIAL


def setup_hosts_sdk1(task, trunk_speedup=True, additional_restrictions=None):
    db = task.ctx.get('testenv_database', '')
    arc = task.ctx.get('checkout_arcadia_from_url', '')
    # default: no speedup, using cloud only
    task.client_tags = Tag.GENERIC
    begemot_shard = task.ctx.get('ShardName')

    if db == 'ws-begemot-trunk':
        if trunk_speedup:
            task.client_tags |= _SHARD_HOST_TAGS[begemot_shard].trunk
    elif db.startswith('ws-begemot-') or 'tags/begemot/stable-' in arc:  # a release base, not a precommit check
        task.client_tags |= _SHARD_HOST_TAGS[begemot_shard].release
    elif db and 'ws-' not in db:
        task.client_tags |= _SHARD_HOST_TAGS[begemot_shard].precommit
    else:
        # Manual runs come here.
        task.client_tags |= _SHARD_HOST_TAGS[begemot_shard].other

    if additional_restrictions is not None:
        task.client_tags &= additional_restrictions
    task.ctx[constants.SANDBOX_TAGS] = str(task.client_tags)  # SDK1 YaMakeTask expects tags here


def setup_hosts(task, additional_restrictions=None):
    '''
    This function should be called from on_save of SDK2 tasks.
    '''
    begemot_shard = task.Context.ShardName
    if not begemot_shard:
        try:
            begemot_shard_res = task.Parameters.begemot_shard
            if begemot_shard_res:
                resource_name = begemot_shard_res.name
                for name, svc in Begemots.values():
                    if svc.shard_resource_name == resource_name:
                        task.Context.ShardName = begemot_shard = name
                        break
        except Exception:
            pass
    db = task.Context.testenv_database or ''
    arc = task.Context.checkout_arcadia_from_url or ''
    task.Requirements.client_tags = Tag.GENERIC
    if db == 'ws-begemot-trunk':
        task.Requirements.client_tags |= _SHARD_HOST_TAGS[begemot_shard].trunk
    elif db.startswith('ws-begemot-') or 'tags/begemot/stable-' in arc:  # a release base, not a precommit check
        task.Requirements.client_tags |= _SHARD_HOST_TAGS[begemot_shard].release
    elif db and 'ws-' not in db:
        task.Requirements.client_tags |= _SHARD_HOST_TAGS[begemot_shard].precommit
    else:
        # Manual runs come here.
        # No dedicated hosts for them is expected.
        task.Requirements.client_tags |= _SHARD_HOST_TAGS[begemot_shard].other
    if additional_restrictions is not None:
        task.Requirements.client_tags &= additional_restrictions


def shrink_resources_ttl(task, res_type, trunk_ttl=7, other_te_ttl=2, manual_runs=7):
    '''
    For the given task, modify ttl of all its resources of res_type
    trunk_ttl: for runs from ws-begemot-trunk. Binsearch over old builds may require 7 days.
    other_te_ttl: for precommit checks and release builds
    manual_runs: for runs not from TE

    Usually we do not need this data for long, see also BEGEMOT-747.
    '''
    te_db = task.ctx.get('testenv_database', '')
    if te_db == 'ws-begemot-trunk':
        ttl = trunk_ttl
    elif te_db:
        ttl = other_te_ttl
    else:
        ttl = manual_runs
    for r in channel.sandbox.list_resources(resource_type=res_type, task_id=task.id):
        channel.sandbox.set_resource_attribute(resource_id=r.id, attribute_name='ttl', attribute_value=ttl)


def on_enqueue(task):
    if common.config.Registry().common.installation == ctm.Installation.LOCAL:
        return
    try:
        requested_platform = task.arch
        if not requested_platform or requested_platform == ctm.OSFamily.ANY:
            requested_platform = 'linux'

        requested_platform = common.platform.get_platform_alias(requested_platform)
        # save for debug purpose - hosts_match_score is processed on sandbox server,
        # but ctx will be transfered to the task instance and will be available in logs
        task.ctx['requested_platform'] = requested_platform
        task.arch = requested_platform
    except:
        logging.exception("Error in setting requested platform on_enqueue of task #%s type %s", self.id, self.type)

    try:
        tags = task.ctx.get(constants.SANDBOX_TAGS, 'GENERIC').strip().upper()
        if not tags:
            return
        tags = Tag.Query(tags)
        task.client_tags = task.__class__.client_tags & tags
    except:
        logging.exception("Error in setting client tags on_enqueue of task #%s type %s", task.id, task.type)


def delete_raw_build_output(task_id):
    # Drop BUILD_OUTPUT. All valuable resources from it have been saved, and this package is not needed.
    # BUILD_OUTPUT is not backed up and only eats up dedicated host HDD space.
    for r in channel.sandbox.list_resources(resource_type=resource_types.BUILD_OUTPUT, task_id=task_id):
        channel.sandbox.delete_resource(r.id, True)


def validate_utf8(path):
    with open(path, "r") as f:
        cur_line = 1
        for r in f:
            try:
                r.decode('utf-8')
                cur_line += 1
            except UnicodeDecodeError as e:
                logging.exception('UnicodeDecodeError occured when trying to parse line number {}'.format(cur_line))
                width = 20
                logging.info('Problem line (number {}):'.format(cur_line))

                print_before = e.start > width
                print_after = len(r) > e.end + width

                if print_before:
                    logging.info(r[:e.start - width].decode('utf-8', errors='backslashreplace') + '...\n\n\n')

                logging.info('Problem is here (invalid symbols are backslash-replaced with \\x):')
                logging.info(('...' if print_before else '') +
                             r[max(0, e.start - width): min(e.end + width, len(r))].decode('utf-8', errors='backslashreplace') +
                             ('...' if print_after else ''))

                if print_after:
                    logging.info('\n\n\n...' + r[e.end + width:].decode('utf-8', errors='backslashreplace'))

                return False
    return True


INVALID_UNICODE_FAILURE_MESSAGE = "Line with invalid unicode detected. See common.log for details. BEGEMOT-2532"
