# coding: utf-8

import time
import urllib2
import logging

from sandbox import sdk2

from sandbox.sandboxsdk import util
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError

from sandbox.projects.common import apihelpers
from sandbox.projects.common import constants as consts

from svn_wizard import SvnWizard


class CommonBuilder(object):
    @staticmethod
    def resource_from_task(task_id, resource_type, arch=None):
        if arch is None:
            arch = util.system_info()['arch']
        logging.info('getting %s (%s) from %s' % (resource_type, arch, task_id))
        task = channel.sandbox.get_task(task_id)
        if not task.is_finished():
            raise SandboxTaskFailureError('task %s is %s' % (task_id, task.status))
        resource_types = resource_type if isinstance(resource_type, (list, tuple)) else [resource_type]
        for res_type in resource_types:
            logging.info('searching res_type {}'.format(res_type))
            resource = apihelpers.list_task_resources(task_id, res_type, arch)
            if resource:
                resource_id = resource[0].id
                logging.info('got %s #%d' % (res_type, resource_id))
                return resource_id
        raise SandboxTaskUnknownError('cannot get %s from task #%s' % (resource_type, task_id))


class WizardBuilder(CommonBuilder):

    ACCEPTED_TASK_TYPES = ["BUILD_BEGEMOT_EXECUTABLE", "RELEASE_BEGEMOT_RESOURCES"]

    @staticmethod
    def build_runtime(executables, runtime_rev=None, arch=None):
        return channel.task.create_subtask(
            'BUILD_WIZARD_2',
            input_parameters={
                'build_runtime_with': executables,
                'runtime_data': runtime_rev,
            },
            inherit_notifications=True,
            description="Wizard runtime build {}".format(time.strftime("%Y-%m-%d %H:%M:%S")),
            arch=arch,
        ).id

    @staticmethod
    def get_production_wizard_tag(arcadia=False, arcadia_tests_data=False):
        production_url = 'http://reqwizard.yandex.net:8891/admin?action=version'
        for attempt in range(5):
            try:
                logging.info('Reading %s', production_url)
                version_info = urllib2.urlopen(production_url).read()
                logging.debug('Version info: %s' % version_info)
                logging.info('Parsing ...')
                branch_mined = None
                for line in version_info.split('\n'):
                    if line.startswith('    URL: svn'):
                        branch_mined = line.replace('    URL: ', '').strip()
                logging.info('Got branch: %s', branch_mined)
                tag = SvnWizard(branch_mined)
                logging.info('Got production tag: %s', tag.branch())
                if arcadia:
                    return tag.arcadia()
                elif arcadia_tests_data:
                    return tag.arcadia_tests_data()
                else:
                    return tag.branch()
            except Exception as e:
                logging.info("get_last_stable_wizard_tag has failed")
                logging.exception(e)
                time.sleep(10 + 5 * attempt)

    @staticmethod
    def get_production_wizard_task(get_any_if_failed=True):
        tag = WizardBuilder.get_production_wizard_tag()
        logging.info('Looking for stable task with tag %s', tag)

        attempt_limit = 10
        first_stable_task = None
        for attempt in range(5):
            stable_releases = channel.sandbox.list_releases(
                resource_type='BEGEMOT_EXECUTABLE',
                limit=attempt_limit,
                offset=attempt * attempt_limit
            )
            if not stable_releases:
                raise SandboxTaskFailureError(
                    'No stable released resources with type {} found'.format('BEGEMOT_EXECUTABLE')
                )
            logging.info('%s releases found', len(stable_releases))
            for stable_release in stable_releases:
                logging.debug('Checking released task #%s', stable_release.task_id)
                try:
                    stable_task = channel.sandbox.get_task(stable_release.task_id)
                    if first_stable_task is None:
                        first_stable_task = stable_task
                except Exception:
                    # Fix for sdk2
                    stable_task = sdk2.Task[stable_release.task_id]
                    if first_stable_task is None:
                        first_stable_task = stable_task
                    arcadia_url = getattr(stable_task.Context, consts.ARCADIA_URL_KEY)
                else:
                    arcadia_url = stable_task.ctx[consts.ARCADIA_URL_KEY]
                if (SvnWizard(tag) == SvnWizard(arcadia_url) and stable_task.type in WizardBuilder.ACCEPTED_TASK_TYPES):
                    logging.info('This is a stable wizard task with tag "%s": "%s"', tag, stable_task.id)
                    return stable_task

        logging.info("Failed to find stable task for tag {}".format(tag))
        if get_any_if_failed:
            return first_stable_task
        raise SandboxTaskUnknownError("can't find stable wizard by tag '%s'" % tag)

    @staticmethod
    def resource_from_production_task(resource_type):
        return WizardBuilder.resource_from_task(WizardBuilder.get_production_wizard_task().id, resource_type)

    @staticmethod
    def wizard_from_task(task_id):
        return WizardBuilder.resource_from_task(task_id, 'REMOTE_WIZARD')

    @staticmethod
    def printwzrd_from_task(task_id):
        return WizardBuilder.resource_from_task(task_id, 'PRINTWZRD')

    @staticmethod
    def evlogdump_from_task(task_id):
        return WizardBuilder.resource_from_task(task_id, 'EVLOGDUMP_EXECUTABLE')

    @staticmethod
    def runtime_package_from_task(task_id):
        return WizardBuilder.resource_from_task(task_id, ['WIZARD_RUNTIME_PACKAGE', 'WIZARD_RUNTIME_PACKAGE_PATCHED'])

    @staticmethod
    def wizard_shard_from_task(task_id):
        return WizardBuilder.resource_from_task(task_id, 'WIZARD_SHARD')
