# -*- coding: utf-8 -*-
import hashlib
import json
import logging
import time
import six

from sandbox import common
from sandbox import sandboxsdk
from sandbox import sdk2

from sandbox.common import config
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.ya_deploy import yp_client


# Common parameter names, to be used in sandbox tasks to parametrize
# our release integration mixin behaviour
YP_API_URL_KEY = 'yp_api_url'
YP_TOKEN_VAULT_KEY = 'yp_token_vault'
YP_TOKEN_YAV_SECRET_ID_KEY = 'yp_token_yav_secret_id'
RELEASE_TO_YA_DEPLOY_KEY = 'release_to_ya_deploy'
YA_DEPLOY_RELEASE_TYPE_KEY = 'release_type'

# Default parameter values
DEFAULT_YP_API_URL = 'https://xdc.yp.yandex.net:8443'
DEFAULT_YP_TOKEN_VAULT = 'yp_token'

# Consts for mixin internal logic (keys from additional_parameters argument of on_release method)
RELEASE_ID_KEY = 'release_id'
RELEASE_SUBJECT_KEY = 'release_subject'
RELEASE_COMMENTS_KEY = 'release_comments'
RELEASER_KEY = 'releaser'
RELEASE_TYPE_KEY = 'release_status'
EMPTY_DOCKER_HASH_PLACEHOLDER = 'EMPTY'
REGISTRY_HOST = 'registry.yandex.net'
REGISTRY_PREFIX = '{}/'.format(REGISTRY_HOST)
RELEASE_TYPES = [
    ('stable', 'stable'),
    ('prestable', 'prestable'),
    ('testing', 'testing'),
    ('unstable', 'unstable'),
    ('cancelled', 'cancelled'),
]
RELEASE_PER_RESOURCE_KEY = 'release_per_resource'

YA_DEPLOY_RELEASE_IDS_CONTEXT_FIELD = "ya_deploy_release_ids"

YA_DEPLOY_URL = "https://deploy.yandex-team.ru"


class YpTokenVaultParameter(sandboxsdk.parameters.SandboxStringParameter):
    name = YP_TOKEN_VAULT_KEY
    description = ('YP token vault name (must be given for export to Yandex.Deploy, default: "yp_token") '
                   'https://wiki.yandex-team.ru/yp/')
    required = False


class YpTokenVaultParameter2(sdk2.parameters.String):
    name = YP_TOKEN_VAULT_KEY
    description = ('YP token vault name (must be given for export to Yandex.Deploy, default: "yp_token") '
                   'https://wiki.yandex-team.ru/yp/')
    required = False


class ReleaseToYaDeployParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = RELEASE_TO_YA_DEPLOY_KEY
    description = 'Enable release to Yandex.Deploy'
    required = False
    default_value = False


class ReleaseToYaDeployParameter2(sdk2.parameters.Bool):
    name = RELEASE_TO_YA_DEPLOY_KEY
    description = 'Enable release to Yandex.Deploy'
    required = False


class ReleaseTypeParameter(sandboxsdk.parameters.SandboxSelectParameter):
    name = YA_DEPLOY_RELEASE_TYPE_KEY
    description = 'Yandex.Deploy release type'
    required = False
    default_value = 'unstable'
    choices = RELEASE_TYPES


class ReleaseTypeParameter2(sdk2.parameters.String):
    name = YA_DEPLOY_RELEASE_TYPE_KEY
    description = 'Yandex.Deploy release type'
    required = False
    default_value = 'unstable'
    choices = RELEASE_TYPES


class ReleasePerResourceParameter(sandboxsdk.parameters.SandboxBoolParameter):
    name = RELEASE_PER_RESOURCE_KEY
    description = 'Release each resource by separate release in Yandex.Deploy'
    required = False
    default_value = False


class ReleasePerResourceParameter2(sdk2.parameters.Bool):
    name = RELEASE_PER_RESOURCE_KEY
    description = 'Release each resource by separate release in Yandex.Deploy'
    required = False
    default_value = False


def make_docker_release_id(images, release_type):
    sorted_images = sorted(images, key=lambda img: (img['name'], img['tag']))
    h = hashlib.md5('')
    for img in sorted_images:
        h.update(img['name'])
        h.update(img['tag'])
    return 'DOCKER-SANDBOX-{}-{}'.format(h.hexdigest()[:12], release_type)


def make_docker_release_payload(images, release_type, release_author, title, description):
    release_id = make_docker_release_id(images=images, release_type=release_type)
    return {
        'meta': {
            'id': release_id,
        },
        'spec': {
            'title': title,
            'description': description,
            'docker': {
                'images': images,
                'release_type': release_type,
                'release_author': release_author,
            }
        }
    }


def parse_docker_image_path(path):
    s = path
    if ":" not in path:
        raise ValueError('Invalid docker image path "{}", it must be "[registry.yandex.net/]<path>:<tag>"'.format(s))
    if path.startswith(REGISTRY_PREFIX):
        path = path[len(REGISTRY_PREFIX):]
    a = path.split(":")
    if len(a) != 2:
        raise ValueError('Invalid docker image path "{}", it must be "[registry.yandex.net/]<path>:<tag>"'.format(s))
    return a


def make_docker_image_from_path(path):
    image_name, image_tag = parse_docker_image_path(path)
    return {
        'name': image_name,
        'tag': image_tag,
        'digest': EMPTY_DOCKER_HASH_PLACEHOLDER,
        'registry_host': REGISTRY_HOST
    }


def create_docker_release_inner(task, images, release_type, release_author, title,
                                description):
    logging.info('Sending release of Docker task to Yandex.Deploy')
    payload = make_docker_release_payload(
        images=images,
        release_type=release_type,
        release_author=release_author,
        title=title,
        description=description,
    )
    logging.info('Release payload: %s', json.dumps(payload, indent=4))
    task.yp_client.create_release(payload)
    task.set_info(
        'Ya.Deploy release request <a href="{0}/releases/{1}">{1}</a> was created.'.format(
            YA_DEPLOY_URL, payload['meta']['id']),
        do_escape=False
    )
    return payload


def create_docker_release(task, images, additional_parameters):
    release_type = additional_parameters[RELEASE_TYPE_KEY]
    release_author = str(additional_parameters.get(RELEASER_KEY, 'anonymous'))
    title = additional_parameters.get(RELEASE_SUBJECT_KEY)
    description = additional_parameters.get(RELEASE_COMMENTS_KEY)
    return create_docker_release_inner(
        task=task,
        images=images,
        release_type=release_type,
        release_author=release_author,
        title=title,
        description=description,
    )


def stringify_resource_attributes(attributes):
    # ttl resource attribute can have value which represents infinity or -infinity.
    # This values cannot be parsed by YP json decoder. So we cast values of all
    # attributes to strings.
    rv = {}
    for k, v in six.iteritems(attributes):
        rv[k] = str(v)
    return rv


def _make_release_id(task_id, release_type):
    return 'SANDBOX-{}-{}'.format(task_id, release_type)


def _get_yd_release_spec(task, release_type, resources, created_seconds,
                         additional_parameters):
    return {
        'title': additional_parameters.get(RELEASE_SUBJECT_KEY),
        'description': additional_parameters.get(RELEASE_COMMENTS_KEY),
        'sandbox': {
            'task_type': str(task.type),
            'task_id': str(task.id),
            'task_author': str(task.author),
            'release_author': str(additional_parameters.get(RELEASER_KEY, 'anonymous')),
            'title': additional_parameters.get(RELEASE_SUBJECT_KEY, ''),
            'description': additional_parameters.get(RELEASE_COMMENTS_KEY, ''),
            'release_type': release_type,
            'resources': resources,
            'task_creation_time': {
                'seconds': created_seconds,
            }
        }
    }


def _get_yd_release_info(task, additional_parameters):
    release_type = additional_parameters[RELEASE_TYPE_KEY]
    spec = _get_yd_release_spec(task=task,
                                release_type=release_type,
                                resources=task.get_yd_release_resources(),
                                created_seconds=int(time.time()),
                                additional_parameters=additional_parameters)
    release_id = _make_release_id(task.id, release_type)
    return {'meta': {'id': release_id}, 'spec': spec}


def _get_yd_release_info_per_resource(task, additional_parameters):
    rv = []
    release_type = additional_parameters[RELEASE_TYPE_KEY]
    base_release_id = _make_release_id(task.id, release_type)
    created_seconds = int(time.time())
    resources = task.get_yd_release_resources()
    for resource in resources:
        spec = _get_yd_release_spec(task=task,
                                    release_type=release_type,
                                    resources=[resource],
                                    created_seconds=created_seconds,
                                    additional_parameters=additional_parameters)
        release_id = '{}-{}'.format(base_release_id, resource['resource_id'])
        rv.append({'meta': {'id': release_id}, 'spec': spec})
    return rv


def _send_yd_release(task, data):
    release_id = data['meta']['id']
    logging.info('Sending release of task %s to Ya.Deploy', task.id)
    logging.info('Release payload: %s', json.dumps(data, indent=4))
    is_created = task.yp_client.create_release(data)
    logging.info('Release of task %s has been sent to Ya.Deploy', task.id)
    if is_created:
        task.set_info(
            'Ya.Deploy release request <a href="{0}/releases/{1}">{1}</a> was created.'.format(
                YA_DEPLOY_URL, release_id),
            do_escape=False
        )
    else:
        task.set_info(
            'Ya.Deploy release {} already exists. Do not recreate.'.format(release_id),
            do_escape=False
        )

    task.store_release_id_in_context(release_id)


def _on_release_ya_deploy_task(task, additional_parameters, release_per_resource=False):
    task.mark_released_resources(additional_parameters[RELEASE_TYPE_KEY])
    registry = config.Registry()
    if registry.common.installation != ctm.Installation.PRODUCTION:
        logging.info('Current installation is not PRODUCTION but %s, will not send release to Ya.Deploy',
                     registry.common.installation)
        return

    logging.info(
        'Gathering information about release of task %s with release parameters %r to send it to Ya.Deploy',
        task.id, additional_parameters
    )
    if release_per_resource:
        for data in _get_yd_release_info_per_resource(task, additional_parameters):
            _send_yd_release(task, data)
    else:
        data = _get_yd_release_info(task, additional_parameters)
        _send_yd_release(task, data)


def _get_yp_token(task, vault_name):
    try:
        return task.get_token_from_vault(task.owner, vault_name)
    except common.errors.VaultNotFound:
        if vault_name is None:
            raise
        # Last try: get token 'shared with' by name.
        return task.get_token_from_vault(vault_name)


def _get_yp_token_handle_error(task, vault_name):
    if not vault_name:
        raise RuntimeError('Cannot get YP token from task context to create release in Yandex.Deploy')
    try:
        return _get_yp_token(task, vault_name)
    except Exception as e:
        raise RuntimeError('Cannot get YP token to create release in Yandex.Deploy: "{}"'.format(e))


class ReleaseToYaDeployTask():
    """
    Mixin class for sdk1 to support sending release requests to Ya.Deploy. Override method on_release.
    """

    def get_token_from_vault(self, owner, name=None):
        return self.get_vault_data(owner, name)

    def get_yd_release_resources(self):
        resources = []
        for r in self.list_resources():
            if r.is_ready() and r.type.releasable:
                resources.append({
                    'resource_id': str(r.id),
                    'type': str(r.type),
                    'filename': r.file_name,
                    'description': r.name,
                    'arch': r.arch,
                    'releasers': r.type.releasers or [],
                    'skynet_id': r.skynet_id,
                    'file_md5': r.file_md5,
                    'attributes': stringify_resource_attributes(r.attrs)
                })
        logging.info('List of resources to release:\n{}'.format(resources))
        return resources

    def get_yp_api_url(self):
        return self.ctx.get(YP_API_URL_KEY) or DEFAULT_YP_API_URL

    def get_yp_oauth_token(self):
        # At first try to get secret id from context
        yav_secret_id = self.ctx.get(YP_TOKEN_YAV_SECRET_ID_KEY)
        if not yav_secret_id:
            # Then try to get secret id from task attrs
            yav_secret_id = getattr(self, 'YP_TOKEN_YAV_SECRET_ID', None)
        if yav_secret_id:
            return sdk2.yav.Secret(yav_secret_id).data()['yp-token']
        # If no secret id given, try to use sandbox vault
        vault_name = self.ctx.get(YP_TOKEN_VAULT_KEY) or DEFAULT_YP_TOKEN_VAULT
        return _get_yp_token_handle_error(self, vault_name)

    @property
    def yp_client(self):
        api_url = self.get_yp_api_url()
        oauth_token = self.get_yp_oauth_token()
        return yp_client.YpClient(api_url=api_url, oauth_token=oauth_token)

    def on_release(self, additional_parameters):
        release_per_resource = self.ctx.get(RELEASE_PER_RESOURCE_KEY, False)
        _on_release_ya_deploy_task(self, additional_parameters, release_per_resource)

    def store_release_id_in_context(self, release_id):
        ya_deploy_release_ids = self.ctx.setdefault(YA_DEPLOY_RELEASE_IDS_CONTEXT_FIELD, [])
        if release_id not in ya_deploy_release_ids:
            ya_deploy_release_ids.append(release_id)


class ReleaseToYaDeployTask2():
    """
    Mixin class for sdk2 to support sending release requests to Ya.Deploy. Override method on_release.
    """
    def _get_parameter_or_context_value(self, name, default):
        value = getattr(self.Context, name)
        if value is not ctm.NotExists:
            return value
        return getattr(self.Parameters, name, default)

    def get_token_from_vault(self, owner, name=None):
        return sdk2.Vault.data(owner, name)

    def get_yd_release_resources(self):
        all_res = sdk2.Resource.find(task=self, state=ctr.State.READY).limit(0)
        resources = []
        for r in all_res:
            if not r.type.releasable:
                continue
            resources.append({
                'resource_id': str(r.id),
                'type': r.type.name,
                'filename': r.path.name,
                'description': r.description,
                'skynet_id': r.skynet_id,
                'arch': r.arch or '',
                'file_md5': r.md5 or '',
                'releasers': r.releasers or [],
                'attributes': stringify_resource_attributes(dict(r))
            })
        logging.info('List of resources to release:\n{}'.format(resources))
        return resources

    def get_yp_api_url(self):
        return self._get_parameter_or_context_value(YP_API_URL_KEY, DEFAULT_YP_API_URL)

    def get_yp_oauth_token(self):
        # At first try to get secret id from params or context
        yav_secret_id = self._get_parameter_or_context_value(YP_TOKEN_YAV_SECRET_ID_KEY, None)
        if not yav_secret_id:
            # Then try to get secret id from task attrs
            yav_secret_id = getattr(self, 'YP_TOKEN_YAV_SECRET_ID', None)
        if yav_secret_id:
            return sdk2.yav.Secret(yav_secret_id).data()['yp-token']
        # If no secret id given, try to use sandbox vault
        vault_name = self._get_parameter_or_context_value(YP_TOKEN_VAULT_KEY, DEFAULT_YP_TOKEN_VAULT)
        if not vault_name:
            vault_name = DEFAULT_YP_TOKEN_VAULT
        return _get_yp_token_handle_error(self, vault_name)

    @property
    def yp_client(self):
        api_url = self.get_yp_api_url()
        oauth_token = self.get_yp_oauth_token()
        return yp_client.YpClient(api_url=api_url, oauth_token=oauth_token)

    def on_release(self, additional_parameters):
        release_per_resource = self._get_parameter_or_context_value(RELEASE_PER_RESOURCE_KEY, default=False)
        _on_release_ya_deploy_task(self, additional_parameters, release_per_resource=release_per_resource)

    def store_release_id_in_context(self, release_id):
        if YA_DEPLOY_RELEASE_IDS_CONTEXT_FIELD not in self.Context:
            setattr(self.Context, YA_DEPLOY_RELEASE_IDS_CONTEXT_FIELD, [])
        ya_deploy_release_ids = getattr(self.Context, YA_DEPLOY_RELEASE_IDS_CONTEXT_FIELD)
        if release_id not in ya_deploy_release_ids:
            ya_deploy_release_ids.append(release_id)


class ReleaseToNannyAndYaDeployTask(nanny.ReleaseToNannyTask, ReleaseToYaDeployTask):
    """
    Mixin class for sdk1 to support sending release requests both to Nanny and to Ya.Deploy. Override method on_release.
    """
    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        ReleaseToYaDeployTask.on_release(self, additional_parameters)


class ReleaseToNannyAndYaDeployTask2(nanny.ReleaseToNannyTask2, ReleaseToYaDeployTask2):
    """
    Mixin class for sdk2 to support sending release requests both to Nanny and to Ya.Deploy. Override method on_release.
    """
    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        ReleaseToYaDeployTask2.on_release(self, additional_parameters)
