# -*- coding: utf-8 -*-
import json
import logging
import os

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


YP_ERROR_TEMPLATE = 'YP responded with error:\n{}'

UNKNOWN_YP_ERROR_TEMPLATE = 'YP responded with unknown error. HTTP-code: {}. Body:\n{}'


def _contains_error_code(yp_error, code):
    if str(yp_error.get("code")) == code:
        return True
    return any(_contains_error_code(e, code) for e in yp_error.get('inner_errors', []))


class YpApiException(Exception):
    def __init__(self, message, yp_error_code, yp_error):
        super(YpApiException, self).__init__(message)
        self.yp_error_code = yp_error_code
        self.yp_error = yp_error

    def contains_code(self, code):
        return _contains_error_code(self.yp_error, code)


def _handle_http_error(e):
    resp = e.response
    error = resp.headers.get('X-YT-Error')
    code = resp.headers.get('X-YT-Response-Code')

    if error:
        msg = YP_ERROR_TEMPLATE.format(error)
    else:
        msg = UNKNOWN_YP_ERROR_TEMPLATE.format(resp.status_code, resp.content)
    raise YpApiException(message=msg,
                         yp_error_code=str(code),
                         yp_error=json.loads(error))


class YpClient(object):
    def __init__(self, api_url, oauth_token):
        self._api_url = api_url.rstrip('/')
        self._oauth_token = oauth_token
        self._session = requests.Session()
        CA_CERTS = '/etc/ssl/certs/ca-certificates.crt'
        if os.path.isfile(CA_CERTS):
            self._session.verify = CA_CERTS
        retries = Retry(total=3, backoff_factor=1.0, status_forcelist=[429, 500, 502, 503, 504])
        self._session.mount(self._api_url, HTTPAdapter(max_retries=retries))
        self._session.headers['Accept'] = 'application/json'
        self._session.headers['X-YT-Request-Format-Options'] = '{encode_utf8=%false}'
        self._session.headers['X-YT-Response-Format-Options'] = '{encode_utf8=%false}'
        if self._oauth_token:
            self._session.headers['Authorization'] = 'OAuth {}'.format(self._oauth_token)

    def _create_object(self, object_type, object_data):
        url = '{}/ObjectService/CreateObject'.format(self._api_url)
        data = {'object_type': object_type, 'attributes': object_data}
        logging.info('Creating object in YP, request payload: %s', json.dumps(data, indent=4))
        resp = self._post_url(url, data)
        return resp['object_id']

    def create_release(self, data):
        try:
            self._create_object(object_type='release', object_data=data)
        except YpApiException as e:
            if e.contains_code('100001'):
                return False
            raise
        return True

    def _post_url(self, url, data):
        try:
            r = self._session.post(url=url, json=data)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise _handle_http_error(e)
        except Exception as e:
            raise YpApiException(
                message='Failed to post url "{}"\nError: {}'.format(url, e),
                yp_error_code="",
                yp_error={}
            )
        return r.json()
