# -*- encoding: utf-8 -*-

import re
import json
import tarfile

from sandbox.projects.common.yabs.yabs_global_info import YabsGlobalInfo
from sandbox.projects.common.apihelpers import get_last_resource
from sandbox.projects.yabs.global_info.resources import YABS_APP_OPTIONS


APP_OPTIONS_YT_PROXY = 'locke.yt.yandex.net'
APP_OPTIONS_YT_DIR = '//home/yabs/hostoptions'


class AppOptions(YabsGlobalInfo):
    RESOURCE_TYPE = YABS_APP_OPTIONS
    PATH = "global-host-options/"
    SOURCE_FILES = ["global-host-options-dev.json", "global-host-options.json"]
    PROD_SOURCE_FILES = ["global-host-options.json"]
    RESULT_FILE = "app-options.json"
    ENV_VARIABLES_PREFIX = "SBYT_ENV:"

    @classmethod
    def get_app_options_resource(cls, task):
        res = get_last_resource(cls.RESOURCE_TYPE)
        if not res:
            raise BaseException("Resource with app-options not found")
        path = task.sync_resource(res.id)
        if not path:
            raise BaseException("Resource with id={} not found".format([res.id]))

        with tarfile.open(path, 'r:gz') as tar:
            tar.extractall()

    @classmethod
    def get_app_options_resource_from_yt(cls, yt_client):
        with yt_client.Transaction():
            if not yt_client.exists(APP_OPTIONS_YT_DIR):
                raise BaseException("There is no directory {}".format(APP_OPTIONS_YT_DIR))
            for options_file in yt_client.list(APP_OPTIONS_YT_DIR):
                with open(options_file, "wb") as f:
                    yt_path = "/".join([APP_OPTIONS_YT_DIR, options_file])
                    f.write(yt_client.read_file(yt_path).read())

    # Все json-значения превращаются в строки, а приведение к нужному типу производится уже в клиентской библиотеке.
    # Так сделано потому, что типизация значений всё равно не решает главной
    # проблемы - отличие фактического типа опции от ожидаемого в приложении.
    @classmethod
    def make_app_options_file(cls, yt_proxy, dev=False, files=[], result_name=RESULT_FILE):
        yt_proxy = yt_proxy.replace(".yt.yandex.net", "")
        if not files:
            files = cls.SOURCE_FILES if dev else cls.PROD_SOURCE_FILES
        result = {}
        opt_mask_len = {}
        parsed_names = set()
        for f in files:
            with open(f) as fp:
                for mask, options in json.load(fp).iteritems():
                    mask_len = len(re.sub("[*?]", "", mask))  # mask lenght without special symbols
                    mask_re = mask.replace(".", "\\.").replace("?", ".").replace("*", ".*") + "$"
                    if re.match(mask_re, yt_proxy):
                        for name, value in options.iteritems():
                            if mask_len >= opt_mask_len.get(name, 0) and name not in parsed_names:
                                result[name] = str(value)  # force string
                                opt_mask_len[name] = mask_len
                parsed_names.update(result.keys())

        with open(result_name, "w") as f:
            json.dump(result, f)

        return result_name

    @classmethod
    def get_env_variables(cls, app_options_file):
        result = {}
        pref_len = len(cls.ENV_VARIABLES_PREFIX)
        with open(app_options_file, "r") as f:
            opts_dict = json.load(f)
            for opt_name, opt_value in opts_dict.iteritems():
                if opt_name.startswith(cls.ENV_VARIABLES_PREFIX):
                    result[opt_name[pref_len:]] = opt_value
        return result
