# -*- coding: utf-8 -*-

import logging
import requests
import re
import httplib
import urllib2
import socket

from sandbox.projects.common.decorators import retries


class WrongResponseException(Exception):
    pass


BAMBOO_AUTH = ('bsint', 'bsint')
NETWORK_EXCEPTIONS = (
    WrongResponseException,
    requests.exceptions.RequestException,
    requests.exceptions.ConnectionError,
    requests.exceptions.Timeout,
    httplib.HTTPException,
    urllib2.URLError,
    socket.error
)


@retries(max_tries=3, delay=0.2, backoff=2, exceptions=NETWORK_EXCEPTIONS)
def bamboo_request(bamboo_server, params):
    params['wait_builds_info'] = 'yes'

    if bamboo_server == 'antifraud':
        handle = 'https://afpb-bamboo.yandex-team.ru/rest/cloud/1.0/build-notify'
    else:
        handle = 'https://qabs-bamboo.yandex-team.ru/rest/cloud/1.0/build-notify'
    logging.debug('Bamboo request: %s %s', handle, str(params))

    response = requests.request(
        'POST', handle,
        auth=BAMBOO_AUTH,
        verify=False,
        data=params
    )

    if response.status_code == 500:
        raise WrongResponseException('Bamboo wants to answer later (error 500)')

    if response.status_code != 200:
        raise WrongResponseException('Wrong response status {}'.format(response))

    if not response.text:
        raise WrongResponseException('Empty response text {}'.format(response))

    ret = response.json()
    logging.debug('Bamboo response: %s', str(ret))

    builds = ret.get('builds')
    if not builds:
        return []

    success_builds = []
    for b in builds:
        build = b.get('build')
        success = b.get('success')
        key = b.get('key')

        if not success or not key:
            code = b.get('code', '')
            message = b.get('message', '')
            logging.warn('Build %s not succeed:\ncode: %s\nmessage: %s', build, code, message)
        else:
            success_builds.append('<a href="https://qabs-bamboo.yandex-team.ru/browse/{0}">{0}</a>'.format(key))

    return success_builds


def bamboo_build_notify(params, build_params):
    if 'test-imagelist' in build_params:
        params['imagelist'] = build_params['test-imagelist']
    if 'test-also-install' in build_params:
        params['also_install'] = build_params['test-also-install']

    result = []
    for bamboo_server in ['default', 'antifraud']:
        params['parasprite'] = build_params['parasprite'] if 'parasprite' in build_params else 'on'
        if re.search('^qabs-', params['package']):
            params['parasprite'] = 'off'

        pt_servers = build_params['parasprite-option'] if 'parasprite-option' in build_params else 'default'
        if not any(bamboo_server == _ for _ in pt_servers.split('\s*,\s*')):
            params['parasprite'] = 'off'

        try:
            result.extend(bamboo_request(bamboo_server, params))
        except:
            result.append('Failed send notifier in ' + bamboo_server + ' bamboo for test ' + params['package'])

    return result
