"""
Cachedaemon-based stub for yabs-server tests
"""
import logging
import os
import shutil
import subprocess
import time
from multiprocessing.pool import ThreadPool

import requests

from sandbox.common import rest
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.projects.common.yabs.cachedaemon import checklog
from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_CACHE_DAEMON_STUB_DATA
from sandbox.projects.yabs.qa.sut.utils import reserve_port
from sandbox.sandboxsdk.paths import add_write_permissions_for_path, get_logs_folder, make_folder


logger = logging.getLogger(__name__)


SERVICES_BY_TYPE = {
    'rtcrypta': ['rtcrypta_get'],
    'bsbts_all': ['bsbts'],
    'awaps': ['default_dsp'],
    'pnocsy': [
        'dsp_proxy_v4_{}_{}'.format(n, p) for n, ports in [(1, range(50250, 50265, 2)), (2, range(50500, 50515, 2))] for p in ports
    ] + [
        'pnocsy_{}'.format(p) for p in list(range(50250, 50265, 2)) + list(range(50500, 50515, 2))
    ] + [
        'pnocsy'
    ],
    'bscount': ['bscount'],
    'rank_cache_get': ['rank_cache_get'],
}


class CacheDaemonStubSandboxNonIntegrated(object):
    """
    CacheDaemon-based stub.
    https://wiki.yandex-team.ru/users/vmordovin/cachedaemon/
    https://st.yandex-team.ru/SEARCH-1757

    On /POST request, CacheDaemon calculates key from request line and headers (as specified in config)
    and stores key-reply pairs in file-based cache.
    On /GET request, CacheDaemon calculates key and finds reply in cache by this key.

    This class handles multiple CacheDaemon instances with one data resource.
    """

    cache_daemon_res_id_key = 'cache_daemon_res_id'
    DataResource = YABS_SERVER_CACHE_DAEMON_STUB_DATA

    def __init__(
        self,
        cache_daemon_executable_path,
        instance_types=None,  # Ignored in non-legacy mode with dump
        dump_path=None,
        data_dir='cache_daemon_data',
        log_subdir='cache_daemon',
        use_sub_path=False,
        services=None,
        start_on_creation=True,
        key_header='x-yabs-request-id',
        n_threads=1,
        max_queue_size=500,
        skip_services=(),
        services_ports=None,
        threads_per_service=None,
    ):
        self._executable_path = cache_daemon_executable_path

        self._services = services or {}
        self._use_sub_path = use_sub_path
        self._key_header = key_header
        self.instances = {}
        self.skip_services = set(skip_services)
        self.data_dir = os.path.abspath(data_dir)
        self._key_header = key_header
        self._services_ports = services_ports or {}

        threads_per_service = threads_per_service or {}

        if dump_path:
            if os.path.isdir(self.data_dir):
                logger.debug("Use existing cachedaemon data from %s", self.data_dir)
            else:
                logger.debug("Copy %s to %s", dump_path, self.data_dir)
                shutil.copytree(dump_path, self.data_dir)
                add_write_permissions_for_path(self.data_dir)  # FIXME make cached work without write permissions

            instance_types_from_dump = set(os.listdir(self.data_dir))
            logger.debug('Got instance types from dump: %s', instance_types_from_dump)
            if instance_types_from_dump == {'empty_stub_flag'}:  # horrible workaround due to impossibility of empty resources
                os.remove(os.path.join(self.data_dir, 'empty_stub_flag'))
                instance_types_from_dump = set()

            instance_types = instance_types_from_dump | set(self._services.keys())

        else:
            instance_types = set(instance_types or []) | set(self._services.keys())
            logger.info("Creating empty cachedaemon data dir")

        self.log_dir = os.path.join(get_logs_folder(), log_subdir)
        make_folder(self.log_dir)

        logger.debug('Got instance types: %s', instance_types)
        for instance_type in instance_types:
            if instance_type not in self.skip_services:
                self.create_instance(
                    instance_type,
                    start_on_creation=start_on_creation,
                    n_threads=threads_per_service.get(instance_type, n_threads),
                    max_queue_size=max_queue_size
                )

    def create_instance(self, instance_type, data_dir=None, *args, **kwargs):
        if instance_type in self.instances:
            raise RuntimeError('create_instance method was used to override existing cachedaemon')
        data_dir = data_dir or self.data_dir
        instance_data_dir = os.path.join(data_dir, instance_type)
        if not os.path.isdir(instance_data_dir):
            logger.debug('Instance %s has no data in %s, will create empty dir', instance_type, data_dir)
            make_folder(instance_data_dir)
        self.instances[instance_type] = CacheDaemon(
            executable_path=self._executable_path,
            inst_type=instance_type,
            data_root=data_dir,
            logs_root=self.log_dir,
            port=self._services_ports.get(instance_type),
            use_sub_path=self._use_sub_path,
            services=self._services,
            key_header=self._key_header,
            *args,
            **kwargs
        )
        return self.instances[instance_type]

    def get_or_create_instance(self, item, *args, **kwargs):
        if item not in self.instances:
            return self.create_instance(item, *args, **kwargs)
        else:
            return self.instances[item]

    def get_provided_services(self):
        provided_services = {}
        for inst in self.instances.itervalues():
            provided_services.update(inst.get_provided_services())
        return provided_services

    def get_provided_ext_tags(self):
        return set(self.instances.viewkeys())

    def get_ports_by_tag(self):
        return {inst_type: inst.port for inst_type, inst in self.instances.iteritems()}

    def start_instances(self):
        instances = self.instances.values()
        if not instances:
            return

        logger.debug("Start cachedaemon instances: %s", sorted([instance.inst_type for instance in instances]))
        for instance in instances:
            instance.start(wait=False)

        logger.debug("Wait until cachedaemon instances are alive")
        pool = ThreadPool(processes=min(len(instances), 4))
        pool.map(lambda instance: instance.wait(), instances)

    def shutdown_instances(self):
        instances = self.instances.values()
        if not instances:
            return

        logger.debug("Shut down cachedaemon instances: %s", sorted([instance.inst_type for instance in instances]))
        pool = ThreadPool(processes=min(len(instances), 4))
        pool.map(lambda instance: instance.shutdown(), instances)

    def __enter__(self):
        self.start_instances()
        logger.debug("Cachedaemon started")
        return self

    def __exit__(self, *_):
        self.shutdown_instances()
        logger.debug("Cachedaemon shut down")

    def check_access_log(self, broken_threshold=0.04, broken_thresholds=None):
        """
        Raise exception if error (non-1xx,2xx, or 3xx) fraction
        in access log of any cachedaemon instance exceeds broken_threshold
        """
        broken_thresholds = broken_thresholds or {}
        for inst_type, inst in self.instances.iteritems():
            inst.check_access_log(broken_thresholds.get(inst_type, broken_threshold))


class CacheDaemonStubStandalone(CacheDaemonStubSandboxNonIntegrated):
    def __init__(
        self,
        sync_resource,
        instance_types=None,  # Ignored in non-legacy mode with dump
        dump_res_id=None,  # May be sdk2.Resource, not int
        cache_daemon_res_id=None,  # May be sdk2.Resource, not int
        data_dir='cache_daemon_data',
        log_subdir='cache_daemon',
        use_sub_path=False,
        services=SERVICES_BY_TYPE,
        key_header='x-yabs-request-id',
    ):
        """
        Sync CacheDaemon data and executable, setup directories and start CacheDaemon instances.

        :param list instance_types: Example: ['bigb','turl'].
            Corresponding .cfg files should exist in config/ and ports should be specified in CacheDaemonStub.PORTS

        :param dump_res_id:
            YABS_SERVER_CACHE_DAEMON_STUB_DATA resource id. If None, CacheDaemons are started with empty cache.

        :param cache_daemon_res_id:
            CACHE_DAEMON resource id. Overrides CacheDaemon binary specified via attributes of the dump resource.
        """
        if cache_daemon_res_id is None:
            if dump_res_id is None:
                raise ValueError("Both cache_daemon_res_id and dump_res_id are None")

            dump_res = rest.Client().resource[dump_res_id].read()
            try:
                cache_daemon_res_id = dump_res['attributes'][self.cache_daemon_res_id_key]
            except KeyError:
                raise KeyError('Resource {} has no valid attribute "{}"'.format(dump_res_id, self.cache_daemon_res_id_key))

        self.cache_daemon_res_id = int(cache_daemon_res_id)

        if dump_res_id:
            logger.info("Syncing cachedaemon dump %s", dump_res_id)
            dump_res_dir = sync_resource(dump_res_id)
        else:
            dump_res_dir = None

        logger.info("Syncing cachedaemon executable %s" % self.cache_daemon_res_id)
        cache_daemon_executable_path = sync_resource(cache_daemon_res_id)
        super(CacheDaemonStubStandalone, self).__init__(
            instance_types=instance_types,
            dump_path=dump_res_dir,
            cache_daemon_executable_path=cache_daemon_executable_path,
            data_dir=data_dir,
            log_subdir=log_subdir,
            use_sub_path=use_sub_path,
            services=services,
            key_header=key_header,
        )

    def stop_and_store_dump_impl(self, create_resource, additional_attrs):
        """
        Stop cachedaemons and store data as a resource
        """
        self.shutdown_instances()
        attrs = {
            self.cache_daemon_res_id_key: self.cache_daemon_res_id,
            'ttl': 30,
            'provided_tags': ' '.join(self.instances.iterkeys())
        }
        attrs.update(additional_attrs or {})

        res = create_resource(
            description="Cache daemon data for yabs-server tests",
            resource_path=self.data_dir,
            resource_type=self.DataResource,
            arch='any',
            attributes=attrs,
        )
        return res.id


class CacheDaemonStub(CacheDaemonStubStandalone):
    # FIXME get rid of this class, switch to CacheDaemonStubStandalone
    def __init__(
        self, task, instance_types, dump_res_id=None, cache_daemon_res_id=None,
        data_dir='cache_daemon_data', key_header='x-yabs-request-id'
    ):
        self.task = task
        super(CacheDaemonStub, self).__init__(
            task.sync_resource,
            instance_types,
            dump_res_id=dump_res_id,
            cache_daemon_res_id=cache_daemon_res_id,
            data_dir=data_dir,
            key_header=key_header,
        )

    def stop_and_store_dump(self, create_resource, additional_attrs=None):
        return self.stop_and_store_dump_impl(self.task.create_resource, additional_attrs)


class CachedaemonStartFailure(Exception):
    pass


class CacheDaemon(object):
    def __init__(
        self,
        executable_path,
        inst_type,
        data_root,
        logs_root,
        port=None,
        use_sub_path=False,
        services=SERVICES_BY_TYPE,
        key_header='x-yabs-request-id',
        start_on_creation=True,
        n_threads=1,
        max_queue_size=500,
    ):
        self.inst_type = inst_type
        self._key_header = key_header

        if port is None:
            self._port, self._reserve_port_socket = reserve_port()
        else:
            self._port = port
        if services:
            self._services = services.get(inst_type, [inst_type])
        else:
            self._services = [inst_type]

        prefix = '{}_{}'.format(inst_type, self.port)

        logger.debug("Creating %s cachedaemon stub on port %s", inst_type, self.port)
        cfg_path = os.path.join(logs_root, prefix + '.cfg')
        write_cachedaemon_config(cfg_path, use_sub_path, key_header, n_threads, max_queue_size)
        data_dir = os.path.join(data_root, inst_type)
        self._log_dir = os.path.join(logs_root, prefix)
        make_folder(self._log_dir)

        self.access_log_path = os.path.join(self._log_dir, 'access.log')
        self._log_prefix = 'cachedaemon_{}_out'.format(prefix)

        self._cmdline = [
            executable_path,
            cfg_path,
            '-v', 'data_dir={}'.format(data_dir),
            '-v', 'server_port={}'.format(self.port),
            '-v', 'log_dir={}'.format(self._log_dir),
        ]
        self._started = False
        if start_on_creation:
            self.start()

    def start(self, wait=True, timeout=60):
        if self._started:
            return

        with open(os.path.join(self._log_dir, self._log_prefix), 'w') as stdout:
            self.process = subprocess.Popen(self._cmdline, stdout=stdout, stderr=subprocess.STDOUT)

        if wait:
            self.wait(timeout)

        self._started = True
        logger.debug("cachedaemon instance \"%s\" on port %s successfully started", self.inst_type, self.port)

    def wait(self, timeout=60):
        """Wait until instance starts and ready to accept requests

        :param timeout: Wait for `timeout` seconds until instance start, defaults to 60
        :type timeout: int, optional
        :raises CachedaemonStartFailure: Raises if instance is not started in timeout
        """
        start = time.time()
        sleep_timeout = 1
        while not self.is_alive():
            if time.time() - start > timeout:
                logger.error(
                    "cachedaemon instance \"%s\" on port %s failed to start in %s seconds",
                    self.inst_type, self.port, timeout)
                raise CachedaemonStartFailure(self.inst_type)

            logger.debug(
                "Failed to ping cachedaemon instance \"%s\" on port %s, retry in %s seconds",
                self.inst_type, self.port, sleep_timeout
            )
            time.sleep(sleep_timeout)

    def is_alive(self):
        version_url = "http://localhost:{}/admin?action=version".format(self.port)
        try:
            response = requests.get(version_url)
        except requests.exceptions.ConnectionError:
            return False
        return response.ok

    def __enter__(self):
        self.start()
        return self

    def __exit__(self, *_):
        self.shutdown()

    @property
    def port(self):
        return self._port

    @property
    def pid(self):
        return self.process.pid

    def get_provided_services(self):
        return dict.fromkeys(self._services, self.port)

    def check_access_log(self, broken_threshold):
        counts = checklog.get_code_counts(self.access_log_path)
        total_count = sum(counts.values())
        bad_count = sum(counts[code] for code in counts if _is_bad_http_code(code))
        discounted_bad_count = max(0, bad_count - 3 * bad_count**0.5)
        if discounted_bad_count > total_count * broken_threshold and self.inst_type not in ('yacoland', 'new_saas_location', 'yacofast_ssp'):  # FIXME: problem with yacoland
            raise TaskFailure(
                "Too many errors in the access log of the {} stub:\n{}".format(
                    self.inst_type,
                    '\n'.join("{}: {}".format(key, val) for key, val in counts.iteritems())
                )
            )

    def shutdown(self, timeout=30):
        if not self._started:
            logger.warning('Trying to shutdown cachedaemon that is not started, possible bug')

        shutdown_url = 'http://localhost:{}/admin'.format(self.port)
        try:
            requests.get(shutdown_url, params={'action': 'shutdown'})
        except Exception:
            logger.info("Shutdown request %s for cachedaemon %s failed", shutdown_url, self.inst_type)
            raise

        deadline = time.time() + timeout
        pause = 0.25
        killed = False
        while True:
            retcode = self.process.poll()
            if retcode == 0 or (killed and retcode == -9):
                self._started = False
                return
            if retcode is not None and not (killed and retcode == -9):
                raise TaskError("Cachedaemon instance {} exit with code {}".format(self.inst_type, retcode))
            if time.time() > deadline:
                logger.warning("Timed out while waiting for {} cachedaemon exit, killing it".format(self.inst_type))
                self.process.kill()
                killed = True
            time.sleep(pause)
            pause = max(5, pause * 1.5)


def _is_bad_http_code(code):
    try:
        c = int(code)
    except (ValueError, TypeError):
        return True
    return c < 100 or c >= 400


def write_cachedaemon_config(cfg_path, use_sub_path, key_header, n_threads, max_queue_size):
    with open(cfg_path, 'w') as cfg:
        cfg.write(_COMMON_CACHEDAEMON_CONFIG_TEMPLATE.format(use_sub_path=int(use_sub_path), key_header=key_header, n_threads=n_threads, max_queue_size=max_queue_size))


_COMMON_CACHEDAEMON_CONFIG_TEMPLATE = """
instance = {{
     user_error_log = {{
         file_name = {{ log_dir .. '/usererror.log' }};
         max_log_queue_size = 0;
     }};
     access_log = {{
         file_name = {{ log_dir .. '/access.log' }};
         max_log_queue_size = 0;
     }};
     stat_log = {{
         file_name = {{ log_dir .. '/stat.log' }};
         max_log_queue_size = 0;
     }};
     server = {{
         port = {{ server_port }};
         threads = {{ {n_threads} }};
         max_queue_size = {{ {max_queue_size} }};
     }};
     monitoring = {{
         answer_time_buckets = '0.1, 0.01, 0.001, 0.0001, 0.00001';
     }};
     modules = {{
         yabs_stub = {{

             key_builder = {{
                 nonsignificant_cgi_params = '*';
                 include_headers = '{key_header}';
                 use_sub_path = {use_sub_path};
             }};

            collection = {{
                id = '*';
                allow_sub_path = 1;
            }};

             storage = {{
                 dir = {{ data_dir }};
                 arenas = 1;
                 compression = 'lzf';
                 memory_limit = '50M';
                 file_cache_size = '20G';
                 block_size = '1K';
             }};
         }};

     }};
}};
"""
