# -*- coding: utf-8 -*-
import logging
import os
import json
import pickle
import sys
import re
import time

from sandbox import sandboxsdk
from sandbox import projects
from sandbox import common

from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess as sp


def normalize_wd():
    run_sh('chown -R `stat -c "%U:%G" .` .')


def _encode(data, data_type='raw'):
    if data is None:
        pass
    elif data_type == 'json':
        data = json.dumps(data)
    elif data_type == 'pickle':
        data = pickle.dumps(data)
    return data


def _decode(data):
    re_out = re.search(r'#%%([\w \t]+)\n(.*)\n#%%', data, re.DOTALL | re.IGNORECASE)
    if re_out:
        (out_type, out_data) = re_out.groups()
        out_type = out_type.strip().lower()
        if out_type == 'json':
            data = json.loads(out_data)
        elif out_type == 'pickle':
            data = pickle.loads(out_data)
        elif out_type in ['raw', 'txt', 'text']:
            data = out_data
        else:
            raise ValueError("Unknown type: {}".format(out_type))

    return data


def _run_cmd(cmd, stdin=None, env=None, tmp_error_re=None, retry=0):
    # We detect TemporaryError by parsing STDERR,
    # therefore we must set LC_ALL=C,
    # otherwise we might get message in Russian and not recognize a temporary error.
    env = env or os.environ.copy()
    env['LC_ALL'] = 'C'

    stdout, stderr, returncode = '', '', 0

    cmd_str = ' '.join(cmd)
    for iter_num in xrange(retry+1):
        time.sleep(iter_num**3)
        logging.info("Attempt #%s to run %s", iter_num, cmd_str)
        p = sp.popen_and_communicate(cmd, stdin=sp.PIPE, stdout=sp.PIPE, stderr=sp.PIPE, env=env, input=stdin)
        stdout = p.stdout
        stderr = p.stderr
        returncode = p.returncode
        if returncode == 0:
            break

    if returncode:
        exception_type = common.errors.TaskError
        if tmp_error_re and re.search(tmp_error_re, stderr, re.MULTILINE):
            exception_type = common.errors.TemporaryError
        raise exception_type(
            "Process {} failed with exit code: {}\nSTDIN:\n{}\nSTDOUT:\n{}\nSTDERR:\n{}".format(
                cmd_str, returncode, repr(stdin), stdout, stderr
            )
        )

    stdout = _decode(stdout)
    logging.debug("STDOUT of %s: %s", cmd_str, stdout)
    logging.debug("STDERR of %s: %s", cmd_str, stderr)
    return stdout, stderr


def run_py(script_name, args=[], data=None, data_type='pickle', paths=[],
           py_path=sys.executable, tmp_error_re=r'^[\w\.]*TemporaryError: ', retry=0):
    from sandbox import yasandbox
    paths += [
        os.path.dirname(os.path.dirname(yasandbox.__file__)),
        os.path.dirname(os.path.dirname(sandboxsdk.__file__)),
        os.path.dirname(os.path.dirname(projects.__file__)),
    ]
    env = os.environ.copy()
    if env.get("PYTHONPATH"):
        paths.append(env["PYTHONPATH"])

    cmd = [
        'sudo', '-H', "PYTHONPATH={}".format(":".join(paths)),
        py_path, '-u', script_name,
    ] + args

    return _run_cmd(cmd, _encode(data, data_type), env=env, tmp_error_re=tmp_error_re, retry=retry)[0]


def run_sh(cmd, data=None, data_type='raw', check_exit_code=True,
           tmp_error_re=r'^[\w\.]*TemporaryError: ', retry=0):
    if isinstance(cmd, (list, tuple)):
        cmd = ' '.join(cmd)
    cmd = ['sudo', 'bash', '-c', cmd]
    stdin = _encode(data, data_type)

    try:
        stdout, _ = _run_cmd(cmd, stdin, None, tmp_error_re, retry)
    except Exception:
        if check_exit_code:
            raise
        logging.exception("Swallowing an exception in run_sh (check_exit_code=False)")
        stdout = None

    return stdout
