# -*- coding: utf-8 -*-
"""
Simple module to use Yabs Juggler.

Usage example:
>>> from juggler import ServiceStatus, Juggler, STATUS_OK, STATUS_CRIT
>>> ok = ServiceStatus('sbyt-alyx', 'lbyt_jobs', STATUS_OK)
>>> crit = ServiceStatus('sbyt-hahn', 'lbyt_jobs', STATUS_CRIT, 'Unknown job "test"')
>>> Juggler().send([ok, crit])
"""

import logging

import requests

STATUS_OK = 'OK'
STATUS_WARN = STATUS_WARNING = 'WARN'
STATUS_CRIT = STATUS_CRITICAL = 'CRIT'

JUGGLER_SERVERS = [
    'juggler-push.search.yandex.net:80',
]


class ServiceStatus(object):
    def __init__(self, host, service, status, description=None):
        if status not in (STATUS_OK, STATUS_WARN, STATUS_CRIT):
            raise ValueError('"{}" is not a valid Juggler status'.format(status))
        if not service:
            raise ValueError('Service name must not be empty')
        if not host:
            raise ValueError('Host name must not be empty')

        self.host = host
        self.service = service
        self.status = status
        self.description = description

        if self.description is None:
            self.description = self.status.title()

    def __str__(self):
        return 'ServiceStatus: {} is {} ({}) on {}'.format(
            self.service, self.status, self.description, self.host,
        )

    def as_dict(self):
        return dict(
            host=self.host,
            service=self.service,
            status=self.status,
            description=self.description,
        )


class Juggler(object):
    def __init__(self, hosts=JUGGLER_SERVERS, timeout=5, logger=None):
        if logger is None:
            logger = logging
        self.hosts = hosts
        self.timeout = timeout
        self.logger = logger

    def send(self, service_statuses):
        for host in self.hosts:
            resp = None
            try:
                with requests.Session() as session:
                    session.max_redirects = 0  # to raise exception in case of 3xx code
                    for service_status in service_statuses:
                        resp = None  # to not print previous response text in case of error in get
                        resp = session.get(
                            'http://{}/juggler-fcgi.py'.format(host),
                            params=service_status.as_dict(),
                            timeout=self.timeout,
                        )
                        resp.raise_for_status()  # raise exception in case of 4xx or 5xx code
            except Exception as err:
                resp_text = ''
                if resp is not None:
                    resp_text = '; response body: {}'.format(resp.text)
                self.logger.warn('Send metrics to %s failed: %s%s', host, err, resp_text)
