# -*- coding: utf-8 -*-

ALL_YT_CLUSTERS = (
    'freud', 'zeno',
    'hahn', 'arnold',
    'seneca-sas', 'seneca-vla',
)

DEFAULT_ERR_EXPIRATION = 160  # hours
DEFAULT_TASK_FREQUENCY = 150  # seconds
DEFAULT_KEEP_COMPLETED_TASKS = 160  # hours
DEFAULT_MAX_SMOOTHING_SLEEP = 15  # seconds. Max sleep before first iteration in READER_JOBS
DEFAULT_TIMEOUT_COEFFICIENT = 7  # How long could take a job comparing with frequency


def add_stat_jobs(
    manager_jobs, reader_jobs,
    logtype, yt, lb,
    max_dl=2, quota=1024, count=15,
    keep_completed_tasks=DEFAULT_KEEP_COMPLETED_TASKS,
    client_account='',
    bs_log=True,
    **reader_params
):
    if yt not in ALL_YT_CLUSTERS:
        raise ValueError('Unknown Yt cluster "{}", may be typo?'.format(yt))

    name = 'stat-{}-{}'.format(logtype, yt)
    if name in manager_jobs:
        raise ValueError('Manager job for {} at {} is already defined'.format(logtype, yt))
    if name in reader_jobs:
        raise ValueError('Reader job for {} at {} is already defined'.format(logtype, yt))

    logtype = '{}-log'.format(logtype)
    if bs_log:
        logtype = 'bs-' + logtype
    manager_jobs[name] = {
        'dest_path': '/yabs-rt/' + logtype,
        'logtypes': logtype,
        'lb': lb,
        'yt': yt,
    }
    reader_jobs[name] = {
        'client': '{}ytstat-{}'.format(client_account, yt),
        'max_simultaneous_downloads': max_dl,
        'keep_completed_tasks': keep_completed_tasks,
        'keep_sequential': False,
        'quota': quota,
        'count': count,
    }
    reader_jobs[name].update(reader_params)


def add_xurma_jobs(
    manager_jobs, reader_jobs,
    namespace, logtypes, yt_lb_mapping,
    use_unimportant_client=False,
    yt_pool_params=None,
    use_fat_keys=False,
    quota=1024, count=12,
    **reader_params
):
    manager_job_name = 'xurma-{}-prod'.format(namespace.replace('/', '-'))
    manager_jobs[manager_job_name] = {
        'dest_path': '/xurma/{}/output'.format(namespace),
        'logtypes': logtypes,
        'yt': sorted(yt_lb_mapping),
        'lb': sorted(set(lb for lb in yt_lb_mapping.itervalues() if lb is not None)),
    }

    for yt, lb in yt_lb_mapping.iteritems():
        reader_job_name = 'xurma-{}-{}'.format(namespace.replace('/', '-'), yt)
        reader_job = reader_jobs[reader_job_name] = {
            'job': manager_job_name,
            'yt': yt,
        }
        if lb is None:
            reader_job['only_for_quorum'] = True
            continue

        if lb != 'lbkx':
            client = 'xurma'
        else:
            importance = ('unimportant' if use_unimportant_client else 'important')
            client = 'xurma@consumers@{}@{}'.format(importance, yt)

        reader_job.update({
            'lb': lb,
            'client': client,
            'max_simultaneous_downloads': 2,
            'keep_sequential': True,
            'quota': quota,
            'count': count,
            'stats_path': '/xurma/{}/stats'.format(namespace),
            'file_storage': '//home/antifraud/xurma/file-storage',
            'max_smoothing_sleep': 3,
        })
        if use_fat_keys:
            reader_job['fat_keys_path'] = '/xurma/{}/fat_keys'.format(namespace)

        if yt_pool_params and yt in yt_pool_params:
            if yt_pool_params[yt]:
                reader_job.update(yt_pool_params[yt])

        reader_job.update(reader_params)


def make_yt_pool_params(pool_trees_map, use_isolated_pool=None, custom_pools=None):
    yt_pool_params = {}

    for yt, pool_tree in pool_trees_map.iteritems():
        cluster_pool_params = yt_pool_params[yt] = {}
        if pool_tree:
            cluster_pool_params['pool_tree'] = pool_tree
        if use_isolated_pool:
            cluster_pool_params['pool'] = 'xurma-isolated-{}-core'.format(use_isolated_pool)

    if custom_pools:
        for yt, pool in custom_pools.iteritems():
            yt_pool_params[yt] = {'pool': pool}

    return yt_pool_params


def prepare_and_validate_config(logtypes, manager_jobs, reader_jobs):
    # Processing LOGTYPES
    for logtype, config in logtypes.iteritems():
        _ensure_field(config, 'lb_name', basestring, logtype, default=logtype)
        _ensure_field(config, 'max_offsets_per_task', int, logtype)
        _ensure_field(config, 'max_offsets_per_pqclient', int, logtype, default=0)
        _ensure_field(config, 'frequency', int, logtype, default=DEFAULT_TASK_FREQUENCY)
        _ensure_field(config, 'data_format', basestring, logtype, default='tskv')

    # Processing MANAGER_JOBS
    for job, config in manager_jobs.iteritems():
        _ensure_field(config, 'use_time_binding', bool, job, default=False)
        for field in ('logtypes', 'lb', 'yt'):
            _ensure_field(config, field, (basestring, tuple, list, set), job)
            if isinstance(config[field], basestring):
                config[field] = [config[field]]
            elif not config[field]:
                raise ValueError('Field "{}" for {} can not be empty'.format(field, job))
        _ensure_field(config, 'dest_path', basestring, job)
        for logtype in config['logtypes']:
            if logtype not in logtypes:
                raise ValueError('Logtype {} not found for job {}'.format(logtype, job))

    # Processing READER_JOBS
    for job, config in reader_jobs.iteritems():
        _ensure_field(config, 'job', basestring, job, default=job)
        if config['job'] not in manager_jobs:
            raise ValueError('Manager job {} not found for reader job {}'.format(
                config['job'], job,
            ))
        _ensure_field(config, 'only_for_quorum', bool, job, default=False)
        dependent_fields = ['yt']
        if not config['only_for_quorum']:
            dependent_fields.append('lb')
        for field in dependent_fields:
            manager_job_values = manager_jobs[config['job']][field]
            if field not in config and len(manager_job_values) != 1:
                raise ValueError(
                    'Reader job {} can not get {} config from corresponding manager job {},'
                    ' because there are more than one element'.format(
                        job, field, config['job'],
                    )
                )
            _ensure_field(config, field, basestring, job, default=manager_job_values[0])
            if config[field] not in manager_job_values:
                raise ValueError(
                    'Reader job {} has {} that is not controlled'
                    ' by the corresponding manager job {}'.format(
                        job, field, config['job'],
                    )
                )
        _ensure_field(config, 'err_expiration_time', int, job, DEFAULT_ERR_EXPIRATION)
        _ensure_field(config, 'keep_completed_tasks', int, job, DEFAULT_KEEP_COMPLETED_TASKS)
        _ensure_field(config, 'max_smoothing_sleep', int, job, DEFAULT_MAX_SMOOTHING_SLEEP)
        _ensure_field(config, 'timeout_coefficient', (int, float), job, DEFAULT_TIMEOUT_COEFFICIENT)
        _ensure_field(config, 'pool', basestring, job, default='')
        _ensure_field(config, 'pool_tree', basestring, job, default='')
        _ensure_field(config, 'stats_path', basestring, job, default='')
        _ensure_field(config, 'yt_log_level', basestring, job, default='')
        _ensure_field(config, 'fat_keys_path', basestring, job, default='')
        _ensure_field(config, 'file_storage', basestring, job, default='')
        _ensure_field(config, 'tmp_input_table_attributes', dict, job, default={})
        if config['only_for_quorum']:
            continue
        _ensure_field(config, 'client', basestring, job)
        _ensure_field(config, 'count', int, job)
        _ensure_field(config, 'keep_sequential', bool, job)
        _ensure_field(config, 'max_simultaneous_downloads', int, job)
        _ensure_field(config, 'quota', int, job)


def _ensure_field(config, field, types, name, default=None):
    if field not in config:
        if default is None:
            raise ValueError('Field "{}" is required for {}'.format(field, name))
        config[field] = default
    if not isinstance(config[field], types):
        raise ValueError('Field "{}" in {} must be an instance of {}'.format(field, name, types))
