# -*- coding: utf-8 -*-

import os

from .utils import get_job_logger
from .config import COMMON_LB_CLIENT, LB_REQUEST_TIMEOUT, LOGTYPES

PRESTABLE_SUFFIX = '.prestable'


class LbClient(object):

    def __init__(self, cluster):
        self.localmode = os.path.isfile('/etc/testing')
        self.name = cluster
        prestable_cluster = self._get_prestable_cluster(cluster)
        if prestable_cluster:
            self.proxy = prestable_cluster + '.logbroker-prestable.yandex.net'
        else:
            self.proxy = cluster + '.logbroker.yandex.net'
        self.offsets = {}

    def read_offsets(self, logtypes):
        self.offsets = {}  # invalidate cache
        logger = get_job_logger()
        logger.info('Reading offsets from lb cluster %s via %s', self.name, self.proxy)
        from requests import Session
        with Session() as session:
            session.max_redirects = 0  # to raise exception in case of 3xx code
            for logtype in logtypes:
                resp = None
                try:
                    lb_name = LOGTYPES.get(logtype, {}).get('lb_name', logtype)
                    params = {'client': COMMON_LB_CLIENT, 'log-type': lb_name}
                    ident = LOGTYPES.get(logtype, {}).get('ident')
                    if ident is not None:
                        params['ident'] = ident

                    resp = session.get(
                        'http://{}:8999/pull/offsets'.format(self.proxy),
                        params=params,
                        timeout=LB_REQUEST_TIMEOUT,
                    )
                    resp.raise_for_status()  # raise exception in case of 4xx or 5xx code
                    self._parse_offsets_info(logtype, resp)
                except Exception as err:
                    resp_text = ''
                    if resp is not None:
                        resp_text = '; response body: {}'.format(resp.text)
                    logger.warn(
                        'Failed to read offsets for logtype %s from lb cluster %s: %s%s',
                        logtype, self.name, err, resp_text,
                    )

    def commit_offsets(self, client, offsets_info):
        logger = get_job_logger()
        if self.localmode:
            logger.info('Commiting offsets is disabled in local mode')
            return
        data = ''.join(
            # -1 because LB wants last read offset, while we store first offset to read
            '{}:{}\n'.format(toppart, offset-1)
            for logtype in offsets_info
            for toppart, offset in offsets_info[logtype].iteritems()
        )
        logger.info('Commiting offsets to lb cluster %s via %s', self.name, self.proxy)
        from requests import Session
        with Session() as session:
            session.max_redirects = 0  # to raise exception in case of 3xx code
            resp = None
            try:
                params = {'client': client}
                resp = session.post(
                    'http://{}:8999/pull/commit'.format(self.proxy),
                    params=params,
                    data=data,
                    timeout=LB_REQUEST_TIMEOUT,
                )
                resp.raise_for_status()  # raise exception in case of 4xx or 5xx code
                assert resp.text == 'ok\n', 'Unexpected response'
            except Exception as err:
                resp_text = ''
                if resp is not None:
                    resp_text = '; response body: {}'.format(resp.text)
                logger.warn(
                    'Failed to commit offsets to lb cluster %s: %s%s',
                    self.name, err, resp_text,
                )
                return False
            logger.info('Offsets commited to lb cluster %s', self.name)
            return True

    @staticmethod
    def _get_prestable_cluster(cluster):
        if cluster.endswith(PRESTABLE_SUFFIX):
            return cluster[:-len(PRESTABLE_SUFFIX)]
        return None

    def _parse_offsets_info(self, logtype, response):
        logtype_info = {}
        fields = response.headers['Fields'].strip().split('\t')
        for raw_line in response.text.strip().split('\n'):
            offset_info = dict(zip(fields, raw_line.strip().split('\t')))
            logtype_info[offset_info['topic:partition']] = long(offset_info['logSize'])
        if logtype_info:
            self.offsets[logtype] = logtype_info
