# -*- coding: utf-8 -*-

import logging
import os
import threading
from datetime import datetime

from sandbox.sdk2.paths import get_logs_folder


class ThreadingSuccessTracker(object):

    def __init__(self, func):
        self.success = None  # all calls succeeded
        self.fail = None  # all calls failed
        self.lock = threading.Lock()
        self.func = func

    def __call__(self, *args, **kwargs):
        self._on_call()
        try:
            result = self.func(*args, **kwargs)
        except Exception:
            result = False
        if result is None:
            result = True
        with self.lock:
            if result:
                self.fail = False
            else:
                self.success = False

    def _on_call(self):
        # Inits values on first call.
        with self.lock:
            if self.fail is None:
                self.fail = True
            if self.success is None:
                self.success = True


def forget_attributes(tables):
    # Convert YsonString list to usual string list to make 'in' check possible.
    return [str(table) for table in tables]


def parse_yt_datetime(dt):
    return datetime.strptime(dt[:19], '%Y-%m-%dT%H:%M:%S')


def get_job_logger(job=None):
    logger = logging.getLogger('lbyt_reader')
    if logger.handlers:
        return logger
    # First call, initializing
    if not job:
        raise ValueError('Can not initialize logger without job name')
    handler = logging.FileHandler(os.path.join(get_logs_folder(), job + '.log'))
    handler.setFormatter(logging.Formatter(
        fmt='%(asctime)s %(levelname)-7s (%(module)s.%(funcName)s) %(message)s',
        datefmt='%Y/%m/%d %H:%M:%S',
    ))
    handler.setLevel(logging.INFO)
    logger.addHandler(handler)
    return logger


def do_for_clusters(clusters, operation, *args, **kwargs):
    def do_operation(cluster):
        return getattr(cluster, operation)(*args, **kwargs)

    get_job_logger().info(
        'Doing %s for clusters: %s',
        operation, ', '.join(cluster.name for cluster in clusters),
    )
    threads = []
    do_operation_tracker = ThreadingSuccessTracker(do_operation)
    for cluster in clusters:
        assert hasattr(cluster, operation),\
            'cluster {} does not have "{}"'.format(cluster.name, operation)
        thread = threading.Thread(target=do_operation_tracker, args=(cluster,))
        threads.append(thread)
        thread.start()
    for thread in threads:
        thread.join()
    return do_operation_tracker.success
