# -*- coding: utf-8 -*-
import os.path
import logging
import random
import copy
import json
import tarfile

from sandbox import sdk2

from sandbox import common
from sandbox.common import rest
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr

from sandbox.common.errors import TaskStop

from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxIntegerParameter, SandboxSelectParameter, SandboxBoolParameter, TaskSelector

from sandbox.projects.resource_types import YANDEX_SURF_DATA, CACHE_DAEMON
from sandbox.projects.common.utils import get_or_default, get_and_check_last_released_resource_id

from sandbox.projects.common.yabs.task import TaskWrapper
from sandbox.projects.common.yabs import cachedaemon

from sandbox.projects.common.yabs.server.util.general import CustomAssert

from sandbox.projects.yabs.qa.sut.bases_provider import BinBasesProvider, build_bin_base_list
from sandbox.projects.yabs.qa.sut.factory import YabsServerFactoryStandalone, META_MODES

from sandbox.projects.yabs.qa.resource_types import (
    BS_RELEASE_TAR,
    BS_RELEASE_YT,
    YABS_SERVER_CACHE_DAEMON_STUB_DATA,
    YABS_SERVER_B2B_BINARY_BASE,
    YABS_SERVER_B2B_SPEC,
)

from sandbox.projects.yabs.cs import (
    YabsMlModelsConfig,
)


class YabsMlModelsConfigResource(ResourceSelector):
    name = 'yabs_ml_models_config_resource'
    description = 'Resource with Models\' config file'
    required = False
    multiple = False
    resource_type = YabsMlModelsConfig


class ServerResource(ResourceSelector):
    """Resource with yabs-server & yabs_mkdb"""
    name = 'server_resource'
    description = 'Resource with yabs-server & yabs_mkdb (last stable BS_RELEASE_TAR if empty)'
    required = False
    multiple = False
    resource_type = BS_RELEASE_TAR


class CSResource(ResourceSelector):
    """Resource with yabs-server & yabs_mkdb"""
    name = 'cs_resource'
    description = 'Resource with yabs-cs (last stable BS_RELEASE_YT if empty)'
    required = False
    multiple = False
    resource_type = BS_RELEASE_YT


class PublishSpec(SandboxBoolParameter):
    name = 'publish_spec'
    description = 'Publish spec to be used for local debugging'
    default_value = True
    required = True


class PropagateTags(SandboxBoolParameter):
    name = 'propagate_tags'
    description = 'Use own tags when creating child tasks'
    default_value = False


class GlobalKey(object):
    def __init__(self, res_id):
        resource_data = rest.client().resource[res_id].read()
        self.key_data = resource_data['attributes'].get('global_key')
        self.key_type = resource_data['attributes'].get('global_key_type')

    def get_global_key_search_attrs_dict(self, additional_args={}):
        CustomAssert(self.global_key, 'Missing global key', AssertionError)
        return_dict = {
            'global_key': self.key_data,
            'global_key_type': self.key_type
        }
        return_dict.update(additional_args)
        return return_dict

    def get_global_key_search_attrs_str(self, additional_args={}):
        return json.dumps(self.get_search_attrs_dict(additional_args))

    def __nonzero__(self):
        return self.key_type is not None and self.key_data is not None


class ServerMkdbInstallTask(TaskWrapper, object):
    """
    Parent class for tasks that install yabs_server/yabs_mkdb
    """
    input_parameters = (ServerResource, PublishSpec, PropagateTags)

    def create_subtask(self, *args, **kwargs):
        propagate_tags = get_or_default(self.ctx, PropagateTags)
        input_parameters = copy.copy(kwargs.get('input_parameters', {}))
        input_parameters[PropagateTags.name] = propagate_tags
        new_kwargs = copy.copy(kwargs)
        new_kwargs['input_parameters'] = input_parameters
        if propagate_tags:
            client = rest.Client()
            own_tags = set(client.task[self.id].read()['tags'])
            subtask_tags = set(kwargs.get('tags', []))
            new_kwargs['tags'] = sorted(own_tags | subtask_tags)
        subtask = TaskWrapper.create_subtask(self, *args, **new_kwargs)
        return subtask

    @staticmethod
    def get_global_key_by_id(resource_id):
        resource_data = rest.Client().resource[resource_id].read()
        global_key = resource_data['attributes'].get('global_key')
        global_key_type = resource_data['attributes'].get('global_key_type')
        if not global_key or not global_key_type:
            return None
        else:
            return {
                'global_key': resource_data['attributes'].get('global_key'),
                'global_key_type': resource_data['attributes'].get('global_key_type')
            }

    @property
    def global_key(self):
        try:
            return self._global_key
        except AttributeError:
            self._global_key = self.get_global_key_by_id(self.server_res_id)
            return self._global_key

    def get_global_key_search_dict(self, additional_attrs):
        search_dict = self.global_key.copy()
        search_dict.update(additional_attrs)
        return search_dict

    def get_global_key_search_str(self, additional_attrs):
        return json.dumps(self.get_global_key_search_dict(additional_attrs))

    @property
    def server_res_id(self):
        try:
            return self._server_res_id
        except AttributeError:
            res_id = self.ctx.get(ServerResource.name)
            res_id = res_id or self.ctx.get('mkdb_resource')  # For backward compatibility
            res_id = res_id or get_and_check_last_released_resource_id(BS_RELEASE_TAR)
            self._server_res_id = res_id
            return res_id

    @property
    def spec(self):
        try:
            return self._spec
        except AttributeError:
            self._spec = {
                'resource_data': {
                    'ServerResource': self._single_resource_spec(self.server_res_id)
                }
            }
            if self.debug_server_res_id is not None:
                self._spec['resource_data'].update({
                    'ServerResourceDebug': self._single_resource_spec(self.debug_server_res_id)
                })
            return self._spec

    def _single_resource_spec(self, res_id):
        return {
            'type': 'res_id',
            'data': int(res_id)
        }

    def _publish_spec(self):
        with open('spec.json', 'w') as f:
            f.write(json.dumps(self.spec, indent=4))
        res = self.create_resource(
            'Spec resource',
            resource_path='spec.json',
            resource_type=YABS_SERVER_B2B_SPEC,
            arch='any'
        )
        self.mark_resource_ready(res.id)

    def get_server(self):
        """
        Sync and unpack server if not synced&unpacked yet
        Return path to server
        """
        try:
            return self._server_path
        except AttributeError:
            self._server_path = self._sync_server()
            return self._server_path

    @property
    def debug_server_res_id(self):
        try:
            return self._debug_server_res_id
        except AttributeError:
            try:
                CustomAssert(self.global_key, 'Missing global key', AssertionError)
                search_attrs_str = self.get_global_key_search_str({'build_mode': 'debug'})
                debug_server_data = rest.Client().resource.read({'attrs': search_attrs_str, 'limit': 100, 'type': 'BS_RELEASE_TAR'})
                CustomAssert(len(debug_server_data['items']) > 0, 'No items found', AssertionError)
                self._debug_server_res_id = debug_server_data['items'][0]['id']
            except AssertionError as e:
                logging.warning(e, exc_info=True)
                self._debug_server_res_id = None
            return self._debug_server_res_id


GROUP_TEST_DATA = 'Test data'


class SurfDataResource(ResourceSelector):
    """Resource with yabs-server & yabs_mkdb"""
    name = 'surf_data_resource'
    description = 'Resource with yandex-surf-data'
    group = GROUP_TEST_DATA
    default_value = 121355487
    required = True
    multiple = False
    resource_type = YANDEX_SURF_DATA


class DBResourceList(ResourceSelector):
    """Binary base list parameter. """
    name = 'db_resource_list'
    multiple = True
    resource_type = YABS_SERVER_B2B_BINARY_BASE
    state = (ctr.State.READY, ctr.State.NOT_READY)
    description = "Binary bases"
    group = GROUP_TEST_DATA


class DBResourceTask(TaskSelector):
    name = 'db_resource_task'
    task_type = 'YABS_SERVER_MAKE_BIN_BASES'
    description = "Use binary bases generated by this task"
    group = GROUP_TEST_DATA


class TmpfsSize(SandboxIntegerParameter):
    name = 'tmpfs_size'
    description = 'Size of data directory filesystem (GB). Set to zero to disable using tmpfs.'
    default_value = 165
    required = True


class ServerTask(ServerMkdbInstallTask):
    """
    Parent class for tasks that run yabs_server.
    """

    input_parameters = ServerMkdbInstallTask.input_parameters + (
        SurfDataResource,
        DBResourceList,
        DBResourceTask,
        TmpfsSize,
    )

    max_restarts = 10

    @property
    def spec(self):
        try:
            return self._spec
        except AttributeError:
            self._spec = super(ServerTask, self).spec
            db_resource_data_list = []
            for res_id in self.ctx[DBResourceList.name]:
                db_resource_data_list.append(self._single_resource_spec(res_id))
            surf_data = self._single_resource_spec(self.ctx[SurfDataResource.name])
            self._spec['resource_data'].update({
                'DBResourceList': db_resource_data_list,
                'SurfDataResource': surf_data
            })
            return self._spec

    def postprocess(self):
        resource_id = common.rest.Client().resource.read(task_id=self.id,
                                                         type='TASK_LOGS',
                                                         limit=10,
                                                         order='-id')['items'][0]['id']
        sdk2.Task.current.log_resource = sdk2.Resource[resource_id]
        sdk2.helpers.ProcessRegistry.finish()
        super(ServerTask, self).postprocess()

    def on_enqueue(self):
        TaskWrapper.on_enqueue(self)
        tmpfs_size = self.ctx.get(TmpfsSize.name, 0)
        if tmpfs_size > 0:
            self.ramdrive = self.RamDrive(
                ctm.RamDriveType.TMPFS,
                tmpfs_size * 2**10,
                None
            )
        else:
            self.ramdrive = None

    ABANDONED_HOSTS_CTX_KEY = '__abandoned_hosts'

    def abandon_host(self, msg):
        hostname = self.get_hostname()
        abandoned = self.ctx.get(self.ABANDONED_HOSTS_CTX_KEY, [])
        abandoned.append(hostname)
        time_to_wait = min(300, (1.0 + random.random()) * 2 ** len(abandoned))
        self.ctx[self.ABANDONED_HOSTS_CTX_KEY] = abandoned
        if self.ctx.get('new_life_cycle', False):
            raise TaskStop("Abandoning bad host %s: %s" % (hostname, msg))

        self.set_info("Leaving {}: {}, TIME_WAIT for {} seconds".format(self.client_info['fqdn'], msg, time_to_wait))
        self.wait_time(time_to_wait)

    def get_hostname(self):
        fqdn = self.client_info['fqdn']
        hostname, _, _ = fqdn.partition('.')
        return hostname

    def use_packed_bases(self):
        """Override this to use packed bases"""
        return False

    def create_yabs_server_factory(self, use_tmpfs=True):
        server_resource_path = self.sync_resource(self.server_res_id)
        server_path = os.path.abspath("yabs_server_bundle")
        logging.debug("Extract yabs-server bundle from %s to %s", server_resource_path, server_path)
        with tarfile.open(server_resource_path) as archive:
            archive.extractall(path=server_path)

        return YabsServerFactoryStandalone(
            server_path=server_path,
            surf_data_path=self.sync_resource(self.ctx[SurfDataResource.name]),
            base_dir=self.ramdrive.path if use_tmpfs else None,
        )

    def provide_bases(self, *servers):
        try:
            self._bin_bases_provider
        except AttributeError:
            self._bin_bases_provider = BinBasesProvider(self.agentr, self.use_packed_bases())
        db_resource_list = self.ctx.get(DBResourceList.name, [])
        db_resource_task_id = self.ctx.get(DBResourceTask.name)
        db_list = build_bin_base_list(db_resource_list, [db_resource_task_id] if db_resource_task_id else [])

        self._bin_bases_provider.provide(db_list, *servers)

    def create_meta(self, *args, **kwargs):
        """Deprecated, do not use!"""
        return self._get_yabs_server_factory().create_meta(*args, **kwargs)

    def create_stat(self, *args, **kwargs):
        """Deprecated, do not use!"""
        return self._get_yabs_server_factory().create_stat(*args, **kwargs)

    def create_turl(self, *args, **kwargs):
        """Deprecated, do not use!"""
        return self._get_yabs_server_factory().create_turl(*args, **kwargs)

    def create_null_stub(self, *args, **kwargs):
        """Deprecated, do not use!"""
        return self._get_yabs_server_factory().create_null_stub(*args, **kwargs)

    @property
    def server_base_revision(self):
        """Deprecated, do not use!"""
        return self._get_yabs_server_factory().server_base_revision

    def _get_yabs_server_factory(self):
        try:
            return self._factory
        except AttributeError:
            self._factory = self.create_yabs_server_factory(use_tmpfs=self.ctx.get(TmpfsSize.name, 0))
            return self._factory

    def __getstate__(self):
        # FIXME dirty hack
        try:
            del self._factory
        except AttributeError:
            pass

        try:
            del self._bin_bases_provider
        except AttributeError:
            pass

        try:
            return ServerMkdbInstallTask.__getstate__(self)
        except AttributeError:
            return self.__dict__


class CacheDaemonDump(ResourceSelector):
    name = 'cachedaemon_dump_resource'
    description = 'Cachedaemon dump (in local Sandbox, overrides ID from dolbilka plan attributes)'
    group = GROUP_TEST_DATA
    required = False
    multiple = False
    resource_type = YABS_SERVER_CACHE_DAEMON_STUB_DATA


class CacheDaemonBinary(ResourceSelector):
    name = 'cachedaemon_binary_resource'
    description = 'Cachedaemon binary (in local Sandbox, overrides ID from cachedaemon binary attributes)'
    group = GROUP_TEST_DATA
    required = False
    multiple = False
    resource_type = CACHE_DAEMON


class MetaMode(SandboxSelectParameter):
    name = 'meta_mode'
    default_value = 'bs'
    choices = [(m, m) for m in META_MODES]
    description = "Meta mode: {}".format('{} for {}'.format(m, META_MODES[m]) for m in sorted(META_MODES))
    required = True


class YabsMetaStatShootTask(ServerTask):

    input_parameters = ServerTask.input_parameters + (
        MetaMode,
        CacheDaemonDump,
        CacheDaemonBinary
    )

    def get_cachedaemon_dump_parent_res_id(self):
        """
        Return ID of a resource to which the cachedaemon dump
        is bound via the cachedaemon_dump_res_id attribute
        """
        raise AttributeError("Class is abstract")

    @property
    def spec(self):
        try:
            return self._spec
        except AttributeError:
            self._spec = super(YabsMetaStatShootTask, self).spec
            try:
                dump_res_id, binary_res_id = self._get_cachedaemon_resources()
                self._spec['resource_data'].update({
                    'CacheDaemonDump': self._single_resource_spec(dump_res_id),
                    'CacheDaemonBinary': self._single_resource_spec(binary_res_id)
                })
            except Exception:
                pass
            self._spec.update({
                'meta_mode': self.meta_mode
            })
            return self._spec

    @property
    def meta_mode(self):
        try:
            # FIXME self._meta_mode can also be filled by horrible code in FUNC_SHOOT
            return self._meta_mode
        except AttributeError:
            self._meta_mode = self.get_meta_mode(self.ctx)
            return self._meta_mode

    @classmethod
    def get_meta_mode(cls, task_context):
        return task_context[MetaMode.name]

    def create_cachedaemon_stub(self, services=('bigb', 'turl', 'yacofast'), dplan_res_id=None):
        dump_res_id, binary_res_id = self._get_cachedaemon_resources(dplan_res_id)
        return cachedaemon.CacheDaemonStub(
            self,
            services,
            dump_res_id,
            binary_res_id,
            data_dir=os.path.join(
                '.' if self.ramdrive is None else self.ramdrive.path,
                'cache_daemon_data',
            ),
        )

    def _get_cachedaemon_resources(self, parent_res_id=None):
        parent_res_id = parent_res_id or self.get_cachedaemon_dump_parent_res_id()
        dump_res_id = _obtain_related_res_id(self.ctx, CacheDaemonDump, parent_res_id, 'cachedaemon_dump_res_id')
        binary_res_id = _obtain_related_res_id(
            self.ctx,
            CacheDaemonBinary,
            dump_res_id,
            cachedaemon.CacheDaemonStub.cache_daemon_res_id_key
        )
        return dump_res_id, binary_res_id


def _obtain_related_res_id(ctx, res_param, parent_res_id, related_res_attr):

    if common.config.Registry().client.sandbox_user is None:
        # Local Sandbox
        return ctx[res_param.name]

    parent_res = common.rest.Client().resource[parent_res_id].read()

    res_id = ctx.get(res_param.name) or parent_res.get('attributes', {}).get(related_res_attr)
    if res_id:
        ctx[res_param.name] = res_id
        return res_id

    raise SandboxTaskUnknownError(
        "No {} in task context and no {} in attributes of resource {}".format(
            res_param.name, related_res_attr, parent_res_id
        )
    )
