import logging
import hashlib

from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolGroupParameter, ListRepeater
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.utils import get_or_default
from sandbox.projects.yabs.qa.utils.base import truncate_base_prefix

from sandbox.projects.common.yabs.server.db.task.cs import YabsCSTask, MySQLArchiveContents, OneshotConfig
from sandbox.projects.common.yabs.server.db.utils import calc_combined_settings_md5
from sandbox.projects.common.yabs.server.db.yt_bases import IMPORTER_SETTINGS_VERSION_ATTR, IMPORTER_CODE_VERSION_ATTR, IMPORTER_MKDB_INFO_VERSION_ATTR
from sandbox.projects.yabs.qa.bases.sample_tables.parameters import (
    SamplingStrategy,
    SamplingStrategyParameter,
)

logger = logging.getLogger(__name__)


class Options(SandboxBoolGroupParameter):
    name = 'options_list'
    description = 'extra options'

    RUN_SWITCHER = 'run_switcher'
    RUN_SWITCHER_NETWORK = 'run_switcher_network'
    NEED_SWITCHER_BASES_RESTORE = 'need_switcher_bases_restore'
    choices = [
        ('Run experiment-switcher --new-ctr-prediction all-groups', RUN_SWITCHER),
        ('Run experiment-switcher --network', RUN_SWITCHER_NETWORK),
    ]
    default_value = ' '


class BinDbList(SandboxStringParameter):
    name = 'bin_db_list'
    default_value = 'dbc constant st149'
    description = 'Binary base tags'


class Oneshot(SandboxStringParameter):
    name = 'oneshot'
    description = 'Oneshot'


class OneshotID(SandboxStringParameter):
    name = 'oneshot_id'
    description = 'Oneshot ID (autogenerated if not specified)'


class SwitcherRevision(SandboxStringParameter):
    name = 'switcher_revision'
    description = 'Revision of experiment-switcher dir (trunk if None)'
    default_value = ''


class SwitcherArguments(SandboxStringParameter):
    name = 'switcher_argruments'
    description = 'experiment-switcher running argruments'
    default_value = ''


class ImportPrefix(SandboxStringParameter):
    name = 'import_prefix'
    description = '--import-prefix for cs_cycle (will run cs_import if empty)'
    default_value = ''


class ImportDigest(SandboxStringParameter):
    name = 'import_digest'
    description = 'Hash of CS import result'


class CSImportVer(SandboxStringParameter):
    name = 'cs_import_ver'
    description = 'CS import version'


GROUP_ONESHOTS = 'New oneshots'


class OneShotPath(SandboxStringParameter):
    name = 'oneshot_path'
    description = 'oneshot_path (arcadia path to oneshot sql)'
    group = GROUP_ONESHOTS


class OneShotArgs(SandboxStringParameter):
    name = 'oneshot_args'
    description = 'oneshot_args (arguments for oneshot)'
    group = GROUP_ONESHOTS


class OneShotBases(SandboxStringParameter):
    name = 'oneshot_bases'
    description = 'oneshot_bases (binary bases affected by oneshot, comma-separated)'
    group = GROUP_ONESHOTS


class OneShotTables(SandboxStringParameter):
    name = 'oneshot_tables'
    description = 'oneshot_tables (tables found in oneshot SQL query, comma-separated)'
    group = GROUP_ONESHOTS


class OneShotHosts(SandboxStringParameter):
    name = 'oneshot_hosts'
    description = 'oneshot_hosts (comma-separated, as in bsint GUI)'
    group = GROUP_ONESHOTS


class MkdbInfoMd5(SandboxStringParameter):
    name = 'mkdb_info_md5'
    description = 'MD5 of mkdb_info output'
    default_value = ''


class BaseTagImporterSettingsVersion(SandboxStringParameter):
    name = 'base_tag_importer_settings_version'
    description = 'Version of settings for importers involved in generating base'
    required = False
    default_value = '{}'


class BaseTagImporterCodeVersion(SandboxStringParameter):
    name = 'base_tag_importer_code_version'
    description = 'Version of code for importers involved in generating base'
    required = False
    default_value = '{}'


# Contains md5 hash of dbtool mkdb_info json subtree for every base
class BaseTagMkdbInfoVersion(SandboxStringParameter):
    name = 'base_tag_mkdb_info_version'
    description = 'Version of dbtool mkdb_info for base'
    required = False
    default_value = '{}'


class CommonOneshotsMd5(SandboxStringParameter):
    name = 'common_oneshots_md5'
    description = 'Common oneshots md5'
    default_value = ''


class CommonOneshotsBases(ListRepeater, SandboxStringParameter):
    name = 'common_oneshots_bases'
    description = 'Binary bases, which are touched by common oneshots'
    required = False
    default_value = []


class BasenoList(ListRepeater, SandboxStringParameter):
    name = 'baseno_list'
    description = 'Baseno list is being used by sandbox tests'
    required = False
    default_value = []


BIN_BASE_RES_IDS_KEY = 'bin_base_res_ids'
GENERATION_ID_KEY = 'generation_id'
FETCH_ID_KEY = 'fetch_id'
CS_IMPORT_ID_KEY = "cs_import_id"
FETCH_OUTPUTS_KEY = 'fetch_outputs'
FETCH_PREFIX_KEY = 'fetch_prefix'


class BinBasesTask(YabsCSTask):

    # These parameters affect base contents.
    # Parameters that don't should be defined in child tasks.

    input_parameters = (
        Options, BinDbList, Oneshot, OneshotID, SwitcherRevision, SwitcherArguments, ImportPrefix, ImportDigest, CSImportVer,
        OneShotPath, OneShotArgs, OneShotBases, OneShotTables, OneShotHosts, MkdbInfoMd5,
        BaseTagImporterSettingsVersion, BaseTagImporterCodeVersion, BaseTagMkdbInfoVersion,
        CommonOneshotsMd5, CommonOneshotsBases, BasenoList
    ) + YabsCSTask.input_parameters

    @property
    def options(self):
        return frozenset((self.ctx.get(Options.name) or '').split())

    @property
    def switcher_options(self):
        return self.options & {Options.RUN_SWITCHER, Options.RUN_SWITCHER_NETWORK}

    @property
    def need_switcher_bases_restore(self):
        return self.options & {Options.NEED_SWITCHER_BASES_RESTORE}

    @property
    def switcher_args(self):
        return [arg for arg in self.ctx.get(SwitcherArguments.name, SwitcherArguments.default_value).split() if arg]

    @property
    def oneshot(self):
        return self.ctx.get(Oneshot.name)

    @property
    def oneshot_path(self):
        return self.ctx.get(OneShotPath.name)

    @property
    def oneshot_args(self):
        return self.ctx.get(OneShotArgs.name)

    @property
    def switcher_revision(self):
        return self.ctx.get(SwitcherRevision.name)

    @property
    def import_prefix(self):
        return get_or_default(self.ctx, ImportPrefix)

    @property
    def import_digest(self):
        return get_or_default(self.ctx, ImportDigest)

    @property
    def cs_import_ver(self):
        return get_or_default(self.ctx, CSImportVer)

    @staticmethod
    def get_packed_base_name(db_ver, db_name):
        return "{}.{}.yabs.tr".format(db_ver, db_name)

    @property
    def mkdb_info_md5(self):
        return get_or_default(self.ctx, MkdbInfoMd5)

    @property
    def common_oneshots_md5(self):
        return get_or_default(self.ctx, CommonOneshotsMd5)

    @property
    def common_oneshots_bases(self):
        common_oneshot_bases = get_or_default(self.ctx, CommonOneshotsBases)
        if not common_oneshot_bases:
            return []
        return common_oneshot_bases

    def get_db_list(self):
        return self.ctx.get(BinDbList.name, '').split()

    def iter_binary_bases_attrs(self, tags, db_ver, db_internal_vers=None, base_settings_version=None, base_code_version=None, base_mkdb_info_version=None):
        """Can be called from on_enqueue"""
        db_internal_vers = db_internal_vers or {}
        base_settings_version = base_settings_version or {}
        base_code_version = base_code_version or {}
        base_mkdb_info_version = base_mkdb_info_version or {}
        mysql_archive_contents_res_id = get_or_default(self.ctx, MySQLArchiveContents)
        generation_id = self.get_generation_id()
        for tag in tags:
            sampling_strategy = str(get_or_default(self.ctx, SamplingStrategyParameter))
            attrs = {
                'tag': str(tag),
                'db_ver': str(db_ver),
                'origin': 'sandbox' + self._get_db_origin(tag),
                'pack': 'tr',
                'mysql_archive_contents': str(mysql_archive_contents_res_id),
                'settings_spec_md5': str(calc_combined_settings_md5(self.cs_settings_archive_res_id, self.cs_settings_patch_res_id, self.settings_spec)),
                GENERATION_ID_KEY: generation_id,
                'sampling_strategy': sampling_strategy,
            }

            if base_settings_version.get(tag):
                attrs[IMPORTER_SETTINGS_VERSION_ATTR] = base_settings_version[tag]
            else:
                logger.warning("Not found base %s in settings versions %s", tag, str(base_settings_version))

            if base_code_version.get(tag):
                attrs[IMPORTER_CODE_VERSION_ATTR] = base_code_version[tag]
            else:
                logger.warning("Not found base %s in code versions %s", tag, str(base_code_version))

            tag_short = truncate_base_prefix(tag)
            if base_mkdb_info_version.get(tag_short):
                attrs[IMPORTER_MKDB_INFO_VERSION_ATTR] = base_mkdb_info_version[tag_short]
            else:
                logger.warning("Not found base %s (%s) in mkdb_info %s", tag, tag_short, str(base_mkdb_info_version))

            if db_internal_vers.get(tag):
                attrs['internal_ver'] = str(db_internal_vers[tag])

            if sampling_strategy == SamplingStrategy.sampled.value:
                attrs['sampling_parameters_hash'] = self.sampling_parameters_hash

            if self.input_spec_res_id and self.cs_import_ver:
                attrs['input_spec'] = str(self.input_spec_res_id)
                attrs['cs_import_ver'] = str(self.cs_import_ver)
                logger.info("Using import_ver: %s + input_spec: %s as a key for %s", self.cs_import_ver, self.input_spec_res_id, tag)
            else:
                attrs['import_prefix'] = self.import_prefix
                logger.info("Using import_prefix: %s as a key for %s", self.import_prefix, tag)

            if tag in self.common_oneshots_bases:
                attrs['common_oneshots_md5'] = self.common_oneshots_md5
            else:
                attrs['common_oneshots_md5'] = CommonOneshotsMd5.default_value

            yield attrs

    def get_generation_id(self):
        """
        Should be callable from on_enqueue.
        Return None if no generation_id is needed in attrubutes of our binary bases.
        """
        raise NotImplementedError("Class is abstract")

    def get_oneshot_config(self):
        oneshot_tables = self.ctx.get(OneShotTables.name, '')
        if not oneshot_tables:
            return OneshotConfig(self.get_yabscs(), base_tags=[], query='', tables=[], hosts=[])
        oneshot_tables = [table.strip() for table in oneshot_tables.split(',')]

        oneshot_hosts = [host.strip() for host in self.ctx.get(OneShotHosts.name, '').split(',')]
        base_tags = self.ctx.get(BinDbList.name, '').split()

        query = ''
        oneshot_path = self.ctx.get(OneShotPath.name, '')
        oneshot_path = oneshot_path.replace('//arcadia.yandex.ru/', '')
        if oneshot_path:
            Arcadia.export(
                'arcadia:/' + oneshot_path,
                'oneshot'
            )
            query = open('oneshot').read()

        return OneshotConfig(self.get_yabscs(), base_tags=base_tags, query=query, tables=oneshot_tables, hosts=oneshot_hosts)

    def on_execute(self):
        raise NotImplementedError("Class is abstarct")

    def _get_db_origin(self, db_tag=None):
        """Can be called from on_enqueue"""
        run_switcher = self.options & {Options.RUN_SWITCHER, Options.RUN_SWITCHER_NETWORK}

        labels = []

        if run_switcher:  # We do not really know what bases are modified!
            # TODO: always take from ctx and remove Arcadia.info
            # (wait till all the running tasks will have this param in ctx)
            switcher_revision = self.ctx.get(SwitcherRevision.name) or str(
                Arcadia.info('arcadia:/arc/trunk/arcadia/yabs/utils/experiment-switcher')['entry_revision']
            )
            labels.append('_ES_{}'.format(switcher_revision))
            arcadia_patch = self.ctx.get('arcadia_patch')
            if arcadia_patch:  # and db_tag == 'dbe':
                m = hashlib.new('md5')
                m.update(str(arcadia_patch))
                labels.append(m.hexdigest())

        oneshot = self.ctx.get(Oneshot.name)
        if oneshot:
            oneshot_id = self.ctx.get(OneshotID.name) or '_{:X}'.format(hash(oneshot)).replace('-', 'M')
            labels.append(oneshot_id)

        return '_'.join(labels)
