import os
import re
import logging
import time

from concurrent import futures

import sandbox.sandboxsdk.paths as sdk_paths

from sandbox.projects.common.yabs import dpkg
from sandbox.projects.common.yabs.executers import run_sh
from sandbox.projects.common.yabs.fs import move_to_task_dir

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls
from sandbox.projects.yabs.sandbox_task_tracing.wrappers import subprocess
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sandboxsdk.process import run_process


DEFAULT_MYSQL_YABS_SOCKET = '/var/run/mysqld.yabs/mysqld.sock'
DEFAULT_MYSQL_YABS_DB = 'yabsdb'
DEFAULT_MYSQL_YABS_USER = 'root'
GET_PORT_QUERY = 'SHOW GLOBAL VARIABLES LIKE \'PORT\';'


# TODO: use execute_mysql_query in all shm db queries
@trace_calls(save_arguments=(0, 'query'))
def execute_mysql_query(query, socket=DEFAULT_MYSQL_YABS_SOCKET, db=DEFAULT_MYSQL_YABS_DB, user=DEFAULT_MYSQL_YABS_USER):
    import MySQLdb

    connection = MySQLdb.connect(unix_socket=socket, user=user, db=db)
    cursor = connection.cursor()
    cursor.execute(query)
    rows = cursor.fetchall()
    cursor.close()
    connection.close()
    return list(rows)


def get_mysql_port():
    return int(execute_mysql_query(GET_PORT_QUERY)[0][1])


@trace_calls(save_arguments='all')
def install_yabs_mysql(dbrestore=False, switcher=False, mysql_instances=None, testing_host=None):
    # Always disable yandex-gosky updates from inside of the container
    if not os.path.exists('/var/run/gosky.nocron'):
        run_sh(['>', '/var/run/gosky.nocron'])

    # Set up /etc/testing
    with open('/etc/testing', 'w') as f:
        f.write('''
host bsgm-sandbox-singlestat
yabs-mysql-grants-testing-host 1
dbrestore-bsbackup-conductor-group yabs_bsbackup@sas
@base-map bsdb:yabsdb local:yabsdb
@base-map bsdb:yabsinfo local:yabsinfo
@base-map bsdb.yandex.ru:yabsdb local:yabsdb
@base-map bsdb.yandex.ru:yabsinfo local:yabsinfo
''')

    # Cleanup (for local sandbox only)
    run_sh(
        'for d in /var/lib/mysql.yabs* /mnt/raid /tmp/dbrestore; do\
            if [[ -L $d && ! -e $d ]]; then\
                rm $d;\
            fi;\
        done'
    )

    # Install packages
    repos_list = dpkg.DIST_YANDEX_STABLE + dpkg.DIST_SYSTEM_CONFIGS + dpkg.BSDIST_PRECISE_STABLE
    pkgs_list = ['yabs-mysql-conf']
    if dbrestore:
        pkgs_list.append('yabs-dbrestore')
    if switcher:
        pkgs_list.append('experiment-switcher')

    run_sh('telinit 2')  # Avoid running yabs statistics services
    logging.info("Installing packages: %s", ' '.join(pkgs_list))
    dpkg.install_pkg(pkgs_list, repos_list, clean_repos=False)

    run_sh('/etc/init.d/mysql.yabs stop')

    # Do not ever try to do this while starting mysql!!
    tmp_dir = sdk_paths.make_folder('tmp', delete_content=True)
    move_to_task_dir(['/mnt/raid', '/tmp/dbrestore'], tmp_dir)
    # fix Permission denied problem
    run_sh(['chmod -Rf a+rw /var/lib/mysql.*/ /var/log/yabs*'], check_exit_code=False)

    try:
        run_process(['/etc/init.d/mysql.yabs', 'yabs', 'start'], log_prefix='mysql.yabs_start')
    except Exception as err:
        raise RuntimeError(err)

    return {'yabs'}


def prepare_perl_oneshot_environment(instance='yabs'):
    # https://a.yandex-team.ru/arc/trunk/arcadia/yabs/qcc/config/bscork/smoke_functions.sh
    bsdbport = get_mysql_port()
    logging.info('prepare_perl_oneshot_environment with instance=%s and port=%s', instance, bsdbport)
    with open('/etc/testing', 'a') as f:
        f.write('''
base-local 0
force-mysql-tcpip-connect 1
@base-map bsdb local::{port}
@base-map local local::{port}
'''.format(port=bsdbport))

        all_dc_list = ['e', 'f', 'h', 'i']

        if instance.startswith('yabsistat'):
            for dc in all_dc_list:
                # in our basegen tasks, bsistat01|02|03 with all-different tables
                # are mapped to the only local mysql-instance
                for shard in ['1', '2', '3']:
                    f.write('@base-map bsistat0{}{} local::{}\n'.format(shard, dc, bsdbport))

        if instance.startswith('yabsst'):
            nshard = int(instance[6:])
            f.write('baseno {}\n'.format(nshard))
            for dc in all_dc_list:
                f.write('@base-map bstat{:02d}{} local::{}\n'.format(nshard, dc, bsdbport))

    with open('/etc/testing', 'r') as f:
        logging.info('Contents of /etc/testing:\n%s', f.read())


def synpack_tables(table_providers):
    logging.info("Syncing and unpacking %s tables...", len(table_providers))

    fetch_start = time.time()
    compressed_size = 0
    table_count = len(table_providers)

    with futures.ThreadPoolExecutor(max_workers=18) as sync_pool, futures.ThreadPoolExecutor(max_workers=29) as unpack_pool:
        syncs = [sync_pool.submit(tp.sync) for tp in table_providers]
        unpacks = []
        for n, ft in enumerate(futures.as_completed(syncs)):
            tp = ft.result()
            compressed_size += tp.compressed_size
            logging.info("Table %s synced (%s of %s)", tp, n, table_count)
            unpacks.append(unpack_pool.submit(tp.unpack))

        sync_time = time.time() - fetch_start
        compressed_MiB = 1.0 * compressed_size / 2**20
        logging.info(
            "Synced %.1f MiB of compressed tables in %.1f seconds, sync speed %.1f MiB/s",
            compressed_MiB, sync_time, compressed_MiB / sync_time
        )
        for n, ft in enumerate(futures.as_completed(unpacks)):
            tp = ft.result()
            logging.info("Table %s unpacked (%s of %s)", tp, n, table_count)

    logging.info("Synced and unpacked %s tables in %s seconds.", table_count, time.time() - fetch_start)


TABLES_ROOT = "/var/lib/mysql.yabs/"


class TableProvider(object):
    """
    Class that syncs and unpacks table.
    """

    def __init__(self, task, inst, db, table, res_id):
        self._task = task
        self.inst = inst
        self.db = db
        self.table = table
        self.res_id = res_id
        self.tar_path = None
        self.compressed_size = None

    def sync(self):
        logging.debug("Syncing %s (resource %s)", self, self.res_id)
        self.tar_path = self._task.agentr.resource_sync(self.res_id)
        stat = os.stat(self.tar_path)
        self.compressed_size = stat.st_size
        logging.debug("Synced %s (resource %s), size %s bytes to %s", self, self.res_id, self.compressed_size, self.tar_path)
        return self

    def unpack(self):
        try:
            logging.debug("Unpacking %s -> %s", self.tar_path, TABLES_ROOT)
            subprocess.check_output(['tar', '--keep-old-files', '-xzf', self.tar_path, '-C', TABLES_ROOT], stderr=subprocess.STDOUT)
        except subprocess.CalledProcessError as err:
            logging.exception("%s: failed to unpack %s. Output:\n%s", self, self.tar_path, err.output)
            raise
        except Exception:
            logging.exception("%s: failed to unpack %s", self, self.tar_path)
            raise
        logging.debug("Unpacked %s", self)
        return self

    def __str__(self):
        return "{}:{}.{}".format(self.inst, self.db, self.table)

    def __hash__(self):
        return hash(self.res_id)

    def __eq__(self, other):
        return self.res_id == other.res_id


def use_myisampack(table_name):
    return _MYISAMPACK_TABLES_RE.match(table_name) is not None


def un_myisampack_if_needed(tables, db='yabsdb'):
    for table_name in tables:
        if not use_myisampack(table_name):
            continue
        path = os.path.join(TABLES_ROOT, db, '{}.MYI'.format(table_name))
        cmdline = ['myisamchk', '--unpack', '--recover', path]
        run_process(cmdline, log_prefix='myisamchk_{}'.format(table_name))


_MYISAMPACK_TABLES_RE = re.compile(r'Banner\d+')
