import json
import logging
import requests
import six
import socket
import sys
import time


from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from sandbox.sandboxsdk import svn
from sandbox.common.utils import get_task_link


SOLOMON_TIMEOUT_RESPONSE_CODE = 504
logger = logging.getLogger(__name__)


def get_session_with_retries(retries=5, backoff_factor=0.3, status_forcelist=(500, 502, 504)):
    """Get request's Session with built-in retries

    :param retries: The maximum number of retries each connection should attempt
    :param backoff_factor: A backoff factor to apply between attempts after the second try
    :param status_forcelist: A set of integer HTTP status codes that we should force a retry on
    :return: requests.Session object

    For more information about parameters visit https://urllib3.readthedocs.io/en/latest/reference/urllib3.util.html?highlight=Retry#urllib3.util.retry.Retry
    """
    retry = Retry(total=retries,
                  read=retries,
                  connect=retries,
                  backoff_factor=backoff_factor,
                  status_forcelist=status_forcelist)
    adapter = HTTPAdapter(max_retries=retry)

    session = requests.Session()
    session.mount('http://', adapter)
    session.mount('https://', adapter)

    return session


def fetch_http_data(url, retries=5, raw=False):
    s = get_session_with_retries(retries)
    response = s.get(url)
    if raw:
        return response.content
    else:
        return response.text


def chunk_split(iterable, chunk_size):
    for index in six.moves.xrange(0, len(iterable), chunk_size):
        yield iterable[index:index + chunk_size]


def push_to_solomon(
    common_labels, sensors_list,
    max_sensors_per_push=4000, timeout_retries=0, timeout_sleep_interval=5, spack=True,
    api_v2_token=None,
):
    if spack:
        try:
            sys.path.append(svn.Arcadia.get_arcadia_src_dir(
                'arcadia:/arc/trunk/arcadia/yabs/server/libs/py_json2spack/'
            ))
        except:
            pass
        from yabs.server.libs.py_json2spack import encode_spackv1

    project = common_labels.pop('project')
    service = common_labels.pop('service')
    cluster = common_labels.pop('cluster')
    ack = 'at_least_one'
    solomon_host = 'solomon.yandex.net'
    params = dict(project=project, service=service, cluster=cluster)
    if api_v2_token:
        scheme, api_url = 'https', 'api/v2/push'
    else:
        scheme, api_url = 'http', 'push'
        params['ack'] = ack

    solomon_push_api_url = '{scheme}://{host}/{path}?{params}'.format(
        scheme=scheme, host=solomon_host, path=api_url,
        params=six.moves.urllib.parse.urlencode(params),
    )

    for index, sensors_chunk in enumerate(chunk_split(sensors_list, max_sensors_per_push)):
        logger.info('Chunk #{}'.format(index))
        data_dict = {
            'commonLabels': common_labels,
            'sensors': sensors_chunk,
        }
        logger.debug('Sending data to {}: {}'.format(solomon_push_api_url, data_dict))
        for retry_attempt in six.moves.xrange(timeout_retries + 1):
            try:
                if spack:
                    data = encode_spackv1(data_dict)
                    solomon_push_api_headers = {'Content-Type': 'application/x-solomon-spack'}
                else:
                    data = json.dumps(data_dict)
                    solomon_push_api_headers = {'Content-Type': 'application/json'}
                if api_v2_token:
                    solomon_push_api_headers['Authorization'] = 'OAuth {}'.format(api_v2_token)
                http_request = six.moves.urllib.request.Request(solomon_push_api_url, headers=solomon_push_api_headers)
                http_response = six.moves.urllib.request.urlopen(http_request, data, timeout=31)
                code = http_response.getcode()
            except socket.timeout:
                code = SOLOMON_TIMEOUT_RESPONSE_CODE
            except six.moves.urllib.error.HTTPError as e:
                code = e.code
                logger.debug("%s: %s", e.reason, e.read())
            if code in (requests.codes.ok, requests.codes.accepted):
                logger.info('Push request sent successfully: %s', http_response.read())
                break
            elif code == SOLOMON_TIMEOUT_RESPONSE_CODE and timeout_retries - retry_attempt > 0:
                logger.info('Timeout on push, sleep before trying again')
                time.sleep(timeout_sleep_interval)
                continue
            else:
                raise Exception('Error {} on push to Solomon'.format(code))


def send_startrek_report(token, ticket, st_report, sender_task_id=None):
    logger.info('Try to add Startrek comment')
    from startrek_client import Startrek
    st_client = Startrek(useragent='python', token=token)
    text = st_report
    if sender_task_id:
        text += '\n\nSent by (({task_link} {task_id}))'.format(
            task_link=get_task_link(sender_task_id),
            task_id=sender_task_id
        )
    st_client.issues[ticket].comments.create(text=text)


def convert_dict_to_sandbox_api_format(dict_):
    return [{"name": key, "value": value} for key, value in dict_.items()]


def check_input_parameter(parameters_class, key):
    parameter = getattr(parameters_class, key, None)
    return parameter is not None and not parameter.output


def truncate_output_parameters(parameters_dict, parameters_class):
    return {key: value for key, value in parameters_dict.items() if check_input_parameter(parameters_class, key)}
