import six

from collections import OrderedDict, namedtuple


def split_n(string, delimiter, splits):
    """Splits a string into precisely <splits> parts with respect to <delimiter>, filling with emptystrings if needed
    >>> split_n('', '/', 1)
    ['', '']
    >>> split_n('a', '/', 1)
    ['a', '']
    >>> split_n('a/b', '/', 1)
    ['a', 'b']
    >>> split_n('a/b/c', '/', 1)
    ['a', 'b/c']
    >>> split_n('/a/b/c', '/', 1)
    ['', 'a/b/c']
    """
    split_list = string.split(delimiter, splits)
    list_len = len(split_list)
    if list_len < splits + 1:
        for _ in six.moves.xrange(list_len, splits + 1):
            split_list.append('')
    return split_list


def safe_del(target_dict, key):
    if key in target_dict:
        del target_dict[key]


def rreplace(string, replace_from, replace_to, maxreplace):
    for _ in six.moves.xrange(maxreplace):
        head, sep, tail = string.rpartition(replace_from)
        if not sep:
            break
        else:
            string = head + replace_to + tail
    return string


def strip_http_request(string):
    return rreplace(split_n(string, '\n', 1)[1], '\r\n\r\n\n', '\r\n\r\n', 1)


def _parse_queryargs(queryargs_string):
    """Parses query args string into ordered dict, duplicated values will NOT be overriden
    >>> _parse_queryargs('q=a')
    OrderedDict([('q', ['a'])])
    >>> _parse_queryargs('q=a&q=b')  # This test case now works correctly, 'a' value is not overriden
    OrderedDict([('q', ['a', 'b'])])
    >>> _parse_queryargs('')
    OrderedDict()
    >>> _parse_queryargs('a=1&b=2')
    OrderedDict([('a', ['1']), ('b', ['2'])])
    >>> _parse_queryargs('b=1&a=2')
    OrderedDict([('b', ['1']), ('a', ['2'])])
    """
    queryargs = OrderedDict()
    if queryargs_string:
        for queryarg_atom in queryargs_string.split('&'):
            if queryarg_atom:
                key, value = split_n(queryarg_atom, '=', 1)
                if key not in queryargs:
                    queryargs[key] = []
                queryargs[key].append(value)
    return queryargs


ParsedRequestLine = namedtuple('ParsedRequestLine', ('handler', 'handler_data', 'queryargs'))


def _parse_request_line(path):
    """
    >>> _parse_request_line('/')
    ParsedRequestLine(handler='', handler_data='', queryargs=OrderedDict())
    >>> _parse_request_line('/count/foo?q=1')
    ParsedRequestLine(handler='count', handler_data='foo', queryargs=OrderedDict([('q', ['1'])]))
    >>> _parse_request_line('/count?q=1')
    ParsedRequestLine(handler='count', handler_data='', queryargs=OrderedDict([('q', ['1'])]))
    >>> _parse_request_line('/count/foo')
    ParsedRequestLine(handler='count', handler_data='foo', queryargs=OrderedDict())
    >>> _parse_request_line('/an/count/foo')
    ParsedRequestLine(handler='an', handler_data='count/foo', queryargs=OrderedDict())
    >>> _parse_request_line('/count/foo/')
    ParsedRequestLine(handler='count', handler_data='foo/', queryargs=OrderedDict())
    >>> _parse_request_line('///count/foo')
    ParsedRequestLine(handler='count', handler_data='foo', queryargs=OrderedDict())
    >>> _parse_request_line('//page/670942')
    ParsedRequestLine(handler='page', handler_data='670942', queryargs=OrderedDict())
    >>> _parse_request_line('https://yabs.yandex.ru:80/count/foo?q=1')
    ParsedRequestLine(handler='count', handler_data='foo', queryargs=OrderedDict([('q', ['1'])]))
    >>> _parse_request_line('https://yabs.yandex.ru:80///count/foo?q=1&q=2')
    ParsedRequestLine(handler='count', handler_data='foo', queryargs=OrderedDict([('q', ['1', '2'])]))
    >>> _parse_request_line('/mapuid/yandex/?sysconst-update=request-log-probability:1000000,match-log-mode:0&test-id=4')
    ParsedRequestLine(handler='mapuid', handler_data='yandex/', queryargs=OrderedDict([('sysconst-update', ['request-log-probability:1000000,match-log-mode:0']), ('test-id', ['4'])]))
    """
    parsed = six.moves.urllib.parse.urlparse(path.lstrip('/'))
    handler, handler_data = '', ''
    if parsed.path:
        handler, handler_data = split_n(parsed.path.lstrip('/'), '/', 1)
    queryargs = _parse_queryargs(parsed.query)
    return ParsedRequestLine(handler, handler_data, queryargs)


class HTTPRequest(six.moves.BaseHTTPServer.BaseHTTPRequestHandler, object):
    handler = ''
    handler_data = ''  # PageID or count data
    headers = OrderedDict()
    queryargs = OrderedDict()
    cookies = OrderedDict()

    http_template = (
        '{command} /{path}?{joined_queryargs} {request_version}\r\n'
        '{cookie_string}'  # either cookie: <cookies>\r\n or empty string
        '{joined_headers}'  # joined with \r\n
        '\r\n\r\n'
        '{content}'
    )

    wrap_template = '{http_len} {handler}\n{http_request}\n'

    def __init__(self, request_text=None):
        if request_text:
            self.from_string(request_text)

    def parse_request(self):
        super(HTTPRequest, self).parse_request()
        self.headers = OrderedDict(self.headers)
        try:
            assert self.error_code is None
        except AssertionError:
            raise AssertionError('Error code: {}\nError message: {}\n'.format(self.error_code, self.error_message))
        self.parse_request_line()
        self.parse_cookies()

    def from_string(self, request_text):
        self.headers = OrderedDict()
        self.queryargs = OrderedDict()
        self.cookies = OrderedDict()
        self.rfile = six.StringIO(request_text)
        self.raw_requestline = self.rfile.readline()
        self.error_code = self.error_message = None
        self.parse_request()
        return self

    def send_error(self, code, message):
        self.error_code = code
        self.error_message = message

    def parse_request_line(self):
        parsed = _parse_request_line(self.path)

        self.handler = parsed.handler
        self.handler_data = parsed.handler_data
        self.queryargs = parsed.queryargs

    def parse_cookies(self):
        self.cookies = OrderedDict()
        if 'cookie' in self.headers:
            cookie_string = self.headers['cookie']
            if cookie_string:
                for cookie_atom in cookie_string.split(';'):
                    key, value = split_n(cookie_atom, '=', 1)
                    key = key.strip()
                    if key or value:
                        self.cookies[key] = value
            del self.headers['cookie']

    @staticmethod
    def construct_atom(key, value, atom_delimiter):
        if value:
            return '{}{}{}'.format(key, atom_delimiter, value)
        else:
            return '{}{}'.format(key, atom_delimiter)

    def construct_joined_atom_string(
        self, to_join_dict, join_delimiter, atom_delimiter, key=None, allow_repeated_keys=False
    ):
        if not key:
            to_join_iterable = to_join_dict.items()
        else:
            to_join_iterable = sorted(to_join_dict.items(), key=key)

        to_join_iterable_extended = []
        for key, value in to_join_iterable:
            if isinstance(value, list):
                if allow_repeated_keys:
                    for v in value:
                        to_join_iterable_extended.append((key, v))
                else:
                    to_join_iterable_extended.append((key, value[-1]))
            else:
                to_join_iterable_extended.append((key, value))

        return join_delimiter.join(
            HTTPRequest.construct_atom(key, value, atom_delimiter) for key, value in to_join_iterable_extended
        )

    def update_raw_requestline(
        self,
        remove_connection_header=False,  # because apparently it's too hard to shoot requests as is
        allow_repeated_queryargs=False,
        omit_handler_data=False,
    ):
        headers_dict = {key.lower(): value for key, value in self.headers.items()}
        if remove_connection_header:
            safe_del(headers_dict, 'connection')
        if 'content-length' in headers_dict:
            content = self.rfile.read(int(headers_dict['content-length']))
        else:
            content = ''

        path = '{}/{}'.format(self.handler, self.handler_data) if not omit_handler_data else self.handler
        self.raw_requestline = self.http_template.format(
            command=self.command,
            path=path,
            joined_queryargs=self.construct_joined_atom_string(
                self.queryargs, '&', '=', allow_repeated_keys=allow_repeated_queryargs
            ),
            request_version=self.request_version,
            cookie_string=(
                'cookie: {}\r\n'.format(self.construct_joined_atom_string(self.cookies, '; ', '='))
                if self.cookies
                else ''
            ),
            joined_headers=self.construct_joined_atom_string(headers_dict, '\r\n', ': ', key=lambda x: x[0].lower()),
            content=content,
        )

    def to_string(self, wrapped=False, remove_connection_header=False, allow_repeated_queryargs=False, omit_handler_data=False):  # ergo
        self.update_raw_requestline(
            remove_connection_header,
            allow_repeated_queryargs=allow_repeated_queryargs,
            omit_handler_data=omit_handler_data
        )

        if wrapped:
            return self.wrap_template.format(
                http_len=len(self.raw_requestline), handler=self.handler, http_request=self.raw_requestline
            )
        else:
            return self.raw_requestline
