# -*- coding: utf-8 -*-

import time
import logging

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskUnknownError
from sandbox.sandboxsdk.channel import channel

from sandbox import common
import sandbox.common.types.task as ctt

from sandbox.common.types.client import Tag

from sandbox.projects.common.yabs.executers import normalize_wd
from sandbox.projects.common.yabs.parameters import AlsoInstall

from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_new_resources, trace_new_tasks


def _ctx_items_cmp_equal(_, val1, val2):
    """
    For use in find_task. Default key-dependent compare function for task.ctx items.
    """
    return val1 == val2


def find_task(ctx=None, ctx_items_cmp_equal=_ctx_items_cmp_equal, **args):
    """
    Finds task with all context keys specified in in ctx
    equal to values specified in ctx
    """
    args['limit'] = args.get('limit', 2000)
    args['status'] = args.get('status', 'FINISHED')  # TODO: or executing
    ctx = ctx or {}
    logging.debug("[find_task] search task with params %s, ctx=%s", args, ctx)
    for task in channel.sandbox.list_tasks(**args):
        missing_keys = set(ctx) - set(task.ctx)
        if missing_keys:
            logging.debug("[find_task] task %s: no keys %s", task.id, ' '.join(missing_keys))
            continue
        suitable = True
        for key, val in ctx.iteritems():
            if not ctx_items_cmp_equal(key, task.ctx[key], val):
                logging.debug(
                    "[find_task] task %s : value of %s is %s, does not match %s",
                    task.id, key, task.ctx[key], ctx[key]
                )
                suitable = False
        if suitable:
            logging.debug("[find_task] found suitable task %s", task.id)
            return task
    logging.debug("[find_task] no tasks with params %s found", args)
    return None


class TaskWrapper(SandboxTask):
    # XXX: Deprecated, do not use

    class DebugMe(parameters.SandboxBoolParameter):
        description = "Sleep 10 hours before terminate"
        group = "Debug"
        name = "debug_me"

    client_tags = Tag.LINUX_PRECISE & Tag.GENERIC
    input_parameters = [DebugMe]
    max_restarts = 2

    def get_resource(self, id=None, type='', attrs={}):
        res = None
        rest_client = common.rest.Client()
        if id:
            res = rest_client.resource[id].read()
        else:
            logging.debug("Looking for resource {} with {}".format(type, attrs))
            items = rest_client.resource.read(
                type=str(type),
                state='READY',
                attrs=attrs,
                limit=1
            )["items"]
            logging.debug("found: {}".format(items))
            if items:
                res = items[0]
        return res

    def get_resource_id(self, rtype, attrs={}):
        item = self.get_resource(type=rtype, attrs=attrs)
        return item['id'] if item else None

    @trace_calls
    def check_and_wait_tasks(self, new_task_ids=[]):
        """
        Get task ids from self.ctx and new_task_ids.
        Check that they have finished successfully.
        Fail if any of them have failed.
        If any of them are running, store their ids in context and start waiting them.
        """
        if not isinstance(new_task_ids, (list, tuple, dict)):
            new_task_ids = [new_task_ids]
        else:
            new_task_ids = list(new_task_ids)

        check_task_ctx_key = "__task_ids_to_check_{}".format(self.id)
        check_task_ids = list(self.ctx.get(check_task_ctx_key, []))
        check_task_ids += new_task_ids

        logging.info("[check_and_wait_tasks] check_task_ids: %s", check_task_ids)
        wait_task_ids = []
        failed_task_ids = []

        end_statuses = list(ctt.Status.Group.FINISH) + list(ctt.Status.Group.BREAK)
        rest_client = common.rest.Client()
        for task_id in check_task_ids:
            task = rest_client.task[task_id].read()
            status = task['status']
            if status not in end_statuses:
                wait_task_ids.append(task_id)
            elif status != ctt.Status.SUCCESS:
                failed_task_ids.append(task_id)
        if failed_task_ids:
            raise SandboxTaskUnknownError(
                "Something wrong with tasks: {}".format(failed_task_ids)
            )
        logging.info("[check_and_wait_tasks] wait_task_ids: %s", wait_task_ids)
        self.ctx[check_task_ctx_key] = wait_task_ids
        if wait_task_ids:
            self.wait_all_tasks_stop_executing(wait_task_ids)
        logging.info("All subtasks well done")

        return check_task_ids

    @trace_calls
    def _find_or_start_task(
        self,
        task_type,
        required_params,
        start_params={},
        ctx_items_cmp_equal=_ctx_items_cmp_equal,
    ):
        """
        Either find existing successful task wuith specified parameters
        or launch a new one.
        :param dict required_params: parameters used both for lookup and launching a new task
        :param dict start_params: additional parameters to use when task is started
        :return dict: task (found or started)
        """
        task = find_task(ctx=required_params, ctx_items_cmp_equal=ctx_items_cmp_equal, task_type=task_type)
        if task is None:
            in_params = required_params.copy()
            in_params.update(start_params)
            logging.debug("Creating subtask %s with params: %s", task_type, in_params)
            task = self.create_subtask(
                task_type=task_type,
                description="{} {} subtask".format(self.type, self.id),
                input_parameters=in_params,
            )
        else:
            logging.debug("Found finished %s task with params %s", task_type, required_params)
        return task

    def postprocess(self):
        if self.ctx.get(self.DebugMe.name):
            time.sleep(36000)
        SandboxTask.postprocess(self)

    def create_subtask(self, *args, **kwargs):
        with trace_new_tasks() as new_tasks:
            # cannot use `super` here since `SandboxTask` is an old-style class
            task = SandboxTask.create_subtask(self, *args, **kwargs)
            new_tasks.append(task)
            return task

    def create_resource(self, *args, **kwargs):
        with trace_new_resources() as new_resources:
            # cannot use `super` here since `SandboxTask` is an old-style class
            resource = SandboxTask.create_resource(self, *args, **kwargs)
            new_resources.append(resource)
            return resource

    @trace_calls(save_arguments=(1, 'resource_id'))
    def sync_resource(self, *args, **kwargs):
        # cannot use `super` here since `SandboxTask` is an old-style class
        return SandboxTask.sync_resource(self, *args, **kwargs)

    @trace_calls(save_arguments=(1, 'resource_id'))
    def mark_resource_ready(self, *args, **kwargs):
        # cannot use `super` here since `SandboxTask` is an old-style class
        return SandboxTask.mark_resource_ready(self, *args, **kwargs)


class SudoTaskWrapper(TaskWrapper):
    # XXX: Deprecated, do not use

    privileged = True

    client_tags = (Tag.YABS | Tag.GENERIC) & Tag.LINUX_PRECISE
#    client_tags = Tag.YABS & Tag.LINUX_PRECISE

    input_parameters = [AlsoInstall] + TaskWrapper.input_parameters

    @property
    def also_install(self):
        return self.ctx.get(AlsoInstall.name, '').split()

    def cleanup(self):
        """ Required for local installations only, drop the method after SANDBOX-2869 close. """
        if not self.privileged:
            return
        if not common.config.Registry().client.sandbox_user:
            normalize_wd()
        TaskWrapper.cleanup(self)
