# -*- encoding: utf-8 -*-
import collections
import sys
import os
import re
import logging
import calendar
import time
import tarfile
import cgi
from contextlib import contextmanager

from sandbox.common.rest import Client
from sandbox.common.types.task import ReleaseStatus, Status
from sandbox.sandboxsdk import channel

import sandbox.projects.resource_types as rt
import sandbox.projects.common.yabs.parameters


YT_LOCK_PATH = "//yabs/Locks/YabsYtStat"
YT_RELEASES_PATH = "//home/yabs/release"
COLLECTOR_ATTR_PREFIX = "collector_"


def get_yt_client(proxy=None, token=None, config=None):
    from yt.wrapper import YtClient
    return YtClient(proxy, token, config)


def get_version_from_yt(resource_type, ytc, release_status=None):
    release_status = release_status or ReleaseStatus.STABLE
    node_path = '/'.join((YT_RELEASES_PATH, release_status, str(resource_type)))
    return str(ytc.get(node_path))


def get_versions_from_yt(resource_type, ytc, release_status=None):
    """Return versions of subtypes of given resource_type."""
    release_status = release_status or ReleaseStatus.STABLE
    node_path = '/'.join((YT_RELEASES_PATH, release_status))
    table = ytc.get(node_path)
    versions = {}
    resource_type = str(resource_type)
    for resource_name, version in table.iteritems():
        if resource_name.startswith(resource_type):
            subtype = resource_name[len(resource_type) + len(':'):]
            versions[subtype] = version
    return versions


def get_autoreleased_resource(resource_type, ytc, released=True, **attrs):
    if isinstance(ytc, basestring):
        ytc = get_yt_client(ytc)

    if released is True:
        attrs['released'] = ReleaseStatus.STABLE
    elif released is not False:
        attrs['released'] = released

    attrs['version'] = get_version_from_yt(resource_type, ytc, attrs.get('released'))

    resources = channel.channel.sandbox.list_resources(resource_type, omit_failed=True, all_attrs=attrs, limit=1)
    if not resources:
        raise Exception('Failed to get autoreleased resource {} #{}'.format(resource_type, attrs['version']))
    return resources[0]


def get_and_sync_autoreleased_resource(resource_type, ytc, task, released=True, **attrs):
    resource = get_autoreleased_resource(resource_type, ytc, released, **attrs)
    task.sync_resource(resource.id)
    return resource


def get_autoreleased_resources(resource_type, ytc, released=True):
    """Return a list of all autoreleased resources.

    :param basestring|sandbox.projects.resource_types.AbstractResource resource_type: Type of resource to find
    :param basestring|yt.wrapper.client.Yt ytc: Proxy url or YT client
    :param basestring|bool released: If True will be 'stable', if False will be omitted, else given one will be used
    :rtype: list[sandboxsdk.sandboxapi.SandboxResource]
    """
    if isinstance(ytc, basestring):
        ytc = get_yt_client(ytc)

    attrs = {}

    if released is True:
        attrs['released'] = ReleaseStatus.STABLE
    elif released is not False:
        attrs['released'] = released

    resources = channel.channel.sandbox.list_resources(
        resource_type, omit_failed=True, all_attrs=attrs, limit=None
    ) or []
    resources_map = collections.defaultdict(dict)
    for resource in resources:
        subtype = resource.attributes.get('subtype')
        version = resource.attributes.get('version')
        if not subtype or not version:
            continue
        conflict_resource = resources_map[subtype].get(version)
        if conflict_resource:
            logging.warn('Resources with same subtype and version are found: %s, %s', resource, conflict_resource)
        else:
            resources_map[subtype][version] = resource

    versions_map = get_versions_from_yt(resource_type, ytc, attrs.get('released'))

    current_resources = []
    for subtype, version in versions_map.iteritems():
        current_resource = resources_map[subtype].get(version)
        if not current_resource:
            raise Exception('Failed to find autoreleased resource {}:{} #{}'.format(resource_type, subtype, version))
        current_resources.append(current_resource)

    return current_resources


def install_resource(task, resource_id=None):
    path = task.sync_resource(resource_id)
    if not path:
        raise BaseException("Resource id={} not found".format([resource_id]))

    with tarfile.open(path, 'r:gz') as tar:
        tar.extractall()


def install_supervisor(task, ytc, released=True):
    res = get_autoreleased_resource(rt.YABS_YTSTAT_SUPERVISOR, ytc, released)
    install_resource(task, res.id)
    sys.path.append(os.getcwd())


def _str2time(s):
    m = re.match(r'(\d+)-(\d+)-(\d+)T(\d+):(\d+):(\d+)', s)
    if m:
        items = [int(n) for n in m.groups()]
        items.extend((0, 0, 0))
        return calendar.timegm(time.struct_time(items))
    else:
        raise Exception("Cannot parse time {}".format(s))


def get_task_create_time(task_id):
    rest_client = Client()
    task = rest_client.task[task_id].read()
    tm = task["time"]["created"]
    return _str2time(tm)


def stderr_to_file(flog):
    sys.stderr.flush()
    old_stderr_no = os.dup(sys.stderr.fileno())
    f = open(flog, "a")
    os.dup2(f.fileno(), sys.stderr.fileno())
    return (old_stderr_no, f)


def revert_stderr(ctx):
    old_stderr_no, f = ctx
    os.dup2(old_stderr_no, sys.stderr.fileno())
    f.close()


@contextmanager
def stderr_redirect(flog):
    ctx = stderr_to_file(flog)
    yield
    revert_stderr(ctx)


def prepare_log_file(task, fname):
    with task.memoize_stage.prepare_log_file:
        open(fname, "a").close()  # touch
        task.create_resource(description="Custom log",
                             resource_path=fname,
                             resource_type=rt.TASK_CUSTOM_LOGS)


class SbApi(object):
    def __init__(self, task):
        self._task = task

    def get_collectors(self, ytc):
        collectors = {}
        released = not self._task.ctx["is_development"]
        resources = get_autoreleased_resources(rt.YABS_YTSTAT_COLLECTOR, ytc, released)
        for resource in resources:
            for attr, value in resource.attributes.iteritems():
                attr = attr.strip()
                if attr.startswith(COLLECTOR_ATTR_PREFIX):
                    col_name = attr[len(COLLECTOR_ATTR_PREFIX):]
                    logtype, binary = value.strip().split(":")
                    collectors.setdefault(logtype, []).append({
                        "name": col_name,
                        "binary": binary,
                        "resource_type": resource.type,
                        "resource_id": resource.id,
                        "logtype": logtype,
                    })
        return collectors

    def _group_info_from_resource(self, res):
        res_attrs = res["attributes"]
        collectors = []
        for attr, value in res_attrs.iteritems():
            if attr.startswith(COLLECTOR_ATTR_PREFIX):
                col_name = attr[len(COLLECTOR_ATTR_PREFIX):]
                log_type, binary = value.strip().split(":")
                collectors.append({
                    "name": col_name,
                    "log_type": log_type,
                    "binary": binary,
                })

        return {
            "group_name": res_attrs["subtype"],
            "version": res_attrs["version"],
            "resource_id": res["id"],
            "resource_md5": res["md5"],
            "dest_tables": res_attrs.get("dest_tables", "").split(","),  # TODO Replace 'get' by '[]' when all collectors have 'dest_tables' attribute
            "collectors": collectors,
            "issue": res_attrs.get("issue", ""),
        }

    def get_collector_info_by_resource_id(self, resource_id, collector_name):
        client = Client()
        res_info = client.resource[resource_id].read()
        group_info = self._group_info_from_resource(res_info)
        cols = [c for c in group_info["collectors"] if c["name"] == collector_name]
        if not cols:
            raise Exception("Collector {} not found in resource with id={}".format(collector_name, resource_id))
        col, = cols
        return {
            "group_name": group_info["group_name"],
            "resource_md5": group_info["resource_md5"],
            "version": group_info["version"],
            "binary": col["binary"],
            "issue": group_info["issue"],
        }

    def get_collector_group_by_name(self, yt_version, group_name, version):
        client = Client()
        found = client.resource.read(
            type=str(rt.YABS_YTSTAT_COLLECTOR),
            attrs={
                "subtype": group_name,
                "version": version,
            },
            limit=2,
        )
        items = found["items"]

        if not items:
            raise Exception("No resource found for: subtype={}, version={}".format(group_name, version))
        elif len(items) > 1:
            raise Exception("Found more than 1 resource with subtype={}, version={}. Ids: {}".format(
                group_name, version, ",".join([str(item["id"]) for item in items])))

        return self._group_info_from_resource(items[0])

    def start_collector(self, collector):
        ctx = self._task.ctx
        yt_token_secret_id = ctx["yt_token_secret_id"]
        yt_proxy = ctx["yt_proxy"]
        log_path = ctx["log_path"]

        collector_name = collector["name"]
        logging.info("Starting collector {}".format(collector_name))
        release_type = collector.get("release_type", "prod")  # missed in old supervisor
        resource_id = collector["resource_id"]

        import projects.YabsYTStatCollector

        desc_fmt = \
            '[<a href="https://{yt_proxy}/">{yt_proxy}</a>]' + \
            'resource_id=<a href="https://{sb_host}/resource/{resource_id}/view">{resource_id}</a>:' + \
            '({release_type})' + \
            '<b>{ename}</b> (<a href="https://{yt_proxy}/#page=navigation&path={log_path}/{log_type}">{log_type}</a>)'

        # TODO Replace by collector["log_type"] when new supervisor deployed
        log_type = collector.get("log_type", collector.get("logtype", "UNKNOWN"))

        collector_tag = collector_name.replace('<', '').replace('>', '')
        yt_proxy_tag = yt_proxy.split('.', 1)[0] if '.' in yt_proxy else yt_proxy

        logging.info("Starting task")
        params = sandbox.projects.common.yabs.parameters.CollectorParameters
        collector_kill_timeout = 90 * 60 # default 1h:30m
        if "ExperimentStat" in collector_name:
            collector_kill_timeout = 24 * 60 * 60 # 24h
        if "AdfoxEventCollector" in collector_name or "ChEventArchive" in collector_name or "ActionLog" in collector_name:
            collector_kill_timeout = 6 * 60 * 60 # 6h
        self._task.create_subtask(
            task_type=projects.YabsYTStatCollector.__Task__.type,
            description=desc_fmt.format(
                yt_proxy=yt_proxy,
                resource_id=resource_id,
                sb_host="sandbox.yandex-team.ru",
                release_type=release_type,
                ename=cgi.escape(collector_name),
                log_path=log_path,
                log_type=log_type,
            ),
            input_parameters={
                "yt_proxy": yt_proxy,
                "yt_token_secret_id": yt_token_secret_id,
                "collector_name": collector_name,
                "resource_id": resource_id,
                "log_path": log_path,
                "release_type": release_type,
                params.is_development.name: ctx[params.is_development.name],
                params.is_auditable.name: ctx[params.is_auditable.name],
                params.allow_man.name: ctx[params.allow_man.name],
                params.allow_vla.name: ctx[params.allow_vla.name],
                params.lock_wait_time.name: ctx[params.lock_wait_time.name],
                "kill_timeout": collector_kill_timeout
            },
            tags=[yt_proxy_tag, log_type, collector_tag],
            execution_space=24 * 1024,  # 24GB
            inherit_notifications=True,
        )
        logging.info("Task started")

    def get_running_collectors(self, yt_proxy):
        """ Return all executing in sandbox collectors"""
        import projects.YabsYTStatCollector

        rest_client = Client()
        run_statuses = Status.Group.expand(["QUEUE", "EXECUTE", "WAIT"])
        sbtasks = rest_client.task.read(
            type=projects.YabsYTStatCollector.__Task__.type,
            children=True,
            limit=2000,  # 'unlimited'
            status=",".join(run_statuses),
            fields=["id"]
        )
        cols = set()
        for task in sbtasks["items"]:
            ctx = rest_client.task[task["id"]].context.read()
            if ctx["collector_name"] and ctx["yt_proxy"] == yt_proxy:
                cols.add(ctx["collector_name"])
        return cols

    def get_running_collectors2(self, yt_proxy, return_counters=False):
        """ Return all executing in sandbox collectors"""
        import projects.YabsYTStatCollector

        LIMIT = 100
        RUN_STATUSES = Status.Group.expand(["QUEUE", "EXECUTE", "WAIT"])

        rest_client = Client()
        if return_counters:
            cols = collections.defaultdict(int)
        else:
            cols = set()
        offset = 0
        while True:
            sbtasks = rest_client.task.read(
                type=projects.YabsYTStatCollector.__Task__.type,
                children=True,
                offset=offset,
                limit=LIMIT,
                status=",".join(RUN_STATUSES),
                fields=["id", "context"],
            )

            for task in sbtasks["items"]:
                ctx = task["context"]
                if ctx["collector_name"] and ctx["yt_proxy"] == yt_proxy:
                    if return_counters:
                        cols[(ctx["collector_name"], ctx["release_type"])] += 1
                    else:
                        cols.add((ctx["collector_name"], ctx["release_type"]))

            offset += LIMIT

            if len(sbtasks["items"]) < LIMIT:
                break

        return cols

    def start_cloner(self, action, group_name, tables):
        ctx = self._task.ctx
        yt_token_vault_name = ctx["yt_token_vault_name"]
        yt_proxy = ctx["yt_proxy"]
        tables_str = ",".join(tables)

        import projects.YabsYTStatTableCloner

        self._task.create_subtask(
            task_type=projects.YabsYTStatTableCloner.__Task__.type,
            description="{}: {}".format(group_name, tables_str),
            input_parameters={
                "yt_proxy": yt_proxy,
                "yt_token_vault_name": yt_token_vault_name,
                "action": action,
                "group_name": group_name,
                "tables": tables_str,
            },
            inherit_notifications=True,
        )

    def get_running_cloners(self, yt_proxy):
        """ Return all executing in sandbox collectors"""
        import projects.YabsYTStatTableCloner

        LIMIT = 100
        RUN_STATUSES = Status.Group.expand(["QUEUE", "EXECUTE", "WAIT"])

        rest_client = Client()
        group_names = set()
        offset = 0
        while True:
            sbtasks = rest_client.task.read(
                type=projects.YabsYTStatTableCloner.__Task__.type,
                children=True,
                offset=offset,
                limit=LIMIT,
                status=",".join(RUN_STATUSES),
                fields=["id", "context"],
            )

            for task in sbtasks["items"]:
                ctx = task["context"]
                if ctx["group_name"] and ctx["yt_proxy"] == yt_proxy:
                    group_names.add(ctx["group_name"])

            offset += LIMIT

            if len(sbtasks["items"]) < LIMIT:
                break

        return group_names
