import itertools
import logging
import os
import re
import requests

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess

import sandbox.common.types.task as ctt

from sandbox.projects.common.yappy.github import BuildWithGithubTask


class BuildPatched(BuildWithGithubTask):

    class Requirements(BuildWithGithubTask.Requirements):
        disk_space = 10 * 1024

    class Context(BuildWithGithubTask.Context):
        ya = None
        svn_git_diff = None
        svn_dir = None
        arcadia_dir = None
        run_tests = True
        svn_revision = ''
        is_master = False
        paste_retries = 3

    class Parameters(BuildWithGithubTask.Parameters):
        try_sync = sdk2.parameters.Bool("Try to sync git master with svn")
        just_print_status = sdk2.parameters.Bool("Do not commit changes", default=True)
        ya_make_task_type = sdk2.parameters.String('Build task type', default='YA_MAKE')

    @property
    def github_context(self):
        raise NotImplementedError

    @property
    def bundle_resource_type(self):
        return None

    @property
    def bundle_description(self):
        return None

    @property
    def bundle_info(self):
        return {}

    @property
    def static_resources(self):
        # paths from root
        return {}

    @property
    def additional_binaries(self):
        return {}

    @property
    def arcadia_project_path(self):
        return self.Parameters.arcadia_path

    def install(self):
        raise NotImplementedError

    def on_execute(self):
        with self.memoize_stage.prepare_and_run_yamake(max_runs=1):
            self.github_statuses.report_self_status(description='Running')
            self.Context.sources_path = self.checkout()
            self.before_install()
            self.install()
            self.after_install()
            self.before_build()
            self.build()

        with self.memoize_stage.collect_binaries:
            self.after_build()

    def build(self):
        with self.info_section('<build> creating YA_MAKE task'):
            with sdk2.helpers.ProcessLog(self, 'build'):

                # convert bundle_info keys to format required by
                # argument "arts" of task

                def get_path_from_project_dir(path_from_src):
                    cut_binary_name = os.path.split(path_from_src)[0]
                    if cut_binary_name[-3:] == 'bin':
                        cut_binary_name = os.path.split(cut_binary_name)[0]
                    return os.path.join(self.arcadia_project_path, 'src', cut_binary_name)

                build_arts = ';'.join(
                    map(
                        get_path_from_project_dir,
                        list(self.bundle_info.keys()) + list(self.additional_binaries.keys())
                    )
                )

                task = sdk2.Task[self.Parameters.ya_make_task_type](
                    self,
                    description='Ya make task for yappy\n{}'.format(self.Parameters.description.encode('utf-8')),
                    owner=self.owner,
                    targets=self.arcadia_project_path,
                    build_type='release',
                    checkout_mode='auto',
                    arts=build_arts,
                    checkout_arcadia_from_url='arcadia:/arc/trunk/arcadia@{}'.format(self.Context.svn_revision),
                    arcadia_patch=self.Context.svn_git_diff,
                    test=self.Context.run_tests,
                    use_aapi_fuse=True
                    )
                task.enqueue()

                raise sdk2.WaitTask(task, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

    def process_branch(self):
            with self.info_section('<setup> svn diff'), sdk2.helpers.ProcessLog(self, 'svn_diff') as plog:
                proc_pipes = {'stdout': plog.stdout, 'stderr': plog.stderr}
                subprocess.check_call(
                    [self.Context.ya, 'tool', 'svn', 'add', '--force', '.'], cwd=self.Context.svn_dir, **proc_pipes)
                try:
                    self.Context.svn_git_diff = subprocess.check_output(
                        [self.Context.ya, 'svn', 'diff', '--notice-ancestry'], cwd=self.Context.svn_dir)
                except subprocess.CalledProcessError as error:
                    self.Context.svn_git_diff = error.output

                # patch should have paths from root of arcadia
                def add_prefix_to_diff_line(diff_line, prefix):
                    if diff_line[:4] in ('+++ ', '--- '):
                        return '{}{}/{}'.format(diff_line[:4], prefix, diff_line[4:])
                    elif diff_line[:7] == 'Index: ':
                        return '{}{}/{}'.format(diff_line[:7], prefix, diff_line[7:])
                    return diff_line

                def process_one_line(prefix):
                    def _wrapped(diff_line):
                        return add_prefix_to_diff_line(diff_line, prefix)
                    return _wrapped

                normalized_prefix = self.arcadia_project_path.strip('/')
                diff_text = '\n'.join(map(process_one_line(normalized_prefix), self.Context.svn_git_diff.split('\n')))
                if diff_text:
                    for _ in range(self.Context.paste_retries):
                        try:
                            diff_r = requests.post('https://paste.yandex-team.ru/', data={'syntax': 'diff', 'text': diff_text})
                            paste_id = re.search('a href="/([0-9]*)/html">HTML', diff_r.text).group(1)
                            self.Context.svn_git_diff = 'https://paste.yandex-team.ru/{}/text'.format(paste_id)
                            break
                        except Exception as e:
                            self.set_info('Diff paste raised an exception: "{}", retrying'.format(e))

    def process_master(self):
        with self.info_section('<setup> svn sync'), sdk2.helpers.ProcessLog(self, 'svn_sync') as plog:
            proc_pipes = {'stdout': plog.stdout, 'stderr': plog.stderr}
            svn_status = subprocess.check_output((self.Context.ya, 'svn', 'st'), cwd=self.Context.svn_dir)
            self.set_info('Svn status is:\n{}'.format(svn_status))
            svn_status = svn_status.split('\n')

            files_to_delete = []
            files_to_add = []

            if not self.Parameters.just_print_status:
                for status_line in svn_status:
                    if status_line.startswith('!'):
                        files_to_delete.append(status_line.split(None, 1)[1])
                    elif status_line.startswith('?'):
                        files_to_add.append(status_line.split(None, 1)[1])

                if files_to_add:
                    subprocess.check_call(
                        itertools.chain((self.Context.ya, 'svn', 'add'), files_to_add),
                        cwd=self.Context.svn_dir, **proc_pipes
                    )

                if files_to_delete:
                    subprocess.check_call(
                        itertools.chain((self.Context.ya, 'svn', 'rm'), files_to_delete),
                        cwd=self.Context.svn_dir, **proc_pipes
                    )

                subprocess.check_call(
                    (
                        self.Context.ya, 'svn', 'commit', '-m',
                        'SKIP_CHECK SKIP_REVIEW github sync({}@master)'.format(self.Parameters.project_github_commit),
                    ),
                    cwd=self.Context.svn_dir, **proc_pipes
                )

                subprocess.check_call(
                    (self.Context.ya, 'svn', 'up'),
                    cwd=self.Context.svn_dir, **proc_pipes
                )

        if self.Parameters.just_print_status:
            self.Context.is_master = False
            self.process_branch()

    def before_install(self):
        with sdk2.helpers.ProcessLog(self, 'before_install') as plog:
            proc_pipes = {'stdout': plog.stdout, 'stderr': plog.stderr}

            with self.info_section('<setup> publish static resources'):
                for path, resource_type in self.static_resources.items():
                    sdk2.ResourceData(
                        resource_type(self, description=path, path=os.path.join(self.Context.sources_path, path))
                        ).ready()

            with self.info_section('<setup> checkout arcadia'):
                self.Context.ya = sdk2.svn.Arcadia.export('arcadia:/arc/trunk/arcadia/ya', 'ya')

                self.Context.arcadia_dir = str(self.path('arcadia_dir'))
                subprocess.check_call([self.Context.ya, 'clone', self.Context.arcadia_dir])

                self.Context.svn_dir = os.path.join(self.Context.arcadia_dir, self.arcadia_project_path)

                # svn up all parent dirs of project
                splitted = self.arcadia_project_path.split('/')
                for i in range(1, len(splitted)):
                    logging.info('>>> --set-depth empty %s', os.path.join(*splitted[:i]))
                    subprocess.check_call([self.Context.ya, 'tool', 'svn', 'up', '--set-depth', 'empty', os.path.join(*splitted[:i])],
                                          cwd=self.Context.arcadia_dir, **proc_pipes)

                logging.info('>>> --set-depth infinity %s', self.arcadia_project_path)
                subprocess.check_call([self.Context.ya, 'tool', 'svn', 'up', '--set-depth', 'infinity', self.arcadia_project_path],
                                      cwd=self.Context.arcadia_dir, **proc_pipes)

            with self.info_section('<prepare> move .git'):
                subprocess.check_call(['mv', os.path.join(
                    self.Context.sources_path, '.git'), self.Context.svn_dir], **proc_pipes)

            with self.info_section('<setup> git reset --hard HEAD'):
                subprocess.check_call(
                    ['git', 'reset', '--hard', 'HEAD'], cwd=self.Context.svn_dir, **proc_pipes)

            with self.info_section('<setup> git clean -fd'):
                subprocess.check_call(
                    ['git', 'clean', '-fd'], cwd=self.Context.svn_dir, **proc_pipes)

            self.Context.is_master = self.Parameters.project_git_base_ref == 'master'

        if self.Context.is_master and self.Parameters.try_sync:
            self.process_master()
        else:
            self.process_branch()

        self.Context.svn_revision = subprocess.check_output(
            [self.Context.ya, 'svn', 'info', '--show-item', 'last-changed-revision'], cwd=self.Context.svn_dir,
        ).strip()
