# -*- coding: utf-8 -*-

import logging
import os
import tarfile

from sandbox import sdk2
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
from sandbox.common.utils import singleton_property

from sandbox.projects.common.yappy.utils.node_manager import NodeManager
from sandbox.projects.common.yappy.resources import NodeJsArchive

from sandbox.projects.common.yappy.github import BuildWithGithubTask


VCS_GITHUB = 'GitHub Webhook'
VCS_GIT = 'Git'
VCS_SVN = 'SVN'


class BaseBuildUI(BuildWithGithubTask):
    """
    Base class for UI-related tasks.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 * 1024
        ram = 1024

        # NPM is not accessible in LXC containers.
        client_tags = ctc.Tag.IPV4 & ~ctc.Tag.LXC

        # NPM has troubles with IPv6.
        dns = ctm.DnsType.DNS64

    class Parameters(BuildWithGithubTask.Parameters):
        with sdk2.parameters.Group('Node JS') as block_nodejs:
            nodejs_archive = sdk2.parameters.Resource(
                'Node JS',
                resource_type=NodeJsArchive,
                state=(ctr.State.READY, ),
                required=True
            )
            npm_registry = sdk2.parameters.String(
                'NPM registry',
                default='http://npm.yandex-team.ru',
                required=False
            )

        with sdk2.parameters.Group('Build') as block_build:
            dist_path = sdk2.parameters.String('Distribution directory path', default='dist', required=True)
            build_command = sdk2.parameters.String('NPM build script', default='build_production', required=True)

    class Context(BuildWithGithubTask.Context):
        node_binary_path = None

    @singleton_property
    def node(self):
        return NodeManager(self, registry=self.Parameters.npm_registry)

    @property
    def github_context(self):
        raise NotImplementedError

    @property
    def bundle_description(self):
        raise NotImplementedError

    @property
    def bundle_resource_type(self):
        raise NotImplementedError

    @property
    def footer(self):
        if self.Context.tests_results:
            data = []

            for suite_data in self.Context.tests_results.get('testResults', []):
                cases = suite_data.get('testResults', [])
                for case_data in cases:
                    current_suite_ancestors = case_data.get('ancestorTitles')
                    suite_name = current_suite_ancestors[-1] if current_suite_ancestors else '-'
                    entry = {
                        'Suite': suite_name,
                        'Case': case_data.get('title', '-'),
                    }

                    if 'status' in case_data:
                        if case_data['status'] == 'passed':
                            entry['Result'] = '<span class="status status_success">passed</span>'
                        else:
                            entry['Result'] = '<span class="status status_exception">{}</span>'.format(
                                case_data['status']
                            )
                    else:
                        entry['Result'] = '<span class="status status_draft">unknown</span>'

                    data.append(entry)

            return {
                '': data
            }

        else:
            return {
                '<h4>No tests results</h4>': [],
            }

    def before_install(self):
        with self.info_section('<setup> node'):
            self.node.setup(self.Context.sources_path, self.Parameters.nodejs_archive)

    def install(self):
        with self.info_section('<install> target'):
            self.node.install_modules(cwd=self.Context.sources_path)

    def build(self):
        with self.info_section('<build> target'):
            self.node.run_script(self.Parameters.build_command, cwd=self.Context.sources_path)

        with self.info_section('<bundle>'):
            with tarfile.TarFile('bundle.tar.gz', mode='w') as tar:
                tar.add(
                    os.path.join(self.Context.sources_path, self.Parameters.dist_path),
                    self.Parameters.dist_path
                )

            logging.info('>>> ARCHIVED {}'.format(tar))

            self.create_resource(
                self.bundle_resource_type,
                self.bundle_description,
                'bundle.tar.gz'
            )
            logging.info('>>> RESOURCE CREATED {}'.format(tar))

    def teardown(self):
        self.node.teardown()
