# -*- coding: utf-8 -*-

import contextlib
import datetime
import logging
import os
import time

from sandbox import sdk2
from sandbox.common.utils import singleton_property
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.sandbox_ci import managers
from sandbox.projects.sandbox_ci.utils.github import GitHubStatus
from sandbox.projects.sandbox_ci.parameters import CommitHash

from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2


class BuildWithGithubTask(ReleaseToNannyTask2, sdk2.Task):
    """
    Base class for tasks using github webhooks.
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 2 * 1024
        ram = 1024

    class Parameters(sdk2.Task.Parameters):
        # Required for `GitHubStatusesManager`.
        with sdk2.parameters.Group('GitHub') as block_github:
            project_git_base_ref = sdk2.parameters.String('Git base ref', default='master')
            project_git_base_commit = CommitHash('Git base commit')
            project_git_merge_ref = sdk2.parameters.List('Git merge ref')
            project_git_merge_commit = CommitHash('Git merge commit')

            report_github_statuses = sdk2.parameters.Bool('Report GitHub statuses', default=True)
            project_github_owner = sdk2.parameters.String('Project owner')
            project_github_repo = sdk2.parameters.String('Project repository')
            project_github_commit = sdk2.parameters.String('Commit hash')

            project_github_clone_url = sdk2.parameters.String('Repository clone URL')

        # Required for `WebHookManager`.
        with sdk2.parameters.Group('Webhooks') as block_webhooks:
            webhook_urls = sdk2.parameters.List('Urls')

    class Context(sdk2.Context):
        sources_path = None
        failed_tests_count = 0

    @singleton_property
    def github_statuses(self):
        # noinspection PyTypeChecker
        return managers.GitHubStatusesManager(self)

    @singleton_property
    def webhook(self):
        return managers.WebHookManager(self)

    @property
    def github_context(self):
        """
        GitHub check name.

        :rtype: str
        """
        raise NotImplementedError

    @property
    def project_conf(self):
        """
        Replacement for Genisys configuration.

        :rtype: dict
        """
        return {}

    @property
    def bundle_resource_type(self):
        """
        Resource type for resulting bundle.

        :rtype: sdk2.Resource
        """
        raise NotImplementedError

    @property
    def bundle_description(self):
        """
        Description for resulting bundle.

        :rtype: str
        """
        raise NotImplementedError

    @staticmethod
    def fail_task(message='Task failed expectedly'):
        raise SandboxTaskFailureError(message)

    @staticmethod
    def get_project_folder(project_path):
        return os.path.basename(os.path.abspath(project_path))

    @contextlib.contextmanager
    def info_section(self, name):
        """
        :type name: str | unicode
        """
        start = time.time()
        exception = None
        try:
            yield
        except Exception as e:
            exception = e
        finally:
            delta = datetime.timedelta(seconds=time.time() - start)
            delta -= datetime.timedelta(microseconds=delta.microseconds)

            state = 'DONE'
            message = '({}) {{}} {}'.format(delta, name)
            if exception:
                state = 'FAIL'
                message = '{}: {}: {}'.format(message, exception.__class__.__name__, exception)

            try:
                message = message.format(state)
            except Exception as e:
                message = 'info_section ({}) crashed (state={}): {}: {}'.format(name, state, e.__class__.__name__, e)

            self.set_info(message)

            if exception:
                raise exception

    def _propagate_webhook_data(self):
        event = self.Context.githubEvent

        if not event:
            return

        # defaults
        git_base_ref = None
        git_base_commit = None
        git_merge_ref = None
        git_merge_commit = None
        repo_owner = None

        if event['type'] == 'pull_request':
            pr = event['payload']['pull_request']
            git_base_ref = pr['base']['ref']
            git_merge_ref = 'pull/{}'.format(pr['number'])
            git_merge_commit = pr['head']['sha']
            repo_owner = event['payload']['repository']['owner']['login']
        elif event['type'] == 'push':
            repo_owner = event['payload']['repository']['owner']['name']
            # В push событие передается полное имя ветки.
            # Например:
            # - refs/heads/dev для ветки dev
            # - refs/heads/release/v1.0.0 для ветки release/v1.0.0
            git_base_ref = event['payload']['ref'].replace('refs/heads/', '')
            git_base_commit = event['payload']['head_commit']['id']

        self.Parameters.project_git_base_ref = git_base_ref  # TODO: check base/head on PR
        self.Parameters.project_git_base_commit = git_base_commit
        self.Parameters.project_git_merge_ref = [git_merge_ref] if git_merge_ref else []
        self.Parameters.project_git_merge_commit = git_merge_commit

        self.Parameters.project_github_repo = event['payload']['repository']['name']
        self.Parameters.project_github_owner = repo_owner
        self.Parameters.project_github_commit = git_merge_commit or git_base_commit

        self.Parameters.project_github_clone_url = event['payload']['repository']['clone_url']

    def on_break(self, prev_status, status):
        self.send_webhooks(status)

        # Should specify state explicitly because of task is in STOPPING status
        self.github_statuses.report_self_status(
            state=GitHubStatus.ERROR,
            description='Stopped' if status == 'STOPPED' else 'Broken'
        )

        super(BuildWithGithubTask, self).on_break(prev_status, status)

    def on_enqueue(self):
        self.Context.github_context = self.github_context

        with self.memoize_stage.report_enqueue_status:
            self.github_statuses.safe_report_self_status(description='Enqueued')

        super(BuildWithGithubTask, self).on_enqueue()

    def on_execute(self):
        self.github_statuses.report_self_status(description='Running')
        self.Context.sources_path = self.checkout()

        try:
            self.before_install()
            self.install()
            self.after_install()

            self.before_build()
            self.build()
            self.after_build()
        finally:
            self.teardown()

    def on_failure(self, prev_status):
        if self.Context.failed_tests_count != 0:
            description = '{} test(s) failed'.format(self.Context.failed_tests_count)
        else:
            description = 'Build failed'

        # Should specify state explicitly because of task is in FINISHING status.
        self.github_statuses.report_self_status(state=GitHubStatus.FAILURE, description=description)
        super(BuildWithGithubTask, self).on_failure(prev_status)

    def on_finish(self, prev_status, status):
        self.send_webhooks(status)
        super(BuildWithGithubTask, self).on_finish(prev_status, status)

    def on_prepare(self):
        self.github_statuses.report_self_status(description='Preparing')
        super(BuildWithGithubTask, self).on_prepare()

    def on_save(self):
        self._propagate_webhook_data()
        super(BuildWithGithubTask, self).on_save()

    def on_success(self, prev_status):
        # Should specify state explicitly because of task is in FINISHING status.
        self.github_statuses.report_self_status(state=GitHubStatus.SUCCESS, description='Built')
        super(BuildWithGithubTask, self).on_success(prev_status)

    def checkout(self):
        """
        :return: checkout path
        :rtype: str
        """
        with self.info_section('<checkout>'):
            checkout_dir = str(self.path('sources'))

            git_url = self.Parameters.project_github_clone_url
            git_commit = self.Parameters.project_github_commit

            logging.info('>>> GIT URL: {}'.format(git_url))

            with sdk2.helpers.ProcessLog(self, 'git_checkout') as plog:
                proc_pipes = {'stdout': plog.stdout, 'stderr': plog.stderr}
                subprocess.check_call(['git', 'clone', '--recursive', git_url, checkout_dir], **proc_pipes)
                subprocess.check_call(['git', 'checkout', git_commit], cwd=checkout_dir, **proc_pipes)

                # For debugging purposes only.
                subprocess.check_call(['git', 'log', '-n', '5'], cwd=checkout_dir, **proc_pipes)

            logging.info('>>> [GITHUB] CHECKED TO {}'.format(checkout_dir))

        return checkout_dir

    def before_install(self):
        pass

    def install(self):
        raise NotImplementedError

    def after_install(self):
        pass

    def before_test(self):
        pass

    def after_test(self):
        pass

    def before_build(self):
        pass

    def after_build(self):
        pass

    def teardown(self):
        pass

    def build(self):
        raise NotImplementedError

    def send_webhooks(self, status):
        if self.Parameters.webhook_urls:
            self.webhook.send(status, self.Parameters.webhook_urls)

    def create_resource(self, resource_type, resource_description, data_path):
        """
        Creates sandbox resource.

        :type resource_type: type
        :type resource_description: str
        :type data_path: Union[str, sdk2.Path]
        """
        sdk2.ResourceData(resource_type(self, resource_description, data_path)).ready()
