# -*- coding: utf-8 -*-

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2


class RetrySessionCarrier(object):

    RETRY_CNT = 3
    BACKOFF = 0.2

    def __init__(self):
        self._session = None

    @property
    def session(self):
        if self._session is None:
            self._session = requests.Session()
            adapter = HTTPAdapter(max_retries=Retry(
                total=self.RETRY_CNT,
                read=self.RETRY_CNT,
                connect=self.RETRY_CNT,
                backoff_factor=self.BACKOFF,
                status_forcelist=[500, 502, 504],
            ))
            for scheme in 'http://', 'https://':
                self._session.mount(scheme, adapter)
        return self._session


class ClickhouseClient(RetrySessionCarrier):
    """
    Simple wrapper for task profile data retrieval (SANDBOX-4672)
    """

    DEFAULT_CLICKHOUSE_URL = 'http://clickhouse-sandbox.n.yandex-team.ru/'

    def __init__(self, url=None):
        self.url = url or self.DEFAULT_CLICKHOUSE_URL
        super(ClickhouseClient, self).__init__()

    def do_request(self, request_str, *params):
        if params:
            request_str = request_str.format(*params)
        req = requests.Request('POST',
                               self.url,
                               data=request_str,
                               headers={'Content-Type': 'application/x-www-form-urlencoded'})
        prepped = self.session.prepare_request(req)
        return self.session.send(prepped, stream=True).content

    def get_task_disk_usage(self, task_type, limit=10, parse_tsv=True):
        """
        Returns data about maximum and last hdd usage by given task type
        :param task_type: sandbox task type
        :param limit: how many results to return
        :param parse_tsv: whether return dict of query results or return result as is
        :return: data about task type hdd usage
        """
        result = self.do_request(
            b'select max_working_set_usage, last_working_set_usage, place_delta '
            b'from sandbox.taskdiskusaged where task_type = \'{}\' '
            b'order by timestamp desc limit {} format TSV',
            task_type, limit
        )
        return [dict(zip(('max_working_set_usage', 'last_working_set_usage', 'place_delta'), line.strip().split('\t')))
                for line in result.splitlines() if line] if parse_tsv else result


class SetRequirementsFromClickhouseMixin(object):
    """
    Retrieves last N entries about task hdd-usage and updates task disk requirements according to them
    """

    def calculate_disk_usage_limit(self, hdd_limits):
        # Intended to be overrided
        return int(max(hdd_limits) * 1.2)

    def get_last_runs_limit(self):
        # Intended to be overrided
        return 30

    def set_requirements(self):
        stats = self.retrieve_last_run_stats()
        if stats is not None:
            try:
                hdd_limits = [int(entry['place_delta']) for entry in stats]
                if hdd_limits:
                    self.set_info(
                        'Calculated task hdd usage for current run: {} bytes'.format(
                            self.calculate_disk_usage_limit(hdd_limits)))
                    sdk2.Task.server.task[self.id].update({
                        'requirements': {
                            'disk_space': self.calculate_disk_usage_limit(hdd_limits)
                        }
                    })
            except Exception as e:
                self.set_info('Got exception during hdd usage limit applying: {}'.format(e))

    def retrieve_last_run_stats(self):
        try:
            run_stats = ClickhouseClient().get_task_disk_usage(self.type, limit=self.get_last_runs_limit())
        except Exception as e:
            self.set_info('Cannot retrieve run stats for {}. Exception was: {}'.format(self.type, e))
            return None
        return run_stats if run_stats else None
